package cn.com.duiba.nezha.alg.alg.adx;

import cn.com.duiba.nezha.alg.alg.adx.rtbbid.RtbBidAlg;
import cn.com.duiba.nezha.alg.alg.enums.AdxIndex;
import cn.com.duiba.nezha.alg.alg.enums.AdxLevel;
import cn.com.duiba.nezha.alg.alg.enums.AdxStrategy;
import cn.com.duiba.nezha.alg.alg.vo.adx.*;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.AdxDirectlyFactorDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.directly.AdxIndexStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.flowfilter.AdxFilterCntDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.flowfilter.AdxIndexStatDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.pd.AdxStatsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.pd.ClickValueRectifyDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.Target;
import java.util.*;

public class AdxStatData {


    private static final Logger logger = LoggerFactory.getLogger(AdxStatData.class);

    private static long ADX_MULTIPLIER = 10000000;


    /**
     * 计算策略各指标数据
     *
     * @param strategy
     * @param strategyDoList
     * @return 策略各指标数据（key：指标）
     */
    public static Map<String, Long> getStrategyStatData(String strategy,
                                                        ArrayList<AdxStrategyDo> strategyDoList) {


        Map<String, Long> retMap = new HashMap<>();

        Long defaultCnt = 0L;
        Long defaultConsume = 0L;
        Long bidCnt = defaultCnt, sucCnt = defaultCnt, expCnt = defaultCnt, clickCnt = defaultCnt;
        Long adxConsume = defaultConsume, advertConsume = defaultConsume;

        if (AssertUtil.isAllNotEmpty(strategy, strategyDoList)) {

            List<AdxLevelDo> strategyDo = StrategyBid.getStrategyInfo(strategyDoList, strategy);
            bidCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.BID.getCode(), defaultCnt);
            sucCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.BID_SUC.getCode(), defaultCnt);
            expCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.EXP.getCode(), defaultCnt);
            clickCnt = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.CLICK.getCode(), defaultCnt);
            adxConsume = DataUtil.double2Long((StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.ADX_CONSUME.getCode(), defaultConsume) / ADX_MULTIPLIER), 1L);
            advertConsume = StrategyBid.getSumLevelIndex(strategyDo, AdxIndex.ADVERT_CONSUME.getCode(), defaultConsume);

        }

        retMap.put(AdxIndex.BID.getCode(), bidCnt);                   //竞价量
        retMap.put(AdxIndex.BID_SUC.getCode(), sucCnt);               //竞价成功量
        retMap.put(AdxIndex.EXP.getCode(), expCnt);                   //曝光量
        retMap.put(AdxIndex.CLICK.getCode(), clickCnt);               //点击量
        retMap.put(AdxIndex.ADX_CONSUME.getCode(), adxConsume);       //adx消耗(分)
        retMap.put(AdxIndex.ADVERT_CONSUME.getCode(), advertConsume); //广告消耗(分)

        return retMap;
    }


    /**
     * 获取指定策略信息
     *
     * @param strategyInfo
     * @param strategy
     * @return
     */
    public static List<AdxLevelDo> getStrategyInfo(Map<String, AdxStrategyDo> strategyInfo, String strategy) {

        List<AdxLevelDo> ret =  new ArrayList();

        if (AssertUtil.isNotEmpty(strategyInfo)) {
            AdxStrategyDo strategyDo = strategyInfo.get(strategy);

            if (AssertUtil.isNotEmpty(strategyDo)) {
                ret = strategyDo.getLevelDoList();
            }
        }

        return ret;
    }


    /**
     * 获取指定策略指标
     *
     * @param levelDoList
     * @param timeIndex
     * @return
     */
    public static AdxIndexStatsDo getStrategyIndex(List<AdxLevelDo> levelDoList, String timeIndex) {

        AdxIndexStatDo sumLevelStatDo = new AdxIndexStatDo();
        Long bidCnt = 0L, sucCnt = 0L, expCnt = 0L, clickCnt = 0L , adxConsume = 0L, advertConsume = 0L;
        Long advertLaunch = 0L, advertClick = 0L;
        Double preClickValueSum = 0.0;

        if (AssertUtil.isNotEmpty(levelDoList)) {
            for (AdxLevelDo levelDo : levelDoList) {
                if (AssertUtil.isNotEmpty(levelDo)) {

                    AdxIndexStatDo adxIndexStatDo = new AdxIndexStatDo();
                    if (timeIndex.equals("20min")) {
                        adxIndexStatDo = levelDo.getLast20MinStat();
                    } else if (timeIndex.equals("1day")) {
                        adxIndexStatDo = levelDo.getLast1DayStat();
                    }

                    AdxIndexStatsDo indexCompute = AdxStatData.adxIndexCompute(adxIndexStatDo);
                    bidCnt += indexCompute.getBidCnt();
                    sucCnt += indexCompute.getSucCnt();
                    expCnt += indexCompute.getExpCnt();
                    clickCnt += indexCompute.getClickCnt();
                    adxConsume += indexCompute.getAdxConsume();
                    advertConsume += indexCompute.getAdvertConsume();
                    advertLaunch += indexCompute.getAdvertLaunch();
                    advertClick += indexCompute.getAdvertClick();
                    preClickValueSum += indexCompute.getPreClickValueSum();

                }
            }
        }

        sumLevelStatDo.setBid(bidCnt);
        sumLevelStatDo.setBidSuc(sucCnt);
        sumLevelStatDo.setExp(expCnt);
        sumLevelStatDo.setClick(clickCnt);
        sumLevelStatDo.setAdxConsume(adxConsume);
        sumLevelStatDo.setAdvertConsume(advertConsume);
        sumLevelStatDo.setAdvertLaunch(advertLaunch);
        sumLevelStatDo.setAdvertClick(advertClick);
        sumLevelStatDo.setPreClickValue(preClickValueSum);

        sumLevelStatDo.setAdxConsume(multiply(sumLevelStatDo.getAdxConsume(), ADX_MULTIPLIER));
        AdxIndexStatsDo ret = AdxStatData.adxIndexCompute(sumLevelStatDo);
        return ret;
    }


    /**
     * 获取分level指标
     *
     * @param levelDoList
     * @param timeIndex
     * @return
     */
    public static Map<String, Long> getLevelIndex(List<AdxLevelDo> levelDoList, String statIndex, String timeIndex, Long label) {

        Map<String, Long> retMap = new HashMap<>();

        if (AssertUtil.isNotEmpty(levelDoList)) {
            for (AdxLevelDo levelDo : levelDoList) {
                if (AssertUtil.isNotEmpty(levelDo)) {
                    String key = levelDo.getLevel();
                    AdxIndexStatDo adxIndexStatDo = new AdxIndexStatDo();
                    if (timeIndex.equals("20min")) {
                        adxIndexStatDo = levelDo.getLast20MinStat();
                    } else if (timeIndex.equals("1day")) {
                        adxIndexStatDo = levelDo.getLast1DayStat();
                    }

                    AdxIndexStatsDo indexCompute = AdxStatData.adxIndexCompute(adxIndexStatDo);

                    Long ret = 0L;
                    if (statIndex.equals(AdxIndex.BID.getCode())){
                        ret = indexCompute.getBidCnt();
                    } else if (statIndex.equals(AdxIndex.BID_SUC.getCode())){
                        ret = indexCompute.getSucCnt();
                    } else if (statIndex.equals(AdxIndex.EXP.getCode())){
                        ret = indexCompute.getExpCnt();
                    } else if (statIndex.equals(AdxIndex.CLICK.getCode())){
                        ret = indexCompute.getClickCnt();
                    } else if (statIndex.equals(AdxIndex.ADX_CONSUME.getCode())){
                        ret = indexCompute.getAdxConsume();
                    } else if (statIndex.equals(AdxIndex.ADVERT_CONSUME.getCode())) {
                        ret = indexCompute.getAdvertConsume();
                    }

                    retMap.put(key, ret);
                }
            }
        }

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (retMap.get(key) == null) { retMap.put(key, 0L); }
        }

        return retMap;
    }


    /**
     * 获取分level指标
     *
     * @param levelDoList
     * @param timeIndex
     * @return
     */
    public static Map<String, Double> getLevelIndex(List<AdxLevelDo> levelDoList, String statIndex, String timeIndex) {

        Map<String, Double> retMap = new HashMap<>();

        if (AssertUtil.isNotEmpty(levelDoList)) {
            for (AdxLevelDo levelDo : levelDoList) {
                if (AssertUtil.isNotEmpty(levelDo)) {

                    String key = levelDo.getLevel();
                    AdxIndexStatDo adxIndexStatDo = new AdxIndexStatDo();
                    if (timeIndex.equals("20min")) {
                        adxIndexStatDo = levelDo.getLast20MinStat();
                    } else if (timeIndex.equals("1day")) {
                        adxIndexStatDo = levelDo.getLast1DayStat();
                    }

                    AdxIndexStatsDo indexCompute = AdxStatData.adxIndexCompute(adxIndexStatDo);

                    Double ret = null;
                    if (statIndex.equals("roi")){
                        ret = indexCompute.getRoi();
                    } else if (statIndex.equals("sucRate")){
                        ret = indexCompute.getSucRate();
                    } else if (statIndex.equals("ctr")){
                        ret = indexCompute.getCtr();
                    } else if (statIndex.equals("rpm")){
                        ret = indexCompute.getRpm();
                    } else if (statIndex.equals("clickValue")){
                        ret = indexCompute.getClickValue();
                    } else if (statIndex.equals("launchPv")){
                        ret = indexCompute.getLaunchPv();
                    } else if (statIndex.equals("cpc")){
                        ret = indexCompute.getCpc();
                    }

                    retMap.put(key, ret);
                }
            }
        }

        return retMap;
    }


    /**
     * 计算策略-level指定指标数据
     *
     * @param strategy
     * @param strategyDoList
     * @return 策略-level指定指标数据（key：level）
     */
    public static Map<String, Long> getlevelIndexStat(String strategy, String indexLabel,
                                                      ArrayList<AdxStrategyDo> strategyDoList) {

        Map<String, Long> retMap = new HashMap<>();

        Long defaultValue = 0L;

        List<AdxLevelDo> strategyDo = StrategyBid.getStrategyInfo(strategyDoList, strategy);
        retMap = StrategyBid.getLevelIndex(strategyDo, indexLabel, defaultValue);

        if (indexLabel.equals(AdxIndex.ADX_CONSUME.getCode())) {
            retMap.replaceAll((k, v) -> DataUtil.double2Long(DataUtil.division(v, ADX_MULTIPLIER)));
        }

        return retMap;
    }


    /**
     * 计算策略/创意/资源位整体的roi
     *
     * @param mapInfo
     * @return roi
     */
    public static Double getRoi(Map<String, Long> mapInfo, Double targetRoi) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultAdxConsume = 50L;

        if (AssertUtil.isNotEmpty(mapInfo)) {
            //adx消耗(分)
            Long adxConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADX_CONSUME.getCode()), defaultValue);
            //广告消耗(分)
            Long advertConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), defaultValue);

            if (adxConsume >= defaultAdxConsume) {
                ret = DataUtil.division(advertConsume, adxConsume, 6);

            } else {
                ret = DataUtil.division((advertConsume + defaultAdxConsume * targetRoi), (adxConsume + defaultAdxConsume), 6);
            }
        }

        return ret;
    }


    /**
     * 计算不同level的roi
     *
     * @param adxMap
     * @param advertMap
     * @param targetRoi
     * @return 默认值
     */
    public static Map<String, Double> getLevelRoi(Map<String, Long> adxMap,
                                                  Map<String, Long> advertMap,
                                                  Double targetRoi) {

        Map<String, Double> retMap = new HashMap<>();

        Double ret = 0.0;
        Long defaultAdxConsume = 25L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isAllNotEmpty(adxMap, advertMap)) {

                    if (adxMap.get(key) >= defaultAdxConsume) {
                        ret = DataUtil.division(advertMap.get(key), adxMap.get(key), 6);
                        retMap.put(key, ret);

                    } else {
                        ret = DataUtil.division((advertMap.get(key) + defaultAdxConsume * targetRoi), (adxMap.get(key) + defaultAdxConsume), 6);
                        retMap.put(key, ret);
                    }

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * ROI置信处理（根据分level调节因子）
     *
     * @param roiMap
     * @param adxConsumeLevelMs
     * @param directlyFactorDoInfo
     * @return 分level-ROI
     */
    public static Map<String, Double> getRoiFactorRectify(Map<String, Double> roiMap,
                                                          Map<String, Long> adxConsumeLevelMs,
                                                          AdxDirectlyFactorDo directlyFactorDoInfo) {

        Map<String, Double> retMap = new HashMap<>();

        Long adxConsumeLimit = 50L;
        Long adxConsumeBaseLimit = 25L;
        Double baseRoi = nullToMinDefault(roiMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 0.0);

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                retMap.put(key, roiMap.get(key));
            }
        }

        if (AssertUtil.isNotEmpty(directlyFactorDoInfo)) {
            Map<String, Double> factorMap = directlyFactorDoInfo.getFactorExploreMap();
            Double baseFactor = 1.0;
            if (AssertUtil.isNotEmpty(factorMap)) {
                baseFactor = nullToMinDefault(factorMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 1.0);
            }

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                    if (AssertUtil.isNotEmpty(adxConsumeLevelMs)) {

                        if (adxConsumeLevelMs.get(key) < adxConsumeLimit
                                && adxConsumeLevelMs.get(AdxLevel.ADX_LEVEL_TWO.getCode()) >= adxConsumeBaseLimit) {

                            Double diff = DataUtil.division(factorMap.get(key), baseFactor);
                            Double roi = baseRoi * diff;

                            retMap.put(key, DataUtil.formatDouble(roi,6));
                        }
                    }
                }
            }
        }

        return retMap;
    }


    /**
     * 计算竞价成功率
     *
     * @param mapInfo
     * @return 默认值
     */
    public static Double getSucRate(Map<String, Long> mapInfo, Long defaultCnt) {

        Double ret = 0.0;
        Long defaultValue = 0L;

        if (AssertUtil.isNotEmpty(mapInfo)) {

            //竞价量
            Long bidCnt = nullToMinDefault(mapInfo.get(AdxIndex.BID.getCode()), defaultValue);
            //竞价成功量
            Long sucCnt = nullToMinDefault(mapInfo.get(AdxIndex.BID_SUC.getCode()), defaultValue);

            if (bidCnt >= defaultCnt) {
                ret = getNormalValue(DataUtil.division(sucCnt, bidCnt, 6), 0.0, 0.0, 1.0);
            }
        }
        return ret;
    }


    /**
     * 计算不同level的竞价成功率
     *
     * @param bidCntMap
     * @param sucCntMap
     * @return 默认值
     */
    public static Map<String, Double> getLevelSucRate(Map<String, Long> bidCntMap,
                                                      Map<String, Long> sucCntMap) {

        Map<String, Double> retMap = new HashMap<>();
        Double ret = 0.0;
        Long defaultCnt = 10L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isAllNotEmpty(bidCntMap, sucCntMap)) {

                    if (bidCntMap.get(key) >= defaultCnt) {
                        ret = getNormalValue(DataUtil.division(sucCntMap.get(key), bidCntMap.get(key), 6), 0.0, 0.0, 1.0);
                        retMap.put(key, ret);

                    } else {
                        retMap.put(key, 0.0);
                    }

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * 计算策略流量配比
     *
     * @param flowRateMap
     * @param ind
     * @param step
     * @return
     */
    public static Map<String, Double> getFlowRateAdjust(Map<String, Double> flowRateMap,
                                                        Integer ind, Double step) {

        Map<String, Double> retMap = new HashMap<>();

        Double defaultFlowRate = 0.33;  // 默认流量配比
        Double firFlowRate = defaultFlowRate, secFlowRate = defaultFlowRate, thiFlowRate = defaultFlowRate;

        if (AssertUtil.isNotEmpty(flowRateMap)) {
            firFlowRate = getNormalValue(flowRateMap.get(AdxStrategy.ADX_STRATEGY_FIR.getCode()), defaultFlowRate, 0.1, 0.8);
            secFlowRate = getNormalValue(flowRateMap.get(AdxStrategy.ADX_STRATEGY_SEC.getCode()), defaultFlowRate, 0.1, 0.8);
            thiFlowRate = getNormalValue(flowRateMap.get(AdxStrategy.ADX_STRATEGY_THI.getCode()), defaultFlowRate, 0.1, 0.8);
        }

        if (AssertUtil.isAllNotEmpty(ind, step)) {

            if (ind == 1) {
                firFlowRate += step;
            } else if (ind == 2) {
                secFlowRate += step;
            } else if (ind == 3) {
                thiFlowRate += step;
            }

        }

        retMap.put(AdxStrategy.ADX_STRATEGY_FIR.getCode(), firFlowRate);
        retMap.put(AdxStrategy.ADX_STRATEGY_SEC.getCode(), secFlowRate);
        retMap.put(AdxStrategy.ADX_STRATEGY_THI.getCode(), thiFlowRate);

        return retMap;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double nullToDefault(Double value, Double defaultValue) {

        Double ret = defaultValue;
        if (value != null) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Integer nullToDefault(Integer value, Integer defaultValue) {

        Integer ret = defaultValue;
        if (value != null) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long nullToDefault(Long value, Long defaultValue) {

        Long ret = defaultValue;
        if (value != null) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double nullToMinDefault(Double value, Double defaultValue) {

        Double ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long nullToMinDefault(Long value, Long defaultValue) {

        Long ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param
     * @return 默认值
     */
    public static Double nullToDefaultList(Double value, Double[] defaultList) {

        Double ret = value;

        for(int i = 0; i < defaultList.length;i++) {

            if (AssertUtil.isEmpty(ret)) {
                ret = defaultList[i];
            }
        }

        return ret;
    }



    /**
     * 获取正常范围内的值
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double getNormalValue(Double value, Double defaultValue, Double lower, Double upper) {

        Double ret = defaultValue;
        if (value != null) {
            ret = value < lower ? lower : (value > upper ? upper : value);
        }
        return ret;
    }


    /**
     * 获取正常范围内的值
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double getNormalValue(Double value, Double defaultValue, Double lower, Double upper, int newScala) {

        Double ret = defaultValue;
        if (value != null) {
            ret = value < lower ? lower : (value > upper ? upper : value);
        }
        return DataUtil.formatDouble(ret, newScala);
    }


    /**
     * 获取正常范围内的值
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long getNormalValue(Long value, Long defaultValue, Long lower, Long upper) {

        Long ret = defaultValue;
        if (value != null) {
            ret = value < lower ? lower : (value > upper ? upper : value);
        }
        return ret;
    }


    /**
     * 判断是否比阈值大
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isLarger(Long value1, Long value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1 > value2) {
            ret = true;
        }

        return ret;
    }


    /**
     * 判断是否比阈值大
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isLarger(Double value1, Double value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1 > value2) {
            ret = true;
        }

        return ret;
    }


    /**
     * 判断两者是否相等
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isEqual(Long value1, Long value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1.equals(value2)) {
            ret = true;
        }

        return ret;
    }


    /**
     * 判断两者是否相等
     *
     * @param value1
     * @param value2
     * @return
     */
    public static boolean isEqual(Double value1, Double value2) {

        boolean ret = false;

        if (value1 != null && value2 != null && value1.equals(value2)) {
            ret = true;
        }

        return ret;
    }


    /**
     * 计算流量配比得分
     *
     * @param mapInfo
     * @return roi
     */
    public static Double getFlowRateScore(Map<String, Long> mapInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long bidCntLimit = 20L;

        if (AssertUtil.isNotEmpty(mapInfo)) {

            //竞价量
            Long bidCnt = nullToMinDefault(mapInfo.get(AdxIndex.BID.getCode()), defaultValue);
            //adx消耗(分)
            Long adxConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADX_CONSUME.getCode()), defaultValue);
            //广告消耗(分)
            Long advertConsume = nullToMinDefault(mapInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), defaultValue);

            if (bidCnt >= bidCntLimit) {
                ret = DataUtil.division((advertConsume / 1.15 - adxConsume), bidCnt) * 1000;
            }
        }

        return DataUtil.formatDouble(ret,3);
    }


    /**
     *冷启动高价组-因子试探流量占比
     *
     * @param roiMsDiff
     * @param adxConsumeTryMs(分/cpm)
     * @param directlyFactorDoInfo
     * @return
     */
    public static Double getTryFlowRate(Double roiMsDiff, Long adxConsumeTryMs,
                                        AdxDirectlyFactorDo directlyFactorDoInfo) {

        Double ret = 0.20;
        Double lastRate = 0.20;

        if (AssertUtil.isNotEmpty(directlyFactorDoInfo)) {
            Map<String, Double> flowRateMap = directlyFactorDoInfo.getFactorFlowRateMap();
            if (AssertUtil.isNotEmpty(flowRateMap)) {
                lastRate = getNormalValue(flowRateMap.get(AdxLevel.ADX_LEVEL_ONE.getCode()), lastRate, 0.02, 1.0);
            }
        }

        if (AssertUtil.isAllNotEmpty(roiMsDiff, adxConsumeTryMs)) {

            Long adxConsumeMs = nullToDefault(adxConsumeTryMs, 0L);
            Double baseRate = getNormalValue((DataUtil.division(100.0, adxConsumeMs) * lastRate), lastRate, 0.02,0.8);

            if (roiMsDiff > 0.98) {
                ret = baseRate + Math.min(DataUtil.division(roiMsDiff, 0.98) - 1.0, 1.0) * (1.0 - baseRate);

            } else {

                if (adxConsumeMs < 100) {
                    ret = baseRate;

                } else if (adxConsumeMs > 1000) {
                    ret = baseRate * 1.5;

                } else {
                    ret = baseRate * 1.2;

                }

            }

        }

        return getNormalValue(ret,0.20, 0.02, 1.0);
    }


    /**
     *冷启动高价组-因子试探流量占比
     *
     * @param tryDiff
     * @param tryAdxCost(分)
     * @param lastTryRate
     * @return
     */
    public static Double getTryFlowRate(Double tryDiff, Long tryAdxCost, Double lastTryRate) {

        Double ret = 0.20;
        Long adxCost = nullToDefault(tryAdxCost, 0L);

        if (AssertUtil.isNotEmpty(tryDiff)) {

            Double baseRate = getNormalValue((DataUtil.division(200.0, adxCost) * lastTryRate), lastTryRate, 0.02,0.8);

            if (tryDiff < 0.98) {
                ret = baseRate + Math.min(1 - DataUtil.division(tryDiff, 0.98), 1.0) * (1.0 - baseRate);

            } else {

                if (adxCost < 200) {
                    ret = baseRate;

                } else if (adxCost > 1000) {
                    ret = baseRate * 1.5;

                } else {
                    ret = baseRate * 1.2;

                }

            }

        }

        return getNormalValue(ret,0.20, 0.02, 1.0);
    }

    /**
     * 消耗置信保障
     *
     * @param roi
     * @return 消耗置信保障(分)
     */
    public static Double getConfidentAdxConsume(Double roi) {

        Double ret = 100.0;

        if (AssertUtil.isNotEmpty(roi)){

            ret = 100.0 + DataUtil.division(roi-1, 1.15-1) * 100.0 ;
        }

        return getNormalValue(ret,100.0,100.0,200.0);

    }


    /**
     * 冷启动-试探涨价幅度
     *
     * @param adxConsume(分/cpm)
     * @param roi
     * @param limit
     * @return
     */
    public static Double getTryUppStepCon(Long adxConsume, Double roi, Double limit) {

        Double ret = 0.0;

        if (AssertUtil.isAllNotEmpty(adxConsume, roi, limit)) {

            Double confidentConsume = getConfidentAdxConsume(roi);

            if (adxConsume <= confidentConsume) {
                ret = (1.0 - DataUtil.division(adxConsume, confidentConsume)) * limit;

            } else {
                ret = Math.max(1.0 - DataUtil.division(adxConsume, confidentConsume), -1) * limit;

            }

        }

        return DataUtil.formatDouble(getNormalValue(ret,0.0, -limit, limit),3);

    }



    /**
     * 选取最优level（左闭右开区间）
     *
     * @param bidMap
     * @param adxMap
     * @param advertMap
     * @return level
     */
    public static String selectBestLevel(Map<String, Long> bidMap,
                                         Map<String, Long> adxMap,
                                         Map<String, Long> advertMap) {

        String ret = AdxLevel.ADX_LEVEL_TWO.getCode();

        if (AssertUtil.isAllNotEmpty(bidMap, adxMap, advertMap)) {
            Double tmpRpm = -10000000.0;

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();

                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    Long bid = nullToMinDefault(bidMap.get(key),0L);
                    Long adx =  nullToMinDefault(adxMap.get(key),0L);
                    Long advert =  nullToMinDefault(advertMap.get(key),0L);

                    if (isLarger(bid, 10L) && isLarger(adx, 25L) && isLarger(advert, 25L)) {

                        Double profit = advert / 1.15 - adx;
                        Double rpm = MathUtil.division((profit * 1000L), bid,6);

                        if (rpm > tmpRpm) {
                            ret = key;
                            tmpRpm = rpm;
                        }
                    }

                }
            }
        }
        return ret;

    }


    /**
     * 选取最优level（左闭右开区间）
     *
     * @param bidMap
     * @param clickMap
     * @param adxCostMap
     * @param targetCpc
     * @return level
     */
    public static String selectCpcBestLevel(Map<String, Long> bidMap,
                                            Map<String, Long> clickMap,
                                            Map<String, Long> adxCostMap,
                                            Double targetCpc) {

        String ret = AdxLevel.ADX_LEVEL_TWO.getCode();

        if (AssertUtil.isAllNotEmpty(bidMap, clickMap, adxCostMap)) {
            Double tmpAbsDiff = 10000000.0;

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();

                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    Long bid = nullToMinDefault(bidMap.get(key),0L);
                    Long click =  nullToMinDefault(clickMap.get(key),0L);
                    Long adxCost =  nullToMinDefault(adxCostMap.get(key),0L);

                    if (isLarger(bid, 20L) && isLarger(click, 5L) && isLarger(adxCost, 25L)) {

                        Double cpc = DataUtil.division(adxCost, click);
                        Double absDiff = Math.abs(cpc/targetCpc - 1.0);

                        if (absDiff < tmpAbsDiff) {
                            ret = key;
                            tmpAbsDiff = absDiff;
                        }
                    }

                }
            }
        }
        return ret;
    }


    /**
     * 指标计算(创意维度)
     *
     * @param indexStatInfo
     */
    public static AdxIndexStatsDo indexStatCompute(Map<String, Long> indexStatInfo) {

        AdxIndexStatsDo ret = new AdxIndexStatsDo();

        boolean isConfident = false;
        Long bidCnt = 0L, sucCnt = 0L, expCnt = 0L, clickCnt = 0L , adxConsume = 0L, advertConsume = 0L;

        if (AssertUtil.isNotEmpty(indexStatInfo)) {

            //指标:竞价量级,竞价成功量级,曝光量级,点击量级,adx消耗(分),广告消耗(分)
            bidCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.BID.getCode()), 0L);
            sucCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.BID_SUC.getCode()), 0L);
            expCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.EXP.getCode()), 0L);
            clickCnt = nullToMinDefault(indexStatInfo.get(AdxIndex.CLICK.getCode()), 0L);
            adxConsume = DataUtil.double2Long((nullToMinDefault(indexStatInfo.get(AdxIndex.ADX_CONSUME.getCode()), 0L) / ADX_MULTIPLIER), 1L);
            advertConsume = nullToMinDefault(indexStatInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), 0L);

            // 判断是否置信
            if (isLarger(bidCnt, 50L) && isLarger(sucCnt, 20L)
                    && isLarger(expCnt, 10L) && isLarger(clickCnt, 2L)
                    && isLarger(adxConsume, 50L) && isLarger(advertConsume, 50L)) {
                isConfident = true;
            }


            if (isConfident) {

                // roi：广告消耗/adx消耗
                Double roi = MathUtil.division(advertConsume, adxConsume, 6);
                // sucRate：竞价成功率
                Double sucRate = MathUtil.division(sucCnt, bidCnt, 6);
                // ctr：adx点击率
                Double ctr = MathUtil.division(clickCnt, expCnt, 6);
                // rpm: 千次竞价收益
                Double profit = advertConsume / 1.15 - adxConsume;
                Double rpm = MathUtil.division(profit * 1000L, bidCnt, 6);

                ret.setRoi(roi);
                ret.setSucRate(sucRate);
                ret.setCtr(ctr);
                ret.setRpm(rpm);

            }
        }

        ret.setConfident(isConfident);
        ret.setBidCnt(bidCnt);
        ret.setSucCnt(sucCnt);
        ret.setExpCnt(expCnt);
        ret.setClickCnt(clickCnt);
        ret.setAdxConsume(adxConsume);
        ret.setAdvertConsume(advertConsume);

        return ret;
    }



    /**
     * 获取ctr预估区间preInterval
     * ctr预估区间个数(划分：0.002, 0.003, 0.005, 0.007, 0.01)
     *
     * @param preValue  ctr预估值
     */
    public static Integer getPreInterval(Double preValue) {

        Integer ret = null;
        Double[] preList = {0.002, 0.003, 0.005, 0.007, 0.01};  // 预估ctr区间划分

        if (AssertUtil.isNotEmpty(preValue) && preValue >= 0) {

            ret = preValue < preList[0] ? 1
                    : (preValue < preList[1] ? 2
                    : (preValue < preList[2] ? 3
                    : (preValue < preList[3] ? 4
                    : (preValue < preList[4] ? 5
                    : 6))));
        }

        return ret;
    }



    /**
     * 指标计算(创意维度)
     *
     * @param adxIndexStats
     */
    public static AdxIndexStatsDo adxIndexCompute(AdxIndexStatDo adxIndexStats) {

        AdxIndexStatsDo ret = new AdxIndexStatsDo();

        boolean isConfident = false;
        Long bidCnt = 0L, sucCnt = 0L, expCnt = 0L, clickCnt = 0L , adxConsume = 0L, advertConsume = 0L;
        Long advertLaunch = 0L, advertClick = 0L;
        Double preClickValueSum = 0.0;

        if (AssertUtil.isNotEmpty(adxIndexStats)) {

            //指标:竞价量级,竞价成功量级,曝光量级,点击量级,adx消耗(分),广告消耗(分)
            bidCnt = nullToMinDefault(adxIndexStats.getBid(), 0L);
            sucCnt = nullToMinDefault(adxIndexStats.getBidSuc(), 0L);
            expCnt = nullToMinDefault(adxIndexStats.getExp(), 0L);
            clickCnt = nullToMinDefault(adxIndexStats.getClick(), 0L);
            adxConsume = DataUtil.double2Long((nullToMinDefault(adxIndexStats.getAdxConsume(), 0L) / ADX_MULTIPLIER), 1L);
            advertConsume = nullToMinDefault(adxIndexStats.getAdvertConsume(), 0L);

            //指标：广告发券数，广告计费点击数，预估点击价值总和
            advertLaunch = nullToMinDefault(adxIndexStats.getAdvertLaunch(), 0L);
            advertClick = nullToMinDefault(adxIndexStats.getAdvertClick(), 0L);
            preClickValueSum = nullToMinDefault(adxIndexStats.getPreClickValue(), 0.0);

            // 判断是否置信
            if (isLarger(bidCnt, 50L) && isLarger(sucCnt, 20L)
                    && isLarger(expCnt, 10L) && isLarger(clickCnt, 2L)
                    && isLarger(adxConsume, 50L) && isLarger(advertConsume, 50L)) {
                isConfident = true;
            }

            if (isConfident) {

                // roi：广告消耗/adx消耗
                Double roi = MathUtil.division(advertConsume, adxConsume, 6);
                // sucRate：竞价成功率
                Double sucRate = MathUtil.division(sucCnt, bidCnt, 6);
                // ctr：adx点击率
                Double ctr = MathUtil.division(clickCnt, expCnt, 6);
                // rpm: 千次竞价收益
                Double profit = advertConsume / 1.15 - adxConsume;
                Double rpm = MathUtil.division(profit * 1000L, bidCnt, 6);
                // clickValue：实际点击价值（分/单次点击）
                Double clickValue = MathUtil.division(advertConsume, clickCnt, 6);
                // launchPv：每PV发券=发券pv/adx点击PV
                Double launchPv = MathUtil.division(advertLaunch, clickCnt, 6);
                // cpc：adx消耗/adx点击（分/单次点击）
                Double cpc = MathUtil.division(adxConsume, clickCnt, 6);
                // chargePv：每PV点券=券计费点击PV/adx点击PV
                Double chargePv = MathUtil.division(advertClick, clickCnt, 6);

                ret.setRoi(nullToMinDefault(roi, 0.0));
                ret.setSucRate(getNormalValue(sucRate, 0.0, 0.0, 1.0));
                ret.setCtr(getNormalValue(ctr, 0.0, 0.0, 1.0));
                ret.setRpm(rpm);
                ret.setClickValue(nullToMinDefault(clickValue, 0.0));
                ret.setLaunchPv(nullToMinDefault(launchPv, 0.0));
                ret.setCpc(cpc);
                ret.setChargePv(nullToMinDefault(chargePv, 0.0));

            }
        }

        ret.setConfident(isConfident);
        ret.setBidCnt(bidCnt);
        ret.setSucCnt(sucCnt);
        ret.setExpCnt(expCnt);
        ret.setClickCnt(clickCnt);
        ret.setAdxConsume(adxConsume);
        ret.setAdvertConsume(advertConsume);
        ret.setAdvertLaunch(advertLaunch);
        ret.setAdvertClick(advertClick);
        ret.setPreClickValueSum(preClickValueSum);

        return ret;
    }


    /**
     * 指标计算(不同时间维度)
     *
     * @param statsDo
     * @param timeIndex
     */
    public static AdxIndexStatsDo getAdxTimeIndex(AdxStatsDo statsDo, String timeIndex) {

        AdxStatsDo adxStatsDo = Optional.ofNullable(statsDo).orElse(new AdxStatsDo());
        AdxIndexStatDo adxIndexStatDo = new AdxIndexStatDo();

        if (timeIndex.equals("20min")) {
            adxIndexStatDo = adxStatsDo.getLast20MinStat();
        } else if (timeIndex.equals("1hour")) {
            adxIndexStatDo = adxStatsDo.getLast1HourStat();
        } else if (timeIndex.equals("1day")) {
            adxIndexStatDo = adxStatsDo.getLast1DayStat();
        } else if (timeIndex.equals("3day")) {
            adxIndexStatDo = adxStatsDo.getLast3DayStat();
        }  else if (timeIndex.equals("7day")) {
            adxIndexStatDo = adxStatsDo.getLast7DayStat();
        }

        AdxIndexStatsDo ret = AdxStatData.adxIndexCompute(adxIndexStatDo);

        return ret;
    }


    /**
     * 指标计算(不同时间维度)
     *
     * @param statsDo
     */
    public static AdxIndexStatsDo getConTimeIndex(AdxStatsDo statsDo) {

        double r1 = 0.6, r2 = 0.3, r3 = 0.1;

        AdxIndexStatsDo t1hourInfo = AdxStatData.getAdxTimeIndex(statsDo, "1hour");
        AdxIndexStatsDo t1DayInfo = AdxStatData.getAdxTimeIndex(statsDo, "1day");
        AdxIndexStatsDo t3DayInfo = AdxStatData.getAdxTimeIndex(statsDo, "3day");

        AdxIndexStatDo conStatDo = new AdxIndexStatDo();
        conStatDo.setBid(DataUtil.double2Long(r1*t1hourInfo.getBidCnt() + r2*t1DayInfo.getBidCnt() + r3*t3DayInfo.getBidCnt()));
        conStatDo.setBidSuc(DataUtil.double2Long(r1*t1hourInfo.getSucCnt() + r2*t1DayInfo.getSucCnt() + r3*t3DayInfo.getSucCnt()));
        conStatDo.setExp(DataUtil.double2Long(r1*t1hourInfo.getExpCnt() + r2*t1DayInfo.getExpCnt() + r3*t3DayInfo.getExpCnt()));
        conStatDo.setClick(DataUtil.double2Long(r1*t1hourInfo.getClickCnt() + r2*t1DayInfo.getClickCnt() + r3*t3DayInfo.getClickCnt()));
        conStatDo.setAdxConsume(ADX_MULTIPLIER * DataUtil.double2Long(r1*t1hourInfo.getAdxConsume() + r2*t1DayInfo.getAdxConsume() + r3*t3DayInfo.getAdxConsume()));
        conStatDo.setAdvertConsume(DataUtil.double2Long(r1*t1hourInfo.getAdvertConsume() + r2*t1DayInfo.getAdvertConsume() + r3*t3DayInfo.getAdvertConsume()));

        AdxIndexStatsDo ret = AdxStatData.adxIndexCompute(conStatDo);

        return ret;
    }


    /**
     * 指标计算(创意维度)
     *
     * @param intervalStats
     */
    public static Map<Integer, AdxIndexStatsDo> intervalIndexCompute(Map<Integer, AdxIndexStatDo> intervalStats) {

        Integer preIntervalSize = 5; // ctr预估区间个数
        Map<Integer, AdxIndexStatsDo> retMap = new HashMap<>(preIntervalSize);

        boolean isConfident = false;
        Long bidCnt = 0L, sucCnt = 0L, expCnt = 0L, clickCnt = 0L , adxConsume = 0L, advertConsume = 0L;

        for (int key = 1; key <= preIntervalSize; ++key) {

            AdxIndexStatsDo ret = new AdxIndexStatsDo();

            if (AssertUtil.isNotEmpty(intervalStats)) {
                AdxIndexStatDo intervalInfo = intervalStats.get(key);

                if (AssertUtil.isNotEmpty(intervalInfo)) {
                    ret = adxIndexCompute(intervalInfo);
                }
            }

            ret.setConfident(isConfident);
            ret.setBidCnt(bidCnt);
            ret.setSucCnt(sucCnt);
            ret.setExpCnt(expCnt);
            ret.setClickCnt(clickCnt);
            ret.setAdxConsume(adxConsume);
            ret.setAdvertConsume(advertConsume);

            retMap.put(key, ret);

        }

        return retMap;
    }



    /**
     * 计算分区间ROI(实时数据不置信，融合全天数据)
     *
     * @param hourDo
     * @param dayDo
     */
    public static Double getConRoi(AdxIndexStatsDo hourDo, AdxIndexStatsDo dayDo) {

        Double ret = null;
        Double conWeight = 0.5; //实时数据权重

        if (AssertUtil.isAllNotEmpty(hourDo, dayDo)) {

            //指标:adx消耗(分),广告消耗(分)
            Long hourAdx = nullToMinDefault(hourDo.getAdxConsume(), 0L);
            Long hourAdvert = nullToMinDefault(hourDo.getAdvertConsume(), 0L);
            Long dayAdx = nullToMinDefault(dayDo.getAdxConsume(), 0L);
            Long dayAdvert = nullToMinDefault(dayDo.getAdvertConsume(), 0L);

            //指标融合
            Double conAdx = conWeight * hourAdx + (1 - conWeight) * dayAdx;
            Double conAdvert = conWeight * hourAdvert + (1 - conWeight) * dayAdvert;

            // roi：广告消耗/adx消耗
            if (isLarger(conAdx, 25.0) && isLarger(conAdvert, 25.0)) {

                ret = MathUtil.division(conAdvert, conAdx, 6);
            }
        }

        return ret;
    }


    /**
     * 计算融合CPC(实时数据不置信，融合全天数据)
     *
     * @param statsDo1 实时数据
     * @param statsDo2 全天数据
     *  @param conWeight 实时数据权重
     */
    public static Double getConCpc(AdxIndexStatsDo statsDo1, AdxIndexStatsDo statsDo2, Double targetCpc, Double conWeight) {

        Double ret = 0.0;

        if (AssertUtil.isAllNotEmpty(statsDo1, statsDo2)) {

            //指标:adx点击量，adx消耗(分)
            Long clickCnt1 = nullToMinDefault(statsDo1.getClickCnt(), 0L);
            Long adxCost1 = nullToMinDefault(statsDo1.getAdxConsume(), 0L);

            Long clickCnt2 = nullToMinDefault(statsDo2.getClickCnt(), 0L);
            Long adxCost2 = nullToMinDefault(statsDo2.getAdxConsume(), 0L);

            //指标融合
            Double conClickCnt = conWeight * clickCnt1 + (1 - conWeight) * clickCnt2;
            Double conAdxCost = conWeight * adxCost1 + (1 - conWeight) * adxCost2;

            //cpc: adx消耗/点击量
            if (isLarger(clickCnt1, 5L) && isLarger(adxCost1, 25L)) {
                ret = MathUtil.division(adxCost1, clickCnt1, 6);

            } else if (isLarger(conClickCnt, 5.0) && isLarger(conAdxCost, 25.0)) {
                ret = MathUtil.division(conAdxCost, conClickCnt, 6);

            } else {
                ret = DataUtil.formatDouble(DataUtil.division((adxCost1 + 5.0 * targetCpc), (clickCnt1 + 5.0)), 6);

            }
        }

        return ret;
    }



    /**
     * 分level计算融合CPC
     *
     * @param
     * @return
     */
    public static Map<String, Double> getConCpcLevel(Map<String, Long> clickLevel,
                                                     Map<String, Long> adxCostLevel,
                                                     Double targetCpc) {

        Map<String, Double> retMap = new HashMap<>();
        Double ret = 0.0;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                //指标:adx点击量，adx消耗(分)
                Long clickCnt = nullToMinDefault(clickLevel.get(key), 0L);
                Long adxCost = nullToMinDefault(adxCostLevel.get(key), 0L);

                //cpc: adx消耗/点击量
                if (isLarger(clickCnt, 5L) && isLarger(adxCost, 25L)) {
                    ret = MathUtil.division(adxCost, clickCnt, 6);
                } else {
                    ret = DataUtil.formatDouble(DataUtil.division((adxCost + 5.0 * targetCpc), (clickCnt + 5.0)), 6);
                }
            }
            retMap.put(key, ret);
        }

        return retMap;
    }


    /**
     * 相乘
     *
     * @param v1
     * @param v2
     * @return
     */
    public static Long multiply(Long v1, Long v2) {

        Long ret = null;
        if (AssertUtil.isAllNotEmpty(v1, v2)) {
            ret = v1 * v2;
        }
        return ret;
    }


    /**
     * 相乘
     *
     * @param v1
     * @param v2
     * @return
     */
    public static Long multiply(Long v1, Double v2) {

        Long ret = null;
        if (AssertUtil.isAllNotEmpty(v1, v2)) {
            ret = DataUtil.double2Long(v1 * v2);
        }
        return ret;
    }


    /**
     * 相除
     *
     * @param v1
     * @param v2
     * @return
     */
    public static Long division(Long v1, Double v2) {

        Long ret = null;
        if (AssertUtil.isAllNotEmpty(v1, v2)) {
            ret = DataUtil.double2Long(DataUtil.division(v1, v2));
        }
        return ret;
    }


    /**
     * 融合CTR
     *
     * @param preValue 预估值
     * @param statValue 统计值
     * @param preWeight 预估权重
     * @return 融合ctr
     */
    public static Double getConCtr(Double preValue, Double statValue, Double preWeight) {

        Double fUpperLimit = 2.0;     // 比例上限
        Double fLowerLimit = 0.0;     // 比例下限
        Double vUpperLimit = 0.9999;  // 点击率上限
        Double vLowerLimit = 0.0000;  // 点击率下限
        Double defaultValue  = 0.001; // 默认ctr

        statValue = getNormalValue(statValue, defaultValue, vLowerLimit, vUpperLimit);
        preValue = getNormalValue(preValue, statValue, (statValue * fLowerLimit), (statValue * fUpperLimit));

        Double ret = preValue * preWeight + statValue * (1 - preWeight);
        ret = ret < vLowerLimit ? vLowerLimit : (ret > vUpperLimit ? vUpperLimit : ret);

        return ret;
    }



    /**
     * 融合CTR
     *
     * @param preValue 预估值
     * @param statValue 统计值
     * @param preWeight 预估权重
     * @param fLowerLimit 统计值*比例下限
     * @param fUpperLimit 统计值*比例上限
     * @return 融合ctr
     */
    public static Double getConCtr(Double preValue, Double statValue, Double preWeight,
                                   Double fLowerLimit, Double fUpperLimit,
                                   Double defaultValue) {

        Double vUpperLimit = 0.9999;  // 点击率上限
        Double vLowerLimit = 0.0000;  // 点击率下限

        statValue = getNormalValue(statValue, defaultValue, vLowerLimit, vUpperLimit);
        preValue = getNormalValue(preValue, statValue, (statValue * fLowerLimit), (statValue * fUpperLimit));

        Double ret = preValue * preWeight + statValue * (1 - preWeight);
        ret = ret < vLowerLimit ? vLowerLimit : (ret > vUpperLimit ? vUpperLimit : ret);

        return ret;
    }


    /**
     * 预估+统计 融合
     *
     * @param preValue 预估值
     * @param statValue 统计值
     * @param preWeight 预估权重
     * @param fLowerLimit 统计值*比例下限
     * @param fUpperLimit 统计值*比例上限
     * @return 融合ctr
     */
    public static Double getConValue(Double preValue, Double statValue, Double preWeight,
                                     Double fLowerLimit, Double fUpperLimit,
                                     Double defaultValue) {

        statValue = nullToDefault(statValue, defaultValue);
        preValue = getNormalValue(preValue, statValue, (statValue * fLowerLimit), (statValue * fUpperLimit));
        Double ret = preValue * preWeight + statValue * (1 - preWeight);

        return ret;
    }


    /**
     * 融合点击价值
     *
     * @param preValue 预估值
     * @param statValue 统计值
     * @param preWeight 预估权重
     * @return 融合ctr
     */
    public static Double getConClickValue(Double preValue, Double statValue, Double preWeight) {

        Double fUpperLimit = 2.0;     // 比例上限
        Double fLowerLimit = 0.0;     // 比例下限
        Double defaultValue = 1.0;    // 默认点击价值(分/单次点击)

        statValue = nullToDefault(statValue, defaultValue);
        preValue = getNormalValue(preValue, statValue, (statValue * fLowerLimit), (statValue * fUpperLimit));
        Double ret = preValue * preWeight + statValue * (1 - preWeight);

        return ret;
    }



    /**
     * 分桶
     *
     * @param value 预估值
     * @param bucketList
     * @return
     *
     */
    public static Integer bucket(Double value, int[] bucketList) {

        Integer ret = 0;
        if (AssertUtil.isAllNotEmpty(value, bucketList) && bucketList.length > 0) {
            ret = bucketList.length + 1;
            for (int i = 0; i < bucketList.length; i++) {
                int bound = bucketList[i];
                if (value < bound) {
                    ret = i + 1;
                    break;
                }
            }
        }

        return ret;
    }



    /**
     * 分桶
     *
     * @param value 预估值
     * @param bucketList
     * @return
     *
     */
    public static Integer bucket(Double value, double[] bucketList) {

        Integer ret = 0;
        if (AssertUtil.isAllNotEmpty(value, bucketList) && bucketList.length > 0) {
            ret = bucketList.length + 1;
            for (int i = 0; i < bucketList.length; i++) {
                double bound = bucketList[i];
                if (value < bound) {
                    ret = i + 1;
                    break;
                }
            }
        }

        return ret;
    }


    /**
     * 分桶函数，获取右边界（左开右闭区间）
     *
     * @param value
     * @param bucketList 不为空，且不含有空值（未判断）
     * @return 分桶值
     */
    public static Double bucket(Double value, Double[] bucketList) {
        Double ret = 0.00;

        if (value != null && bucketList != null && bucketList.length > 0) {
            for (int i = 0; i < bucketList.length; i++) {
                double bound = bucketList[i];
                if (value <= bound) {
                    ret = bound;
                    break;
                }
            }

            if (value > bucketList[bucketList.length - 1]) {
                ret = bucketList[bucketList.length - 1];
            }
        }
        return ret;
    }



    /**
     * 策略流量分配，返回level
     *
     * @param flowRateList
     * @return
     */
    public static Integer flowSplit(List<Double> flowRateList) {

        Integer ret  = 0;

        Double sumRate = 0.00;

        if (AssertUtil.isNotEmpty(flowRateList)) {
            int size = flowRateList.size();
            Double[] sumRateList = new Double[size];

            for (int ind = 0; ind < size; ind++) {
                sumRate += nullToDefault(flowRateList.get(ind), 0.0);
                sumRateList[ind] = sumRate;
            }

            double random = Math.random();
            for (int ind = 0; ind < size; ind++) {
                double bound = DataUtil.division(sumRateList[ind], sumRateList[size - 1]);
                if (random <= bound) {
                    ret = ind;
                    break;
                }
            }
        }

        return ret;
    }



    /**
     * 流量分配
     *
     * @param flowRateList
     * @return
     */
    public static Integer getFlowSplit(List<Long> flowRateList) {

        Integer ret  = 0;
        Long sumRate = 0L;

        if (AssertUtil.isNotEmpty(flowRateList)) {
            int size = flowRateList.size();
            Long[] sumRateList = new Long[size];

            for (int ind = 0; ind < size; ind++) {
                sumRate += nullToDefault(flowRateList.get(ind), 0L);
                sumRateList[ind] = sumRate;
            }

            double random = Math.random();
            for (int ind = 0; ind < size; ind++) {
                double bound = DataUtil.division(sumRateList[ind], sumRateList[size - 1]);
                if (random <= bound) {
                    ret = ind;
                    break;
                }
            }
        }

        return ret;
    }



    /**
     * 计算点击价值纠偏因子(0.5,1.5)
     * 纠偏因子=预估值/实际值
     *
     * @param clickCnt 点击量级
     * @param adConsume 广告消耗/分
     * @param preClickValueSum 预估点击价值总和
     * @return
     */

    public static Double getRectifyFactor(Long clickCnt, Long adConsume, Double preClickValueSum) {

        Double ret = null;
        if (AssertUtil.isAllNotEmpty(clickCnt, adConsume, preClickValueSum)){

            //纠偏因子=预估值/实际值
            if (isLarger(clickCnt, 10L) && isLarger(adConsume, 100L) && isLarger(preClickValueSum, 0.0)) {
                ret = DataUtil.division(preClickValueSum, adConsume,6);
                ret = getNormalValue(ret, 1.0, 0.5, 1.5);
            }
        }
        return ret;
    }



    /**
     * 计算点击价值纠偏因子(0.5,1.5)--融合
     * 纠偏因子=预估值/实际值
     *
     * @param hourDo
     * @param dayDo
     */
    public static Double getConRectifyFactor(AdxIndexStatsDo hourDo, AdxIndexStatsDo dayDo) {

        Double ret = null;
        Double conWeight = 0.5; //实时数据权重

        if (AssertUtil.isAllNotEmpty(hourDo, dayDo)) {

            //实时维度
            Long hourClick = hourDo.getClickCnt();
            Long hourAdvert = hourDo.getAdvertConsume();
            Double hourPreClickValueSum = hourDo.getPreClickValueSum();
            Double hourFactor = getRectifyFactor(hourClick, hourAdvert, hourPreClickValueSum);

            //当天维度
            Long dayClick = dayDo.getClickCnt();
            Long dayAdvert = dayDo.getAdvertConsume();
            Double dayPreClickValueSum = dayDo.getPreClickValueSum();
            Double dayFactor = getRectifyFactor(dayClick, dayAdvert, dayPreClickValueSum);

            //融合
            Long conClick = DataUtil.double2Long(conWeight * hourClick + (1 - conWeight) * dayClick);
            Long conAdvert = DataUtil.double2Long(conWeight * hourAdvert + (1 - conWeight) * dayAdvert);
            Double conPreClickValueSum = conWeight * hourPreClickValueSum + (1 - conWeight) * dayPreClickValueSum;
            Double conFactor = getRectifyFactor(conClick, conAdvert, conPreClickValueSum);

            ret = hourFactor;
            if (AssertUtil.isEmpty(hourFactor)) {
                ret = conFactor;
                if (AssertUtil.isEmpty(conFactor)) {
                    ret = dayFactor;
                }
            }
        }

        return ret;
    }



    /**
     * 获取点击价值纠偏因子
     *
     * @param clickValueLevel
     * @param clickValueRectify
     * @return
     */
    public static Double getClickValueRectifyFactor(Integer clickValueLevel, ClickValueRectifyDo clickValueRectify) {

        Double ret = 1.0;
        if (AssertUtil.isNotEmpty(clickValueRectify)) {
            Map<Integer, Double> rectifyFactorMap = clickValueRectify.getRectifyFactorMap();

            if (AssertUtil.isNotEmpty(rectifyFactorMap)) {
                ret = AdxStatData.getNormalValue(rectifyFactorMap.get(clickValueLevel), 1.0, 0.5, 1.5);
            }
        }
        return ret;
    }


    /**
     *  Map的value值降序排序
     *
     */
    public static <K, V extends Comparable<? super V>> Map<K, V> sortMapByValueDescend(Map<K, V> map) {
        List<Map.Entry<K, V>> list = new ArrayList<>(map.entrySet());
        Collections.sort(list, new Comparator<Map.Entry<K, V>>() {
            @Override
            public int compare(Map.Entry<K, V> o1, Map.Entry<K, V> o2) {
                int compare = (o1.getValue()).compareTo(o2.getValue());
                return -compare;
            }
        });

        Map<K, V> sortedMap = new LinkedHashMap<K, V>();
        for (Map.Entry<K, V> entry : list) {
            sortedMap.put(entry.getKey(), entry.getValue());
        }
        return sortedMap;
    }


    /**
     *  Map的value值升排序
     *
     */
    public static <K, V extends Comparable<? super V>> Map<K, V> sortMapByValueAscend(Map<K, V> map) {
        List<Map.Entry<K, V>> list = new ArrayList<Map.Entry<K, V>>(map.entrySet());
        Collections.sort(list, new Comparator<Map.Entry<K, V>>() {
            @Override
            public int compare(Map.Entry<K, V> o1, Map.Entry<K, V> o2) {
                int compare = (o1.getValue()).compareTo(o2.getValue());
                return compare;
            }
        });

        Map<K, V> sortedMap = new LinkedHashMap<K, V>();
        for (Map.Entry<K, V> entry : list) {
            sortedMap.put(entry.getKey(), entry.getValue());
        }
        return sortedMap;
    }


    /**
     * 计算指定过滤功能的过滤比例
     * filterType：0-不过滤， 1-流量预过滤-roi&ctr过滤，2-每pv发券过滤，3-每pv点券过滤
     *
     * @param ideafilterDo
     * @return
     */
    public static Double getFilterRate(Map<Integer, AdxFilterCntDo> ideafilterDo, String timeIndex, Integer filterType) {

        Double ret = 0.0;

        if (AssertUtil.isNotEmpty(ideafilterDo)) {

            AdxFilterCntDo filterCntDo1 = Optional.ofNullable(ideafilterDo.get(0)).orElse(new AdxFilterCntDo());
            AdxFilterCntDo filterCntDo2 = Optional.ofNullable(ideafilterDo.get(1)).orElse(new AdxFilterCntDo());
            AdxFilterCntDo filterCntDo3 = Optional.ofNullable(ideafilterDo.get(2)).orElse(new AdxFilterCntDo());
            AdxFilterCntDo filterCntDo4 = Optional.ofNullable(ideafilterDo.get(3)).orElse(new AdxFilterCntDo());


            Long cnt1 = 0L, cnt2 = 0L, cnt3 = 0L, cnt4 = 0L;
            if (timeIndex.equals("20min")) {
                cnt1 = AdxStatData.nullToMinDefault(filterCntDo1.getLast20MinCnt(),0L);
                cnt2 = AdxStatData.nullToMinDefault(filterCntDo2.getLast20MinCnt(),0L);
                cnt3 = AdxStatData.nullToMinDefault(filterCntDo3.getLast20MinCnt(),0L);
                cnt4 = AdxStatData.nullToMinDefault(filterCntDo4.getLast20MinCnt(),0L);
            } else if (timeIndex.equals("1day")) {
                cnt1 = AdxStatData.nullToMinDefault(filterCntDo1.getLast1DayCnt(),0L);
                cnt2 = AdxStatData.nullToMinDefault(filterCntDo2.getLast1DayCnt(),0L);
                cnt3 = AdxStatData.nullToMinDefault(filterCntDo3.getLast1DayCnt(),0L);
                cnt4 = AdxStatData.nullToMinDefault(filterCntDo4.getLast1DayCnt(),0L);
            }

            // 计算过滤比例
            if (filterType.equals(1) && AdxStatData.isLarger(cnt2, 10L) && AdxStatData.isLarger((cnt1+cnt2+cnt3+cnt4), 40L)) {
                ret = DataUtil.division(cnt2, (cnt1+cnt2+cnt3+cnt4), 5);

            } else if (filterType.equals(2) && AdxStatData.isLarger(cnt3, 10L) && AdxStatData.isLarger((cnt1+cnt2+cnt3+cnt4), 40L)) {
                ret = DataUtil.division(cnt3, (cnt1+cnt2+cnt3+cnt4), 5);

            } else if (filterType.equals(3) && AdxStatData.isLarger(cnt4, 10L) && AdxStatData.isLarger((cnt1+cnt2+cnt3+cnt4), 40L)) {
                ret = DataUtil.division(cnt4, (cnt1+cnt2+cnt3+cnt4), 5);
            }

        }

        return AdxStatData.getNormalValue(ret, 0.0, 0.0, 1.0);
    }


    /**
     * 计算威尔逊得分
     *
     * 95%的置信水平，z统计量为1.96
     *
     * @param pos  not null
     * @param total not null
     * @return
     */
    public static Double getWilsonScore(Long pos, Long total) {

        if (total == 0) { total = 1L; }
        if (pos > total) { pos = total; }

        double p = pos * 1.0 / total * 1.0;
        double z = 1.96;
        double zSquare = z * z;
        Double score = (p + (zSquare / (2 * total)) - z * Math.sqrt(4 * total * p * (1 - p) + zSquare) / (2 * total))
                / (1 + zSquare / total);

        return DataUtil.formatDouble(score, 6);
    }


    /**
     * 计算千次竞价收益：rpm
     *
     * 数据不置信时，默认收益为0
     *
     * @param
     * @return
     */
    public static Double getRpm(Long bidCnt, Long adxConsume, Long advertConsume) {

        Double ret = 0.0;

        if (isLarger(bidCnt, 50L) && isLarger(adxConsume, 50L) && isLarger(advertConsume, 50L)) {
            ret = DataUtil.division((advertConsume / 1.15 - adxConsume), bidCnt) * 1000;
        }

        return DataUtil.formatDouble(ret,6);
    }


    /**
     * 计算百度分媒体调价因子
     *
     * @param
     * @return
     */
    public static Double getBaiduAppFactor(AdxDo adxInfo) {

        Double ret = 1.0;

        //范围：百度流量，测试组
        Long groupId = adxInfo.getGroupId();
        Integer groupTag = adxInfo.getGroupTag();
        List groupIds = Arrays.asList(129l,96l, 118l);
        if (AssertUtil.isAnyEmpty(groupId, groupTag) || !groupIds.contains(groupId) || !groupTag.equals(2)) {
            return ret;
        }

        //计算创意+百度app，创意维度指标
        AdxStatsDo ideaAppStats = adxInfo.getIdeaAppStats();

        //近三日数据
        AdxIndexStatsDo ideaAppInfo3Day = getAdxTimeIndex(ideaAppStats, "3day");

        //当日数据
        AdxIndexStatsDo ideaAppInfo = getAdxTimeIndex(ideaAppStats, "1day");

        Integer algoBidMode = nullToDefault(adxInfo.getAlgoBidMode(), 1);

        Double realValue = null;
        Double targetValue = null;
        Double realValue3Day = null;

        if(algoBidMode == 1) {                      //roi
            realValue = ideaAppInfo.getRoi();
            realValue3Day = ideaAppInfo3Day.getRoi();
            targetValue = adxInfo.getMinRoi();
        } else if(algoBidMode == 2) {               //cpc
            realValue = ideaAppInfo.getCpc();
            realValue3Day = ideaAppInfo3Day.getCpc();
            targetValue = adxInfo.getTargetCpc();
        }

        if(ideaAppInfo.getAdvertConsume()!=null && ideaAppInfo.getAdvertConsume() > 100*100) {
            ret = DataUtil.division(realValue, targetValue, 6);
        } else{
            ret = DataUtil.division(realValue3Day, targetValue, 6);
        }

        return getNormalValue(ret, 1.0, 0.75, 1.1);
    }

    /**
     * 根据竞价次数调整出价
     */
    public static Long adjustPrice(AdxDo adxDoInfo, Long price, Double costBias) {
        if(adxDoInfo == null || costBias == null) {
            return price;
        }

        Integer groupTag  = adxDoInfo.getBidCntGroupId();
        System.out.println(groupTag);
        Long groupId = adxDoInfo.getGroupId();
        Boolean isApp = Optional.ofNullable(adxDoInfo.getIsApp()).orElse(true);

        if (AssertUtil.isAnyEmpty(groupId, groupTag) || !groupTag.equals(1) || !isApp ||
                (!groupId.equals(129L) && !groupId.equals(96L))) {
            return price;
        }

        costBias = MathUtil.stdwithBoundary(costBias, 0.3, 0.6);

        Integer imeiBidCnt = adxDoInfo.getImeiBidCnt();
        if(imeiBidCnt == null) {imeiBidCnt = 0;}

        Integer threshold = adxDoInfo.getBidCntThreshold();
        if(threshold == null) {
            threshold = 1;
        }

        AdxStatsDo ideaStats = adxDoInfo.getIdeaStats();
        Long avgPrice = AdxStatsDo.getMergeAvgPrice(ideaStats);

        logger.info("imeiBidCnt1:" + imeiBidCnt + ",threshold:" + threshold);

        Long newPrice = price;
        if(imeiBidCnt < threshold && price < avgPrice) {
            newPrice = (long)(price + (avgPrice - price) * costBias);
            newPrice =  (long) Math.min(newPrice, price * 1.25);
            logger.info("imeiBidCnt2:" + imeiBidCnt + ",threshold:" + threshold);

        }

        return newPrice;
    }

    /**
     * 计算创意维度全天ROI
     */
    public static Double getIdeaRoi(Long advCost, Long adxCost) {
        Double ret = 1.;
        if(AssertUtil.isAllNotEmpty(advCost, adxCost)) {
            ret = MathUtil.division(advCost, adxCost, 3);
        }
        return ret;
    }

    /**
     * 计算创意维度偏差--roi
     */
    public static Double getRoiBias(Double targetRoi, Long advCost, Long adxCost) {

        Double ret = 0.;

        Double realRoi = getIdeaRoi(advCost, adxCost);

        if(realRoi == null) {
            return ret;
        }

        if(targetRoi == null) {
            targetRoi = 1.;
        }

        ret = MathUtil.division(realRoi - 0.75 * targetRoi, targetRoi, 3);

        if(ret == null) {ret = 0.;}

        return ret;
    }

}
