package cn.com.duiba.nezha.alg.alg.dpa.intercept;

import cn.com.duiba.nezha.alg.alg.base.MathBase;
import cn.com.duiba.nezha.alg.alg.base.Roulette;
import cn.com.duiba.nezha.alg.alg.constant.DPAConstant;
import cn.com.duiba.nezha.alg.alg.vo.dpa.intercept.InterceptRcmdDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.feature.parse.DPAActFeatureParse;
import cn.com.duiba.nezha.alg.feature.vo.*;
import cn.com.duiba.nezha.alg.model.CODER;
import cn.com.duiba.nezha.alg.model.tf.TFServingClient;
import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * @author lijingzhe
 * @description 推荐引擎接口
 * @date 2020/6/17
 */
public class InterceptRcmder {
    private static final Logger logger = LoggerFactory.getLogger(InterceptRcmder.class);

    static double[] pCpmBucket = {0.4, 0.6, 0.85, 0.95, 0.99, 1};
    static double[] pCpmWeight = {1, 2, 3, 4, 5, 85};

    /**
     * @author: lijingzhe
     * @date: 2020/6/17
     * @methodParameters: [dpaActFeatureDo]
     * @methodReturnType: java.util.Map<java.lang.String, java.lang.String>
     * @description: DPA特征日志解析
     */
    public static Map<String, String> featureParse(DPAActFeatureDo dpa, SlotActFeature slotActFeature, UserProfileFeature userProfileFeature, FeatureDo featureDo, UserActFeature userActFeature) {
        Map<String, String> DPAFeatureMap = DPAActFeatureParse.generateFeatureMapStatic(dpa);
        Map<String, String> slotActFeatureMap = DPAActFeatureParse.generateFeatureMapSlotAct(slotActFeature);
        Map<String, String> userProfileFeatureMap = DPAActFeatureParse.generateFeatureMapUserProfile(userProfileFeature);
        Map<String, String> featureMap = DPAActFeatureParse.generateFeatureMapBase(featureDo);
        Map<String, String> userActFeatureMap = DPAActFeatureParse.generateFeatureMapUserAct(userActFeature);
        DPAFeatureMap.putAll(slotActFeatureMap);
        DPAFeatureMap.putAll(userProfileFeatureMap);
        DPAFeatureMap.putAll(featureMap);
        DPAFeatureMap.putAll(userActFeatureMap);
        return DPAFeatureMap;
    }

    /**
     * @author: lijingzhe
     * @date: 2020/7/17
     * @methodParameters: [coderModel, ltfModel, activityMatchInfos, dpa, slotActFeature, userProfileFeature, featureDo, userActFeature]
     * @methodReturnType: java.util.List<cn.com.duiba.nezha.alg.alg.vo.dpa.InterceptRcmdDo>
     * @description: 只推常规活动
     */
    public static List<InterceptRcmdDo> recRank(CODER coderModel,
                                                TFServingClient tfServingClient,
                                                List<CandidateInterceptDo> candidateInterceptDos,
                                                DPAActFeatureDo dpa,
                                                SlotActFeature slotActFeature,
                                                UserProfileFeature userProfileFeature,
                                                FeatureDo featureDo,
                                                UserActFeature userActFeature) throws Exception {
        List<InterceptRcmdDo> rets = null;
        if (AssertUtil.isAnyEmpty(dpa, candidateInterceptDos)) {
            logger.warn("InterceptRcmder recRank input params is null");
            return rets;
        }

        // 静态特征
        Map<String, String> staticFeatureMap = featureParse(dpa, slotActFeature, userProfileFeature, featureDo, userActFeature);

        Map<CandidateInterceptDo, FeatureMapDo> featureDoMap = new HashMap<>();
        Map<CandidateInterceptDo, InterceptRcmdDo> retMap = new HashMap<>();
        for (CandidateInterceptDo candidateInterceptDo : candidateInterceptDos) {
            // 封装
            InterceptRcmdDo interceptRcmdDo = new InterceptRcmdDo();
            interceptRcmdDo.setCandidateActivityDo(candidateInterceptDo);
            interceptRcmdDo.setActPackageType(2); //插件默认2
            interceptRcmdDo.setAppId(dpa.getAppId());
            interceptRcmdDo.setSlotId(dpa.getSlotId());
            interceptRcmdDo.setRid(dpa.getRid());
            interceptRcmdDo.setDeviceId(dpa.getDeviceId());
            interceptRcmdDo.setImei(dpa.getImei());
            interceptRcmdDo.setUa(dpa.getUa());
            interceptRcmdDo.setAreaCode(dpa.getAreaCode());
            interceptRcmdDo.setPriceSection(dpa.getPriceSection());

            Map<String, String> dynamicFeatureMap = DPAActFeatureParse.generateInterceptFeaturemapDynamic(candidateInterceptDo);

            Map<String, String> retFeatureMap = new HashMap<>();
            retFeatureMap.putAll(staticFeatureMap);
            retFeatureMap.putAll(dynamicFeatureMap);
            interceptRcmdDo.setFeatureMap(retFeatureMap);
            retMap.put(candidateInterceptDo, interceptRcmdDo);

            FeatureMapDo featureMapDo = new FeatureMapDo();
            featureMapDo.setStaticFeatureMap(staticFeatureMap);
            featureMapDo.setDynamicFeatureMap(dynamicFeatureMap);
            featureDoMap.put(candidateInterceptDo, featureMapDo);
        }
        Map<CandidateInterceptDo, Double> pRpmMap = new HashMap<>();
        // 预估
        if (validModel(coderModel, tfServingClient)) {
            pRpmMap = coderModel.predictWithTFNew(featureDoMap, tfServingClient);
//            logger.info("coderModel.predictWithTFNew() 返回结果数量 ：" + pRpmMap.keySet().size());
        }
        Map<CandidateInterceptDo, Map<String, String>> featureMap = new HashMap<CandidateInterceptDo, Map<String, String>>();
        Map<Long, Double> predMap = new HashMap<>();
        for (Map.Entry<CandidateInterceptDo, Double> pRmp : pRpmMap.entrySet()) {
            CandidateInterceptDo candidateInterceptDo = pRmp.getKey();
            Double pcpm = pRmp.getValue();
            if (candidateInterceptDo.getActivityId() != null && pcpm != null) {
                predMap.put(candidateInterceptDo.getActivityId(), pcpm);
            }
            InterceptRcmdDo interceptRcmdDo = retMap.get(candidateInterceptDo);
            interceptRcmdDo.setPCpm(pcpm);
            retMap.put(candidateInterceptDo, interceptRcmdDo);
        }

        rets = retMap.values().stream()
                .filter(e -> e.getPCpm() != null)
                .sorted(Comparator.comparing(InterceptRcmdDo::getPCpm).reversed())
                .collect(Collectors.toList());

        return rets;
    }

    /**
     * @author: lijingzhe
     * @date: 2020/7/17
     * @methodParameters: [coderModel, ltfModel, activityMatchInfos, dpa, slotActFeature, userProfileFeature, featureDo, userActFeature]
     * @methodReturnType: cn.com.duiba.nezha.alg.alg.vo.dpa.InterceptRcmdDo
     * @description: 常规活动
     */
    public static InterceptRcmdDo recEE(CODER coderModel,
                                  TFServingClient tfServingClient,
                                  List<CandidateInterceptDo> activityDos,
                                  DPAActFeatureDo dpa,
                                  SlotActFeature slotActFeature,
                                  UserProfileFeature userProfileFeature,
                                  FeatureDo featureDo,
                                  UserActFeature userActFeature) throws Exception {
        InterceptRcmdDo ret = null;
        List<InterceptRcmdDo> rets = recRank(coderModel, tfServingClient, activityDos, dpa, slotActFeature, userProfileFeature, featureDo, userActFeature);
        if (rets == null) {
            logger.warn("InterceptRcmder reEE output is null");
            return ret;
        }

        ret = ActRoulette(rets);
        return ret;
    }

    /**
     * @author: lijingzhe
     * @date: 2020/7/17
     * @methodParameters: [coderModel, ltfModel, activityMatchInfos, dpa, slotActFeature, userProfileFeature, featureDo, userActFeature]
     * @methodReturnType: cn.com.duiba.nezha.alg.alg.vo.dpa.InterceptRcmdDo
     * @description: 常规活动
     */
    public static InterceptRcmdDo recSort(CODER coderModel,
                                    TFServingClient tfServingClient,
                                    List<CandidateInterceptDo> activityDos,
                                    DPAActFeatureDo dpa,
                                    SlotActFeature slotActFeature,
                                    UserProfileFeature userProfileFeature,
                                    FeatureDo featureDo,
                                    UserActFeature userActFeature) throws Exception {
        logger.info("常规活动候选集数量 ：" + activityDos.size());
        return recEE(coderModel, tfServingClient, activityDos, dpa, slotActFeature, userProfileFeature, featureDo, userActFeature);
    }


    public static InterceptRcmdDo ActRoulette(List<InterceptRcmdDo> interceptRcmdDos) throws Exception {
        return ActRoulette(interceptRcmdDos, DPAConstant.RESORT_ROULETTE_TOP1_WEIGHT);
    }

    public static InterceptRcmdDo ActRoulette(List<InterceptRcmdDo> interceptRcmdDos, double top1Weight) throws Exception {
        InterceptRcmdDo interceptRcmdDo = null;
        if (interceptRcmdDos == null) {
            return interceptRcmdDo;
        }
        // 获取最优RPM
        Double bestCpm = null;
        CandidateInterceptDo bestCpmKey = null;
        if (interceptRcmdDos.size() == 1) {
            return interceptRcmdDos.get(0);
        }
        for (InterceptRcmdDo rcmdDo : interceptRcmdDos) {
            Double cpm = rcmdDo.getPCpm();
            if (cpm != null) {
                if (bestCpm == null || bestCpm < cpm) {
                    bestCpm = cpm;
                    bestCpmKey = rcmdDo.getCandidateActivityDo();
                }
            }
        }
        // 概率分配
        Map<InterceptRcmdDo, Double> weightMap = new HashMap<>();
        Double weightSum = 0.0;
        for (InterceptRcmdDo rcmdDo : interceptRcmdDos) {
            Double cpm = rcmdDo.getPCpm();
            Double weight = getCpmWeight(cpm, bestCpm);
            if (weight == null) {
                weight = 1.0;
            }
            weightSum += weight;
            weightMap.put(rcmdDo, weight);
        }
        for (Map.Entry<InterceptRcmdDo, Double> entry : weightMap.entrySet()) {
            InterceptRcmdDo rcmdDo = entry.getKey();
            Double weight = entry.getValue();

            // 权重调节
            if (bestCpmKey != null && bestCpmKey.equals(rcmdDo.getCandidateActivityDo()) && weight < weightSum * top1Weight) {

                Double weightOther = weightSum - weight;
                weight = weightOther * top1Weight / (1 - top1Weight);
                weightSum = weightOther + weight;

                weightMap.put(rcmdDo, weight);
            }
        }
        // 挑选
        interceptRcmdDo = Roulette.doubleMap(weightMap);

        return interceptRcmdDo;
    }

    private static Boolean validModel(
            CODER coderModel,
            TFServingClient tfServingClient) {
        Boolean ret = true;

        if (tfServingClient == null || coderModel == null) {
            logger.warn("InterceptRcmder input valid ,params tfServing is null or coder is null");
            ret = false;
        }

        return ret;
    }

    private static Double getCpmWeight(Double rpm, Double bestRpm) {
        Double ret = 0.001;

        if (rpm != null && bestRpm != null) {

            Double ratio = DataUtil.division(rpm, bestRpm, 3);
            if (bestRpm >= 0) {
                ret = MathBase.getConfidenceWeight(Math.min(ratio, 1.0), pCpmBucket, pCpmWeight);
            }
        }

        return ret;
    }

}


