package cn.com.duiba.nezha.alg.alg.correct;

import cn.com.duiba.nezha.alg.alg.vo.advert.AdBidParamsDo;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import lombok.Data;

import java.util.Arrays;

/**
 * 保序回归的校准模型
 * 如果后续还有别的校准模型可以考虑让他们实现同一接口，现在懒得弄了
 */
@Data
public class IsotonicCalibrator {
    private static final String redisKeyPrefix = "NZ_CALIBRATION"; // 该类的rediskey前缀
    private long targetConvertCount; // 该维度下的转化次数, 英文名记得要改一下
    private String dim; // 校准所在的维度
    private String redisKey; // 存储该模型所用的rediskey
    private String updateTime; // 模型更新时间

    private double[] boundaries; // 保序回归的边界点
    private double[] predictions; // 保序回归对应边界点的预估值

    private Double lowerBound; //校准因子的下限
    private Double upperBound; //校准因子的上限

    /**
     * 生成redisKey的方法，与预估模型无关
     * @param slotId 广告位id
     * @param advertId 广告id
     * @return 对应维度的校准模型redisKey
     */
    public static String getModelRedisKey(Long slotId, Long advertId) {
        return getModelRedisKey(slotId, null, advertId);
    }

    /**
     * 生成rediskey的方法，其中留了一个abtestID的输入暂且不用。因为现在校准与模型是正交的。
     * 如果想让模型与校准绑定的话，需要abTestId传入预估cvr的模型id，
     * 工程方需要修改降级流程为广告位+广告+模型  广告位+模型  模型
     * 算法需要修改orderStatSample样本初始化方法中的获取key的方法，并且计算模型的任务也要增加一个维度
     * @param slotId 广告位id
     * @param abTestId 预留参数，若想要校准与模型绑定，可以给它赋值
     * @param advertId 广告id
     * @return 对应维度的校准模型redisKey
     */
    public static String getModelRedisKey(Long slotId, String abTestId, Long advertId) {
        String slotAlgPrefix = AdBidParamsDo.concatSlotAlgPrefix(slotId, abTestId);
        return getModelRedisKeyWithSlotAlgPrefix(slotAlgPrefix, advertId);
    }

    //给在线计算工程compute online使用的，似乎不是很有必要……直接根据redisKey做数据聚合又有什么不可以呢？
    public static String getModelRedisKeyWithSlotAlgPrefix(String slotAlgPrefix, Long advertId) {
        if (slotAlgPrefix == null) {
            return redisKeyPrefix;
        } else if (advertId == null) {
            return redisKeyPrefix + "_" + slotAlgPrefix;
        } else {
            return redisKeyPrefix + "_" + slotAlgPrefix + "_" + advertId;
        }
    }

    //验证模型是否有效，需要满足预估跟边界点的长度一致，并且都是单调递增的
    public void valid() throws RuntimeException {
        int sizeB = boundaries.length;
        int sizeP = predictions.length;
        if (sizeB != sizeP) {
            throw new RuntimeException("the length of boundaries and predictions must be same");
        }
        for (int i = 1; i < boundaries.length; i++) {
            if (boundaries[i - 1] > boundaries[i]) {
                throw new RuntimeException("Isotonic need monotone boundaries and predictions");
            }
        }

        for (int i = 1; i < predictions.length; i++) {
            if (predictions[i - 1] > predictions[i]) {
                throw new RuntimeException("Isotonic need monotone boundaries and predictions");
            }
        }
    }

    //计算线性插值
    private double linearInterpolation(double x1, double y1, double x2, double y2, double x) {
        return y1 + (y2 - y1) * (x - x1) / (x2 - x1);
    }

    /**
     * 校准一个浮点数，输出校准后的值
     * @param testData 需要校准的浮点数
     * @return 校准后的浮点数
     */
    public double calibrateRatio(double testData) {
        int foundIndex = Arrays.binarySearch(boundaries, testData);
        int insertIndex = -foundIndex - 1;

        // Find if the index was lower than all values,
        // higher than all values, in between two values or exact match.
        if (insertIndex == 0) {
            return predictions[0];
        } else if (insertIndex == boundaries.length) {
            return predictions[predictions.length - 1];
        } else if (foundIndex < 0) {
            return linearInterpolation(boundaries[insertIndex - 1], predictions[insertIndex - 1], boundaries[insertIndex], predictions[insertIndex], testData);
        } else {
            return predictions[foundIndex];
        }
    }

    /**
     * 校准一个浮点数，输出校准因子
     * @param testData 需要校准的浮点数
     * @return 校准因子
     */
    public double calibrate(double testData) {
        double rawOutput = calibrateRatio(testData);
        return rawOutput / (testData + 1e-12);
    }

}
