package cn.com.duiba.nezha.alg.alg.dpa;

import cn.com.duiba.nezha.alg.alg.constant.DPAConstant;
import cn.com.duiba.nezha.alg.alg.vo.dpa.PackageIdDo;
import cn.com.duiba.nezha.alg.alg.vo.dpa.PackageInfoDo;
import cn.com.duiba.nezha.alg.alg.vo.dpa.PackageRecallDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.vo.GenericPair;
import cn.com.duiba.nezha.alg.feature.vo.PrizeDo;
import jdk.nashorn.internal.runtime.regexp.joni.SearchAlgorithm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.DoubleStream;

import static cn.com.duiba.nezha.alg.alg.dpa.PackageRecall.ucbPackage;

/**
 * @author: tanyan
 * @date: 2020/6/19
 */
public class PrizeRecall {
    private static final Logger logger= LoggerFactory.getLogger(PrizeRecall.class);

    private static final Integer RECALL_GROUP_MAX_NUM = 10;
    private static final Integer TOTAL_TOPK_PICK_RANGE = 100;
    private static final Integer PRIZE_NUM_AT_LEAST = 1;
    private static final Integer AD_PRIZE_NUM_AT_LEAST = 1;
    private static final Integer AD_PRIZE_NUM_AT_MOST = 2;
    private static final Integer TRY_NUM_FOR_ONE_GROUP = 3;

    private static final Integer TOP_SCORE_SUM_GROUP_NUM = 3;
    private static final Integer SWAP_RANDOM_RANGE = 20;

    /**
     * 对外海选接口，不需要传广告奖品列表
     * @param pks 可投奖品列表, pk的ID对象的prizeId、prizeTagId属性不能为空，adPrizeId属性为null
     * @param slotNum 活动槽位数量
     * @return 海选召回的奖品组合列表
     */
    public static List<PackageRecallDo> matchPrize(List<PackageInfoDo> pks, int slotNum){
        List<PackageInfoDo> adPks = new ArrayList<>();
        return matchPrizeAndAd(pks,adPks,slotNum);
    }

//    /**
//     * 对外实时接口，只传广告奖品
//     * @param adPks
//     * @return
//     */
//    public static List<PackageRecallDo> matchAdPrizeOnline(List<PackageInfoDo> adPks){
//        List<PackageInfoDo> pks = new ArrayList<PackageInfoDo>();
//        int slotNum = AD_PRIZE_NUM_AT_MOST;
//        return matchPrize(pks,adPks,slotNum);
//    }

    /**
     * 召回接口
     * @param pks 可投奖品列表，pk的ID对象的prizeId、prizeTagId属性不能为空，adPrizeId属性为null
     * @param adPks 可投券列表，pk的ID对象的adPrizeId不能为空，prizeId属性为null, adPrizeTagId和adPrizeType至少一个不为空
     * @param slotNum 活动槽位数，约束组合的长度
     * @return 匹配的奖品和券的组合列表
     */
    public static List<PackageRecallDo> matchPrizeAndAd(List<PackageInfoDo> pks,
                                                   List<PackageInfoDo> adPks,
                                                   int slotNum){
        if(AssertUtil.isAnyEmpty(pks) && AssertUtil.isAnyEmpty(adPks)){
            logger.error("PackageRecall matchPrize input params prizes and ad has null");
            return null;
        }

        //将券的string类型“券类型”映射到Long类型字段券tagID上，并赋予一个虚拟prizeID
        Map<Long,String> adIdx2Tag = preprocessAdList(adPks, pks);

        List<PackageRecallDo> recallPrizeList = recallPrize(pks);
        List<PackageRecallDo> recallAdList = recallAdPrize(adPks);

        //        List<PackageRecallDo> groupList = generatePrizeGroup(recallList,prizeTypeGraph,groupNum,slotNum);
        return genPrizeGroup(recallPrizeList, recallAdList, slotNum, RECALL_GROUP_MAX_NUM);
    }

    public static Map<Long,String> preprocessAdList(List<PackageInfoDo> adPks, List<PackageInfoDo> pks){
//        long maxPkId = 0;
        long maxPkTagId = 0;
        long offset = 100000;

        if(pks == null || pks.size() == 0) {
//            maxPkId = offset;
            maxPkTagId = offset;
        }else{
            for (PackageInfoDo pk : pks) {
                PackageIdDo pid = pk.getPackageIdDo();
//                maxPkId = pid.getPrizeId() > maxPkId ? pid.getPrizeId() : maxPkId;
                if(AssertUtil.isEmpty(pid.getPrizeTagId())) {
                    pid.setPrizeTagId(-1l);
                }
                maxPkTagId = pid.getPrizeTagId() > maxPkTagId ? pid.getPrizeTagId() : maxPkTagId;
            }
        }

        //暂时忽略溢出风险
//        long fakePrizeId4Ad = maxPkId + offset;
        long startFakeTagId4Ad = maxPkTagId + offset;

        Map<String, Long> adTag2Idx = new HashMap<>();
        Map<Long,String> adIdx2Tag = new HashMap<>();
        for(PackageInfoDo pk: adPks){
//            pk.getPackageIdDo().setPrizeId(fakePrizeId4Ad++);
            pk.getPackageIdDo().setPrizeId(pk.getPackageIdDo().getAdPrizeId());

            if(pk.getPackageIdDo().getAdPrizeType() == null){
                pk.getPackageIdDo().setAdPrizeType("");
            }

            String typeString = pk.getPackageIdDo().getAdPrizeType();
            Long tagIdx;
            if(adTag2Idx.containsKey(typeString)){
                tagIdx = adTag2Idx.get(typeString);
            }else{
                tagIdx = adTag2Idx.size() + startFakeTagId4Ad;
                adTag2Idx.put(typeString,tagIdx);
                adIdx2Tag.put(tagIdx,typeString);
            }
            pk.getPackageIdDo().setPrizeTagId(tagIdx);
        }
        return adIdx2Tag;
    }

    private static TagInfo getTagBucket(List<PackageRecallDo> prds){
        Map<Long, List<PackageRecallDo>> tagBuckets = new HashMap<>();
        Map<Long, Double> tagWeight = new HashMap<>();
        for (PackageRecallDo pkrd : prds) {
            Long prizeTagId = pkrd.getPackageInfoDo().getPackageIdDo().getPrizeTagId();
            if (tagBuckets.containsKey(prizeTagId)) {
                tagBuckets.get(prizeTagId).add(pkrd);
            } else {
                List<PackageRecallDo> bucket = new ArrayList<>();
                bucket.add(pkrd);
                tagBuckets.put(prizeTagId, bucket);
            }
            tagWeight.put(prizeTagId, tagWeight.getOrDefault(prizeTagId, 0.0d) + pkrd.getMatchScore());
        }
        return new TagInfo(tagBuckets,tagWeight);
    }

    private static class TagInfo{
        public Map<Long, List<PackageRecallDo>> tagBucket;
        public Map<Long, Double> tagWeight;
        public TagInfo(Map<Long, List<PackageRecallDo>> tagBucket, Map<Long, Double> tagWeight){
            this.tagBucket = tagBucket;
            this.tagWeight = tagWeight;
        }
    }

    private static List<PackageRecallDo> topKFromMergedPrizeAndAdList(List<PackageRecallDo> pks,
                                                             List<PackageRecallDo> adPks,
                                                             int topk){
        if( pks.size() == 0 && adPks.size() == 0){
            logger.warn("No input need to be merged!");
        }

        int curIdxPK = 0;
        int curIdxAd = 0;
        int totalSize = Math.min(topk, pks.size() + adPks.size());

        List<PackageRecallDo> result = new ArrayList<>(totalSize);
        for (int i = 0; i < totalSize; i++) {
            if (curIdxPK == pks.size()) {
                result.addAll(adPks.subList(curIdxAd, curIdxAd + totalSize-i));
                break;
            }
            if (curIdxAd == adPks.size()) {
                result.addAll(pks.subList(curIdxPK, curIdxPK + totalSize-i));
                break;
            }

            if (pks.get(curIdxPK).getMatchScore() > adPks.get(curIdxAd).getMatchScore()) {
                result.add(pks.get(curIdxPK));
                curIdxPK++;
            } else {
                result.add(adPks.get(curIdxAd));
                curIdxAd++;
            }

        }
        return result;
    }

    private static List<PackageRecallDo> genPrizeGroup(List<PackageRecallDo> pks,
                                                       List<PackageRecallDo> adPks,
                                                       int slotNum,
                                                       int groupNum){
        Set<String>  existingGroupSet = new HashSet<>();
        List<PackageRecallDo> result = new ArrayList<>();

        List<PackageRecallDo> sortedPks = pks.stream().sorted(Comparator.comparing(PackageRecallDo::getMatchScore)
                .reversed()).collect(Collectors.toList());
        List<PackageRecallDo> sortedAdPks = adPks.stream().sorted(Comparator.comparing(PackageRecallDo::getMatchScore)
                .reversed()).collect(Collectors.toList());

        List<PackageRecallDo> allPkrds = new ArrayList<PackageRecallDo>(pks.size() + adPks.size());
        allPkrds.addAll(sortedPks);
        allPkrds.addAll(adPks);
        List<PackageRecallDo> sortedAllPks = allPkrds.stream().sorted(Comparator.comparing(PackageRecallDo::getMatchScore)
                .reversed()).collect(Collectors.toList());

        //生成排序分最好的topK组合
        List<List<PackageRecallDo>> topPkLists = getTopKScoreSumPkLists(sortedAllPks, slotNum, 10);
//        PackageRecallDo theBestGroup = genBestScoreGroup(sortedPks, sortedAdPks,slotNum);
//        result.add(theBestGroup);
//        existingGroupSet.add(groupSignature(theBestGroup));

        //从topK中直接选几组
        List<PackageRecallDo> topKScoreSumGroups = genTopKScoreSumGroups(topPkLists,3);
        topKScoreSumGroups.forEach(pkrd -> existingGroupSet.add(groupSignature(pkrd)));
        result.addAll(topKScoreSumGroups);

        //不考虑类型的topk random pk扩充召回集，基于topK的swap策略
        int swapGenNum = 4;
        List<PackageRecallDo> swapGroups = genSwapGroups(topPkLists, sortedPks, swapGenNum, existingGroupSet);
        result.addAll(swapGroups);

        //不考虑类型的random pk扩充召回集，随机random策略
        int ramdomPickNum = 3;
        List<PackageRecallDo> randomPickGroups = genByRandomPick(sortedPks, sortedAdPks, existingGroupSet, slotNum, ramdomPickNum);
        result.addAll(randomPickGroups);

        //考虑类型桶的random pk 补充
        int tagGenNum = 5;
        List<PackageRecallDo> tagGroups = genByPickByTag(sortedPks, sortedAdPks, existingGroupSet, slotNum, tagGenNum);
        result.addAll(tagGroups);

        return result;
    }

    private static List<List<PackageRecallDo>> getTopKScoreSumPkLists(List<PackageRecallDo> sortedAllPks, int slotNum, int k){
//        List<List<PackageRecallDo>> result = TopKScoreSumGroupGenerater.topKSumGroups(sortedAllPks, k , slotNum);
        List<List<PackageRecallDo>> result = TopKScoreSumGroupGenerater.randomKFromTopPSumGroups(sortedAllPks, k , slotNum);
        return result;
    }

    private static List<PackageRecallDo> genTopKScoreSumGroups(List<List<PackageRecallDo>> pkGroups, int groupNum){
        List<PackageRecallDo> result = new ArrayList<>(groupNum);
        if(pkGroups.size() ==0 || groupNum < 1){
            return result;
        }

        groupNum = Math.min(groupNum, pkGroups.size());
        for (List<PackageRecallDo> pkGroup : pkGroups.subList(0, groupNum)) {
            PackageRecallDo pkWithGroup = groupPkrdFromPkrdList(pkGroup);
            result.add(pkWithGroup);
        }
        return result;

        //随机性已经在传进来的pkGroup中保证了，下面代码注释掉先
//        //如果groupNum比topK集合大或相等，直接返回topK集合
//        if(groupNum >= pkGroups.size()){
//            for (List<PackageRecallDo> pkGroup : pkGroups) {
//                PackageRecallDo pkWithGroup = groupPkrdFromPkrdList(pkGroup);
//                result.add(pkWithGroup);
//            }
//            return result;
//        }
//
//        //添加最高分组合
//        result.add(groupPkrdFromPkrdList(pkGroups.get(0)));
//
//        //剩下随机挑
//        int leftGroupNum = groupNum - 1;
//        List<List<PackageRecallDo>> candidates = new LinkedList<>();
//        for(List<PackageRecallDo> pkGroup : pkGroups.subList(1,pkGroups.size())){
//            candidates.add(pkGroup);
//        }
//        for (int i = 0; i < leftGroupNum; i++) {
//            int leftSize = candidates.size();
//            int randomIdx = (int)(Math.random()*leftSize);
//            List<PackageRecallDo> curGroup = candidates.remove(randomIdx);
//            result.add(groupPkrdFromPkrdList(curGroup));
//        }
//
//        return result;
    }

    private static List<PackageRecallDo> genByPickByTag(List<PackageRecallDo> pks,
                                                        List<PackageRecallDo> adPks,
                                                        Set<String>  existingGroupSet,
                                                        int slotNum,
                                                        Integer groupNum){
        List<PackageRecallDo> result = new ArrayList<>(groupNum);
        TagInfo prizeTagInfo = getTagBucket(pks);
        TagInfo adTagInfo = getTagBucket(adPks);

        for (int i = 0; i < groupNum; i++) {
            int tryNum = 0;
            boolean isNewOne = false;
            while (tryNum < TRY_NUM_FOR_ONE_GROUP && !isNewOne) {
                PackageRecallDo pkrd = genOneGroupByTagInfo(pks, adPks, prizeTagInfo, adTagInfo, slotNum);
                String groupSign = groupSignature(pkrd);
                if(!existingGroupSet.contains(groupSign)){
                    existingGroupSet.add(groupSign);
                    result.add(pkrd);
                    isNewOne = true;
                }
                tryNum++;
            }
        }
        return result;
    }

    private static PackageRecallDo genOneGroupByTagInfo(List<PackageRecallDo> pks, List<PackageRecallDo> adPks, TagInfo prizeTagInfo, TagInfo adTagInfo, int slotNum) {

        int adNum = Math.min(adTagInfo.tagBucket.size(), AD_PRIZE_NUM_AT_MOST);
        int prizeNum = slotNum - adNum;
        List<PackageRecallDo> prizes;
        List<PackageRecallDo> ads;
        if(adNum <= adTagInfo.tagBucket.size()){
            ads = fetchPrizeFromRandomPickedTag(adTagInfo, adNum);
        }else{
            ads = fetchPkrdFromWeightedList(adPks,adNum);
        }
        if(prizeNum <= prizeTagInfo.tagBucket.size()){
            prizes = fetchPrizeFromRandomPickedTag(prizeTagInfo, prizeNum);
        }else{
            prizes = fetchPkrdFromWeightedList(pks,prizeNum);
        }

        List<PackageRecallDo> pkrdList = new ArrayList<>(slotNum);
        ads.forEach(ad -> pkrdList.add(ad));
        prizes.forEach(prize -> pkrdList.add(prize));

        return groupPkrdFromPkrdList(pkrdList);
    }

    private static PrizeDo prizeDoFromPackageRecallDo(PackageRecallDo pkrd){
        PackageIdDo curPid = pkrd.getPackageInfoDo().getPackageIdDo();
        boolean isAd = curPid.getAdPrizeId() != null;
        Long prizeTagId = curPid.getPrizeTagId();
        Long id = isAd ? curPid.getAdPrizeId() : curPid.getPrizeId();
        PrizeDo pd = new PrizeDo(isAd, id,prizeTagId, pkrd.getMatchScore());
        if(isAd){
            if(pkrd.getPackageInfoDo().getPackageIdDo().getAdPrizeType() != null){
                pd.setAdTagId(pkrd.getPackageInfoDo().getPackageIdDo().getAdPrizeType());
            }
        }else{
            if(pkrd.getPackageInfoDo().getPackageIdDo().getPrizeTagId() != null) {
                pd.setAdTagId(pkrd.getPackageInfoDo().getPackageIdDo().getPrizeTagId().toString());
            }
        }
        return pd;
    }

    private static List<PackageRecallDo> fetchPrizeFromRandomPickedTag(TagInfo tagInfo, int k){
        List<PackageRecallDo> result = new ArrayList<>();
        List<Long> choosenAdTags = weightedPickTags(tagInfo.tagWeight, k);
        for(Long tagId: choosenAdTags){
//            PackageRecallDo pkrd = weightedRandomPickByMatchscore(tagInfo.tagBucket.get(tagId));
            PackageRecallDo pkrd = tagInfo.tagBucket.get(tagId).get(0);
            result.add(pkrd);
        }
        return result;
    }

    protected static List<Long> weightedPickTags(Map<Long, Double> tagWeight, int topk){
        List<Long> result = new ArrayList<>(topk);
        if(topk <= 0){
            return result;
        }

        List<Long> tags = new ArrayList<>(tagWeight.size());
        List<Double> weights = new ArrayList<>(tagWeight.size());
        for (Long tagId : tagWeight.keySet()) {
            tags.add(tagId);
            weights.add(tagWeight.get(tagId));
        }
        List<Integer> resultIdx;
        if(topk > tagWeight.size()){
            logger.warn("k大于类目数,截断成类目数");
            topk = tagWeight.size();
        }

        if(topk == tagWeight.size()){
            resultIdx = RecallUtil.argSort(weights);
        }else{
            resultIdx = RecallUtil.randKWithoutReplacement(weights,topk);
        }

        for(Integer idx : resultIdx){
            result.add(tags.get(idx));
        }

        return result;
    }

    private static List<PackageRecallDo> genByRandomPick(List<PackageRecallDo> pks,
                                                           List<PackageRecallDo> adPks,
                                                           Set<String>  existingGroupSet,
                                                           Integer slotNum,
                                                           Integer groupNum){
        List<PackageRecallDo> result = new ArrayList<>(groupNum);
        int mergeNum = Math.min(TOTAL_TOPK_PICK_RANGE, pks.size() + adPks.size());
        List<PackageRecallDo> sortedPrizes = topKFromMergedPrizeAndAdList(pks, adPks, mergeNum);
        for (int i = 0; i < groupNum; i++) {
            int tryNum = 0;
            boolean isNewOne = false;
            while (tryNum < TRY_NUM_FOR_ONE_GROUP && !isNewOne) {
                PackageRecallDo pkrd = randomPickOneGroup(sortedPrizes, slotNum);
                String groupSign = groupSignature(pkrd);
                if(!existingGroupSet.contains(groupSign)){
                    existingGroupSet.add(groupSign);
                    result.add(pkrd);
                    isNewOne = true;
                }
                tryNum++;
            }
        }
        return result;
    }

    private static PackageRecallDo randomPickOneGroup(List<PackageRecallDo> sortedPrizes, Integer slotNum) {
        PackageRecallDo result = new PackageRecallDo();
        List<PrizeDo> group = new ArrayList<>(slotNum);
        List<PackageRecallDo> pkrds = fetchPkrdFromWeightedList(sortedPrizes, slotNum);
        for(PackageRecallDo pkrd : pkrds){
            PrizeDo prizeDo = prizeDoFromPackageRecallDo(pkrd);
            group.add(prizeDo);
        }
        result.getPackageInfoDo().getPackageIdDo().setPrizeGroup(group);
        return result;
    }

    protected static List<PackageRecallDo> fetchPkrdFromWeightedList(List<PackageRecallDo> sortedPrizes, int k){
        List<PackageRecallDo> result = new ArrayList<>(k);
        if( k > sortedPrizes.size()){
            logger.warn("无放回抽样数大于样本数,返回原List");
            return sortedPrizes;
        }
        List<Double> weights = sortedPrizes.stream().map(PackageRecallDo::getMatchScore).collect(Collectors.toList());
        List<Integer> idxs = RecallUtil.randKWithoutReplacement(weights, k);
        for(Integer idx : idxs){
            result.add(sortedPrizes.get(idx));
        }
        return result;
    }

    private static PackageRecallDo genBestScoreGroup(List<PackageRecallDo> pks,
                                                     List<PackageRecallDo> adPks,
                                                     int slotNum){

        List<PackageRecallDo> pkrdGroup = new ArrayList<>(slotNum);

        int adPreSelectNum = Math.min(AD_PRIZE_NUM_AT_LEAST, adPks.size());
        int prizePreSelectNum = Math.min(PRIZE_NUM_AT_LEAST, pks.size());

        for (int i = 0; i < adPreSelectNum; i++) pkrdGroup.add(adPks.get(i));
        for (int i = 0; i < prizePreSelectNum; i++) pkrdGroup.add(pks.get(i));

        int leftNum = slotNum - adPreSelectNum - prizePreSelectNum;
        int pkIdx = prizePreSelectNum;
        int adPkIdx = adPreSelectNum;

        for (int i = 0; i < leftNum; i++) {
            if(pkIdx == pks.size() && adPkIdx == adPks.size()){
                logger.error("No sufficient prize or ad candidate");
                break;
            }
            if(pkIdx == pks.size()){
                pkrdGroup.add(adPks.get(adPkIdx++));
                continue;
            }

            if(adPkIdx == adPks.size()){
                pkrdGroup.add(pks.get(pkIdx++));
                continue;
            }

            if(pks.get(pkIdx).getMatchScore() > adPks.get(adPkIdx).getMatchScore()){
                pkrdGroup.add(pks.get(pkIdx++));
            }else{
                pkrdGroup.add(adPks.get(adPkIdx++));
            }
        }

        return groupPkrdFromPkrdList(pkrdGroup);
    }

    /**
     * 辅助方法，根据MatchScore加权随机挑选组件
     * @param pkrds 带权重的组件列表
     * @return 选中的组件
     */
    static PackageRecallDo weightedRandomPickByMatchscore(List<PackageRecallDo> pkrds){
        if(pkrds.size() == 0){
            logger.error("Weighted random pick from empty list");
            return null;
        }
        double scoreSum = pkrds.stream().mapToDouble(PackageRecallDo::getMatchScore).sum();
        double randomNum = Math.random()*scoreSum;
        int findIdx = 0;
        for (int i = 0; i < pkrds.size(); i++) {
            findIdx = i;
            randomNum -= pkrds.get(i).getMatchScore();
            if(randomNum <= 0.0){
                break;
            }
        }
        return pkrds.get(findIdx);
    }

    protected static PackageRecallDo weightedRandomPickByMatchscore(List<PackageRecallDo> pks, List<PackageRecallDo> choosen){
        List<PackageRecallDo> candidates = new ArrayList<>(pks.size() - choosen.size() );
        for(PackageRecallDo pk : pks){
            if(!choosen.contains(pk)){
                candidates.add(pk);
            }
        }
        return weightedRandomPickByMatchscore(candidates);
    }

    /**
     * 筛选出recallNum个候选奖品，默认数量为PRIZE_NUM_AT_MOST
     * @param pks 可投奖品列表
     * @return EE筛选的奖品列表
     */
    private static List<PackageRecallDo> recallPrize(List<PackageInfoDo> pks){
        if(pks.size() == 0){
            return new ArrayList<>();
        }
        int topN = Math.min(DPAConstant.PRIZETOPN, pks.size());
        return ucbPackage(pks, topN, DPAConstant.RECALLPRIZE, 2);
    }

    /**
     * 筛选出recallAdPrizeNum个候选奖品，默认数量为AD_PRIZE_NUM_AT_MOST
     * @param pks 可投券列表
     * @return EE筛选的券列表
     */
    protected static List<PackageRecallDo> recallAdPrize(List<PackageInfoDo> pks){
        if(pks.size() == 0){
            return new ArrayList<>();
        }
        int topN = Math.min(DPAConstant.AD_PRIZETOPN, pks.size());
        return ucbPackage(pks, topN, DPAConstant.RECALLPRIZE, 2);
    }

    private static String groupSignature(PackageInfoDo packageInfo){
        List<PrizeDo> group = packageInfo.getPackageIdDo().getPrizeGroup();
        return groupSignature(group);
    }

    private static String groupSignature(PackageRecallDo pkrd){
        return groupSignature(pkrd.getPackageInfoDo());
    }

    protected static String groupSignature(List<PrizeDo> group){
        return group.stream().sorted().map(PrizeDo::toString).collect(Collectors.joining("#"));
    }

    private static List<PackageRecallDo> genSwapGroups(List<List<PackageRecallDo>> sourceGroups,
                                                             List<PackageRecallDo> pks,
                                                             int groupNum,
                                                             Set<String> existingGroupSet){
        if(AssertUtil.isEmpty(sourceGroups)){
            logger.error("Sourcegroups of swap action is empty!");
            return null;
        }

        List<PackageRecallDo> result = new ArrayList<>(groupNum);
        for (int i = 0; i < groupNum; i++) {
            int tryNum = 0;
            boolean isNewOne = false;
            while (tryNum < TRY_NUM_FOR_ONE_GROUP && !isNewOne) {
                int sourceIdx = (int)(Math.random())*sourceGroups.size();
                List<PackageRecallDo> pkrdList = aSwapGroup(sourceGroups.get(sourceIdx), pks);
                PackageRecallDo pkrd = groupPkrdFromPkrdList(pkrdList);
                String groupSign = groupSignature(pkrd);
                if(!existingGroupSet.contains(groupSign)){
                    existingGroupSet.add(groupSign);
                    result.add(pkrd);
                    isNewOne = true;
                }
                tryNum++;
            }
        }

        return result;
    }


    private static List<PackageRecallDo> aSwapGroup(List<PackageRecallDo> sourceGroup, List<PackageRecallDo> pks){
        if(pks.size() <= sourceGroup.size()){
            logger.warn("Number is not match, no swap!");
            return sourceGroup;
        }

        List<PackageRecallDo> result = new ArrayList<>(sourceGroup.size());
        Set<Long> existIds = sourceGroup.stream().map(pk -> pk.getPackageInfoDo().getPackageIdDo().getPrizeId()).collect(Collectors.toSet());

        List<PackageRecallDo> candidatePrizes = new ArrayList<>(pks.size() - sourceGroup.size());
        List<Double> weights = new ArrayList<>(pks.size() - sourceGroup.size());
        for (PackageRecallDo pk : pks) {
            if(!existIds.contains(pk.getPackageInfoDo().getPackageIdDo().getPrizeId())){
                candidatePrizes.add(pk);
                weights.add(pk.getMatchScore());
            }
        }

//        int sourceSwapRange = sourceGroup.size();
        int sourceSwapRange = 4;
        int sourceIdx = (int)(Math.random()*sourceSwapRange);
        int candidateIdx = RecallUtil.weightedRandPick(weights);
        for (int i = 0; i < sourceGroup.size(); i++) {
            if(i == sourceIdx){
                result.add(candidatePrizes.get(candidateIdx));
            }else{
                result.add(sourceGroup.get(i));
            }
        }
        return result;
    }

    private static PackageRecallDo groupPkrdFromPkrdList(List<PackageRecallDo> pkrdList){
        PackageRecallDo result = new PackageRecallDo();
        List<PrizeDo> prizeGroup = new ArrayList<>(pkrdList.size());
        pkrdList.forEach(pk -> prizeGroup.add(prizeDoFromPackageRecallDo(pk)));
        result.getPackageInfoDo().getPackageIdDo().setPrizeGroup(prizeGroup);
        return result;
    }
}
