package cn.com.duiba.nezha.alg.alg.dpa;

import cn.com.duiba.nezha.alg.alg.base.MathBase;
import cn.com.duiba.nezha.alg.alg.base.Roulette;
import cn.com.duiba.nezha.alg.alg.constant.DPAConstant;
import cn.com.duiba.nezha.alg.alg.vo.dpa.ActRcmdDo;
import cn.com.duiba.nezha.alg.common.model.activityrecommend.ActivityMatchInfo;
import cn.com.duiba.nezha.alg.common.model.activityrecommend.WilsonInterval;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.feature.parse.DPAActFeatureParse;
import cn.com.duiba.nezha.alg.feature.vo.*;
import cn.com.duiba.nezha.alg.model.CODER;
import cn.com.duiba.nezha.alg.model.tf.LocalTFModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author lijingzhe
 * @description 推荐引擎接口
 * @date 2020/6/17
 */
public class ActRcmder {
    private static final Logger logger = LoggerFactory.getLogger(ActRcmder.class);

    static double[] pCpmBucket = {0.4, 0.6, 0.85, 0.95, 0.99, 1};
    static double[] pCpmWeight = {1, 2, 3, 4, 5, 85};

    /**
     * @author: lijingzhe
     * @date: 2020/6/17
     * @methodParameters: [dpaActFeatureDo]
     * @methodReturnType: java.util.Map<java.lang.String,java.lang.String>
     * @description: DPA特征日志解析
     */
    public static Map<String, String> featureParse(DPAActFeatureDo dpa, SlotActFeature slotActFeature, UserProfileFeature userProfileFeature, FeatureDo featureDo, UserActFeature userActFeature){
        Map<String, String> DPAFeatureMap = DPAActFeatureParse.generateFeatureMapStatic(dpa);
        Map<String, String> DPADynamicFeatureMap = DPAActFeatureParse.generateFeatureMapDynamic(dpa);
        Map<String, String> slotActFeatureMap = DPAActFeatureParse.generateFeatureMapSlotAct(slotActFeature);
        Map<String, String> userProfileFeatureMap = DPAActFeatureParse.generateFeatureMapUserProfile(userProfileFeature);
        Map<String, String> featureMap = DPAActFeatureParse.generateFeatureMapBase(featureDo);
        Map<String, String> userActFeatureMap = DPAActFeatureParse.generateFeatureMapUserAct(userActFeature);
        DPAFeatureMap.putAll(DPADynamicFeatureMap);
        DPAFeatureMap.putAll(slotActFeatureMap);
        DPAFeatureMap.putAll(userProfileFeatureMap);
        DPAFeatureMap.putAll(featureMap);
        DPAFeatureMap.putAll(userActFeatureMap);
        return DPAFeatureMap;
    }

    /**
     * @author: lijingzhe
     * @date: 2020/6/17
     * @methodParameters: [coderModel, ltfModel, packageRecallDos]
     * @methodReturnType: cn.com.duiba.nezha.alg.alg.vo.dpa.ActRcmdDo
     * @description: 排序接口
     */
    public static List<ActRcmdDo> recRank(CODER coderModel,
                                          LocalTFModel ltfModel,
                                          List<CandidateActivityDo> candidateActivityDos,
                                          List<ActivityMatchInfo> activityMatchInfos,
                                          DPAActFeatureDo dpa,
                                          SlotActFeature slotActFeature,
                                          UserProfileFeature userProfileFeature,
                                          FeatureDo featureDo,
                                          UserActFeature userActFeature) throws Exception{
        List<ActRcmdDo> rets = null;
        if(AssertUtil.isAnyEmpty(candidateActivityDos, dpa)){
            logger.error("Actcmder recRank input params is null");
            return rets;
        }
        // 静态特征
        Map<String, String> staticFeatureMap = featureParse(dpa, slotActFeature, userProfileFeature, featureDo, userActFeature);
        Map<CandidateActivityDo, Map<String, String>> featureMap = new HashMap<>();
        for (ActivityMatchInfo activityMatchInfo : activityMatchInfos) {
            CandidateActivityDo candidateActivityDo = new CandidateActivityDo();
            candidateActivityDo.setActivityId(activityMatchInfo.getActivityId());
            double matchScore = getMatchScore(activityMatchInfo);
            candidateActivityDo.setActivityMatchScore(matchScore);
            candidateActivityDos.add(candidateActivityDo);
        }

        Map<CandidateActivityDo, ActRcmdDo> retMap = new HashMap<>();
        for (CandidateActivityDo candidateActivityDo : candidateActivityDos) {
            // 封装
            ActRcmdDo actRcmdDo = new ActRcmdDo();
            actRcmdDo.setCandidateActivityDo(candidateActivityDo);
            actRcmdDo.setActPackageType(candidateActivityDo.getActivityId() == null && candidateActivityDo.getPrizeGroup() ==null ? 0:1);
            actRcmdDo.setAppId(dpa.getAppId());
            actRcmdDo.setSlotId(dpa.getSlotId());
            actRcmdDo.setRid(dpa.getRid());
            actRcmdDo.setDeviceId(dpa.getDeviceId());
            actRcmdDo.setImei(dpa.getImei());
            actRcmdDo.setUa(dpa.getUa());
            actRcmdDo.setAreaCode(dpa.getAreaCode());
            actRcmdDo.setPriceSection(dpa.getPriceSection());
            retMap.put(candidateActivityDo, actRcmdDo);

            DPAActFeatureDo dpaActFeatureDo = new DPAActFeatureDo();
            dpaActFeatureDo.setCandidateActivityDo(candidateActivityDo);
            Map<String, String> dynamicFeatureMap = DPAActFeatureParse.generateFeatureMapDynamic(dpaActFeatureDo);
            dynamicFeatureMap.putAll(staticFeatureMap);
            featureMap.put(candidateActivityDo, dynamicFeatureMap);
        }
        Map<CandidateActivityDo, Double> pRpmMap = new HashMap<>();
        // 预估
        if (validModel(coderModel, ltfModel)) {
            pRpmMap = coderModel.predictWithLocalTF(featureMap, ltfModel);
        }

        for (Map.Entry<CandidateActivityDo, Double> pRmp : pRpmMap.entrySet()) {
           CandidateActivityDo candidateActivityDo =  pRmp.getKey();
           Double pcpm = pRmp.getValue();
            ActRcmdDo actRcmdDo = retMap.get(candidateActivityDo);
            actRcmdDo.setpCpm(pcpm);
            retMap.put(candidateActivityDo, actRcmdDo);
        }

        rets = retMap.values().stream()
                .filter(e -> e.getpCpm() != null)
                .sorted(Comparator.comparing(ActRcmdDo::getpCpm).reversed())
                .collect(Collectors.toList());

        return rets;
    }

    /**
     * @author: lijingzhe
     * @date: 2020/6/17
     * @methodParameters: [actRcmdDos]
     * @methodReturnType: java.util.Map
     * @description: 冷启动接口
     */
    public static ActRcmdDo recEE(CODER coderModel,
                                  LocalTFModel ltfModel,
                                  List<CandidateActivityDo> candidateActivityDos,
                                  List<ActivityMatchInfo> activityMatchInfos,
                                  DPAActFeatureDo dpa,
                                  SlotActFeature slotActFeature,
                                  UserProfileFeature userProfileFeature,
                                  FeatureDo featureDo,
                                  UserActFeature userActFeature) throws Exception{
        ActRcmdDo ret = null;
        List<ActRcmdDo> rets = recRank(coderModel, ltfModel, candidateActivityDos, activityMatchInfos, dpa, slotActFeature, userProfileFeature, featureDo, userActFeature);
        if(rets == null){
            return ret;
        }
        List<ActRcmdDo> packages = rets.stream().filter(e -> e.getActPackageType() == 0).collect(Collectors.toList());
        List<ActRcmdDo> acts = rets.stream().filter(e -> e.getActPackageType() == 1).collect(Collectors.toList());
        if(packages != null && Math.random() < DPAConstant.PACKAGE_REC_RATE){
            ret = ActRoulette(packages);
        }else if(acts != null){
            ret = ActRoulette(acts);
        }else{
            ret = ActRoulette(rets);
        }
        return ret;
    }

    /**
     * @author: lijingzhe
     * @date: 2020/6/17
     * @methodParameters: [actRcmdDos]
     * @methodReturnType: java.util.Map<java.lang.Integer,java.util.List<cn.com.duiba.nezha.alg.alg.vo.dpa.ActRcmdDo>>
     * @description: 重排序接口
     */
    public static ActRcmdDo recSort(CODER coderModel,
                                    LocalTFModel ltfModel,
                                    List<CandidateActivityDo> candidateActivityDos,
                                    List<ActivityMatchInfo> activityMatchInfos,
                                    DPAActFeatureDo dpa,
                                    SlotActFeature slotActFeature,
                                    UserProfileFeature userProfileFeature,
                                    FeatureDo featureDo,
                                    UserActFeature userActFeature) throws Exception{
        ActRcmdDo ret = null;
        ret = recEE(coderModel, ltfModel, candidateActivityDos, activityMatchInfos, dpa, slotActFeature, userProfileFeature, featureDo, userActFeature);
        return ret;
    }

    /**
     * @author: lijingzhe
     * @date: 2020/7/5
     * @methodParameters: [activityMatchInfo]
     * @methodReturnType: java.lang.Double
     * @description: 还原4.1cvr召回分
     */
    public static Double getMatchScore(ActivityMatchInfo activityMatchInfo) throws Exception {
        Double matchScore = 0d;
        double slotScore = WilsonInterval.wilsonCalc(activityMatchInfo.getHisRequest().slotVal / DPAConstant.SLOT_VISIT_PV, activityMatchInfo.getHisRequest().slotVal).lowerBound;
        double globalScore = WilsonInterval.wilsonCalc(activityMatchInfo.getHisRequest().globalVal / DPAConstant.GLOBAL_VISIT_PV, activityMatchInfo.getHisRequest().globalVal).lowerBound;
        double appScore = WilsonInterval.wilsonCalc(activityMatchInfo.getHisRequest().appVal / DPAConstant.APP_VISIT_PV, activityMatchInfo.getHisRequest().appVal).lowerBound;

        double sconfidence = Math.min(activityMatchInfo.getHisRequest().slotVal / DPAConstant.SLOT_VISIT_PV, 1);
        double aconfidence = Math.min(activityMatchInfo.getHisRequest().appVal / DPAConstant.APP_VISIT_PV, 1);
        double gconfidence = Math.min(activityMatchInfo.getHisRequest().globalVal / DPAConstant.GLOBAL_VISIT_PV * 10, 1);

        matchScore = sconfidence * slotScore
                + (1 - sconfidence) * aconfidence * appScore * DPAConstant.APP_CONFIDENCE
                + (1 - sconfidence - (1 - sconfidence) * aconfidence) * globalScore * Math.max(DPAConstant.GLOBAL_CONFIDENCE, gconfidence);
        return matchScore;
    }

    public static ActRcmdDo ActRoulette(List<ActRcmdDo> actRcmdDos) throws Exception {
        ActRcmdDo actRcmdDo = null;
        if(actRcmdDos == null){
            return actRcmdDo;
        }
        // 获取最优RPM
        Double bestCpm = null;
        CandidateActivityDo bestCpmKey = null;
        if(actRcmdDos.size() == 1){
            return actRcmdDos.get(0);
        }
        for (ActRcmdDo rcmdDo : actRcmdDos) {
            Double cpm = rcmdDo.getpCpm();
            if(cpm != null){
                if(bestCpm == null || bestCpm < cpm){
                    bestCpm = cpm;
                    bestCpmKey = rcmdDo.getCandidateActivityDo();
                }
            }
        }
        // 概率分配
        Map<ActRcmdDo, Double> weightMap = new HashMap<>();
        Double weightSum = 0.0;
        for (ActRcmdDo rcmdDo : actRcmdDos) {
            Double cpm = rcmdDo.getpCpm();
            Double weight = getCpmWeight(cpm, bestCpm);
            if (weight == null) {
                weight = 1.0;
            }
            weightSum += weight;
            weightMap.put(rcmdDo, weight);
        }
        for (Map.Entry<ActRcmdDo, Double> entry : weightMap.entrySet()) {
            ActRcmdDo rcmdDo = entry.getKey();
            Double weight = entry.getValue();

            // 权重调节
            if (bestCpmKey != null && bestCpmKey.equals(rcmdDo.getCandidateActivityDo()) && weight < weightSum * 0.8) {

                Double weightOther = weightSum - weight;
                weight = weightOther * 0.8 / (1 - 0.8);
                weightSum = weightOther + weight;

                weightMap.put(rcmdDo,weight);
            }
        }
        // 挑选
        actRcmdDo = Roulette.doubleMap(weightMap);

        return actRcmdDo;
    }

    private static Boolean validModel(
            CODER coderModel,
            LocalTFModel ltfModel) {
        Boolean ret = true;

        if (ltfModel == null || coderModel == null) {
            logger.error("ActTitleDQNRcmder.rcmd() input valid ,params ltfModel is null or coder is null");
            ret = false;
        }

        return ret;
    }

    private static Double getCpmWeight(Double rpm, Double bestRpm) {
        Double ret = 0.001;

        if (rpm != null && bestRpm != null) {

            Double ratio = DataUtil.division(rpm, bestRpm, 3);
            if (bestRpm >= 0) {
                ret = MathBase.getConfidenceWeight(Math.min(ratio, 1.0), pCpmBucket, pCpmWeight);
            }
        }

        return ret;
    }
}

