package cn.com.duiba.nezha.alg.common.model.activityrecommend;

import com.alibaba.fastjson.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lijingzhe
 * @description 组合对冲解决两极分化的badcase
 * @date 2019/11/13
 */

public class ActivityRec45 {
    private static final Logger logger= LoggerFactory.getLogger(ActivityRec45.class);

    static class Constant {
        static double MIN_REWARD=0.1;
        static double DECAY=0.99;
        // 海选召回的数量
        static int RECALL_SUC_TOPN=10;
        static int RECALL_GUC_TOPN=10;
        static int RECALL_SUCK_TOPN=5;
        static int RECALL_GUCK_TOPN=5;
        static int RECALL_SCVR_TOPN=5;
        static int RECALL_GCVR_TOPN=5;
        static int RECALL_CVR_TOPN=5;
        static int RECALL_UR_TOPN=5;
        static int EXCELLENT_TOPN=10;

        static String ALG_UC = "uc";
        static String ALG_UCK = "uck";
        static String ALG_CVR = "cvr";
        static String ALG_UR = "ur";
    }

    static class RankInfo {
        double grpm;
        double gexp;
        double hrpm;
        double hexp;
        double arpm;
        double aexp;
    }

    static class MatchInfo {
        double score=0;
        ActivityMatchInfo act;
    }

    static class SelectInfoDetail {
        int index;
        ArrayList<SelectPoolInfo> condi;
        double maxChangeScore;
        int strategyType;
    }

    /**
     *功能描述 海选分(每pv消耗)降序排
     * @author lijingzhe
     * @date 2019/8/20
     * @param
     * @return
     */
    public static Comparator<MatchInfo> iComparator=new Comparator<MatchInfo>() {
        @Override
        public int compare(MatchInfo c1, MatchInfo c2) {
            return (int) (c2.score - c1.score >= 0 ? 1 : -1);
        }
    };

    /**
     *功能描述 归一化得分降序排
     * @author lijingzhe
     * @date 2019/8/21
     * @param
     * @return
     */
    public static Comparator<MatchInfo> normComparator=new Comparator<MatchInfo>() {
        @Override
        public int compare(MatchInfo m1, MatchInfo m2) {
            return (int) (m2.act.score - m1.act.score >= 0 ? 1 : -1);
        }
    };

    /**
     * @author: lijingzhe
     * @date: 2019/11/13
     * @methodParameters:
     * @methodReturnType:
     * @description: 4.5海选，在4.2基础上增加多路召回(每uv券点击、cvr、每uv券点击*cvr)
     */
    public static List<ActivityMatchInfo> match (List<ActivityInfoData> actData, List<ActivityInfoAd> actAds, List<NormInfo> exActs) {
        int topn= Constant.RECALL_SUC_TOPN+Constant.RECALL_GUC_TOPN+Constant.RECALL_GUCK_TOPN+Constant.RECALL_SUCK_TOPN+Constant.RECALL_GCVR_TOPN+Constant.RECALL_SCVR_TOPN+Constant.RECALL_CVR_TOPN+Constant.RECALL_CVR_TOPN+Constant.RECALL_UR_TOPN;
        int exTopN= Constant.EXCELLENT_TOPN;
        // 海选集初始化
        List<ActivityMatchInfo> result=new ArrayList<>();
        // 可投活动封装
        List<ActivityMatchInfo> actModel=new ArrayList<ActivityMatchInfo>();
        // 按融合券消耗降序活动队列
        Queue<MatchInfo> gCandisUC=new PriorityQueue<>(actData.size(), iComparator);
        // 按券消耗降序活动队列
        Queue<MatchInfo> sCandisUC=new PriorityQueue<>(actData.size(), iComparator);
        // 按融合券点击降序活动队列
        Queue<MatchInfo> gCandisUCK=new PriorityQueue<>(actData.size(), iComparator);
        // 按券点击降序活动队列
        Queue<MatchInfo> sCandisUCK=new PriorityQueue<>(actData.size(), iComparator);
        // 按融合CVR降序活动队列
        Queue<MatchInfo> gCandisCVR=new PriorityQueue<>(actData.size(), iComparator);
        // 按CVR降序活动队列
        Queue<MatchInfo> sCandisCVR=new PriorityQueue<>(actData.size(), iComparator);
        // 按真实CVR降序活动队列
        Queue<MatchInfo> candisCVR=new PriorityQueue<>(actData.size(), iComparator);
        // 按真实券点击*真实cvr降序活动队列
        Queue<MatchInfo> candisUR=new PriorityQueue<>(actData.size(), iComparator);
        // 按归一化score降序活动队列
        Queue<MatchInfo> normCandis=new PriorityQueue<>(exTopN, normComparator);
        // 活动维度最近三天已投放过广告数据
        HashMap<Long, ActivityInfoAd> adHashMap=new HashMap<Long, ActivityInfoAd>();
        // 广告位cvr
        double form_click_cnt = 0.0;
        double notForm_click_cnt = 0.0;
        double click_cnt = 0.0;
        double form_effect_cnt = 0.0;
        double notForm_effect_cnt = 0.0;
        double effect_cnt = 0.0;
        for(ActivityInfoAd ad : actAds){
            adHashMap.put(ad.getActivityId(), ad);
            form_click_cnt += ad.getSumClickForms();
            notForm_click_cnt += ad.getSumClickNotForms();
            click_cnt += (form_click_cnt + notForm_click_cnt);
            form_effect_cnt += ad.getSumEffectPVForms();
            notForm_effect_cnt += ad.getSumEffectPVNotForms();
            effect_cnt += (form_effect_cnt + notForm_effect_cnt);
        }

        double cvr = click_cnt > 0 ? effect_cnt/click_cnt : 0;
        double formCvrBias = form_effect_cnt < form_click_cnt ? WilsonInterval.wilsonCalc(form_effect_cnt,form_click_cnt).lowerBound : WilsonInterval.wilsonCalc(form_click_cnt,form_click_cnt).lowerBound;
        double notFormCvrBias = notForm_effect_cnt < notForm_click_cnt ? WilsonInterval.wilsonCalc(notForm_effect_cnt,notForm_click_cnt).lowerBound : WilsonInterval.wilsonCalc(notForm_click_cnt, notForm_click_cnt).lowerBound;
        double cvrBias = formCvrBias + 0.2 *  notFormCvrBias;

        // 广告位的总请求
        double requests = 0;
        // 广告位的总消耗
        double costs = 0;
        // 广告位的总计费点击
        double clicks = 0;
        for(ActivityInfoData act : actData){
            requests += act.getRequest().slotVal;
            costs += act.getCost().slotVal;
            clicks += act.getClick().slotVal;
        }
        double uc = costs/requests;
        double uck = clicks/requests;

        // 优质活动详情数据
        Map<Long, NormInfo> exHashMap=new HashMap<>();
        if(exActs!=null) {
            for(NormInfo exAct : exActs){
                exHashMap.put(exAct.getActivityId(), exAct);
            }
        }

        //优质活动id集合
        Set<Long> actIds=new HashSet<>(exHashMap.keySet());
        //广告位下投放过活动id集合
        Set<Long> hisActIds=new HashSet<>(actData.stream().map(ActivityInfoData::getActivityId).collect(Collectors.toSet()));

        // 海选池活动id占位
        HashSet<Long> idset=new HashSet();
        for (ActivityInfoData act : actData) {
            try {
                // 封装数据
                ActivityMatchInfo actM=fillData(act, actIds, exHashMap);
                actM.setUvCost(uc);
                actM.setUvClick(uck);
                actM.setCvr(cvr);
                actM.setCvrBias(cvrBias);
                actModel.add(actM);

                // 新活动试投，上架时间小于3天
                if (actM.hisRequest.globalVal < 100) {
                    if (System.currentTimeMillis() - act.createTime < 3 * 24 * 60 * 60 * 1000) {
                        if (Math.random() < 0.0001) {
                            result.add(actM);
                            topn--;
                            idset.add(actM.activityId);
                        }
                    } else if (actData.size() > topn) {
                        if (Math.random() < 0.00001) {
                            result.add(actM);
                            topn--;
                            idset.add(actM.activityId);
                        }
                    }
                }else if(actM.hisRequest.globalVal < 500){
                    if(actM.hisEffect.globalVal < 5){
                        // 转化少以0.1%概率占位
                        if(Math.random() < 0.001){
                            result.add(actM);
                            topn--;
                            idset.add(actM.activityId);
                        }
                    }
                }

                // 置信每uv消耗
                double wuc = getMatchScore(actM.hisRequest, actM.hisCost,100d,1000d, true);
                // 置信每uv券点击
                double wuck = getMatchScore(actM.hisRequest, actM.hisClick, 100d, 1000d, false);
                // 置信cvr
                double wcvr = getMatchScore(actM.hisClick, actM.hisEffect, 100d, 400d, false);

                double changeScore=0.0d;
                ActivityCvrInfo activityChangeValNew=new ActivityCvrInfo();

                if (adHashMap.containsKey(act.activityId)) {
                    activityChangeValNew=calChangeScoreMergeAbsolute(adHashMap.getOrDefault(act.activityId, new ActivityInfoAd()),cvrBias);
                    changeScore=activityChangeValNew.changeScoreMerge;
                }

                actM.changeScore=changeScore;
                actM.activityChangeVal=activityChangeValNew;

                // 归一化
                if(actM.isExcellent){
                    // 已投放过的优质活动召回
                    MatchInfo emi = new MatchInfo();
                    emi.act=actM;
                    emi.score = actM.score;
                    normCandis.add(emi);
                }else{
                    MatchInfo wucmi = new MatchInfo();
                    wucmi.act=actM;
                    wucmi.score=wuc;
                    gCandisUC.add(wucmi);
                    MatchInfo wuckmi = new MatchInfo();
                    wuckmi.act=actM;
                    wuckmi.score=wuck;
                    gCandisUCK.add(wuckmi);
                    MatchInfo wcvrmi = new MatchInfo();
                    wcvrmi.act=actM;
                    wcvrmi.score=wcvr;
                    gCandisCVR.add(wcvrmi);
                    MatchInfo rcvrmi = new MatchInfo();
                    rcvrmi.act=actM;
                    rcvrmi.score=actM.hisEffect.slotVal/actM.hisClick.slotVal;
                    candisCVR.add(rcvrmi);
                    MatchInfo urmi = new MatchInfo();
                    urmi.act=actM;
                    urmi.score=actM.hisEffect.slotVal/actM.hisRequest.slotVal;
                    candisUR.add(urmi);
                    //广告位请求数大于99 或点击大于5
                    if (actM.hisRequest.slotVal > 99 || actM.hisClick.slotVal > 5) {
                        MatchInfo ucmi=new MatchInfo();
                        ucmi.act=actM;
                        // 消耗/请求数
                        ucmi.score=actM.hisCost.slotVal /(100 * actM.hisRequest.slotVal);
                        // 广告位每uv消耗召回
                        sCandisUC.add(ucmi);
                        MatchInfo uckmi=new MatchInfo();
                        uckmi.act=actM;
                        // 券点击/请求数
                        uckmi.score=actM.hisClick.slotVal/actM.hisRequest.slotVal;
                        // 广告位每uv券点击召回
                        sCandisUCK.add(uckmi);
                    }
                    if(actM.hisClick.slotVal > 99 || actM.hisEffect.slotVal > 5){
                        MatchInfo cvrmi = new MatchInfo();
                        cvrmi.act=actM;
                        cvrmi.score=actM.hisEffect.slotVal/actM.hisClick.slotVal;
                        // 真实cvr召回
                        sCandisCVR.add(cvrmi);
                    }
                }
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
        }

        // 广告位每uv消耗top10占位
        for(int i=0; i<Constant.RECALL_SUC_TOPN && i<sCandisUC.size(); i++){
            ActivityMatchInfo act = sCandisUC.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }
        // 全局每uv消耗top10占位
        for(int i=0; i<Constant.RECALL_GUC_TOPN && i<gCandisUC.size(); i++){
            ActivityMatchInfo act = gCandisUC.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }
        // 广告位每uv券点击top10占位
        for(int i=0; i<Constant.RECALL_SUCK_TOPN && i<sCandisUCK.size(); i++){
            ActivityMatchInfo act = sCandisUCK.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }
        // 全局每uv券点击top10占位
        for(int i=0; i<Constant.RECALL_GUCK_TOPN && i<gCandisUCK.size(); i++){
            ActivityMatchInfo act = gCandisUCK.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }
        // 广告位cvr top10占位
        for(int i=0; i<Constant.RECALL_SCVR_TOPN && i<sCandisCVR.size(); i++){
            ActivityMatchInfo act = sCandisCVR.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }
        // 全局cvr top10占位
        for(int i=0; i<Constant.RECALL_GCVR_TOPN && i<gCandisCVR.size(); i++){
            ActivityMatchInfo act = gCandisCVR.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }
        // 真实cvr top10占位
        for(int i=0; i<Constant.RECALL_CVR_TOPN && i<candisCVR.size(); i++){
            ActivityMatchInfo act = candisCVR.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }
        // 真实每uv券点击*真实cvr top10占位
        for(int i=0;i<Constant.RECALL_UR_TOPN && i<candisUR.size(); i++){
            ActivityMatchInfo act = candisUR.poll().act;
            if(idset.contains(act.activityId))
                continue;
            result.add(act);
            idset.add(act.activityId);
            topn--;
        }

        // 未在特定广告位有历史投放记录的活动
        if(exActs!=null){
            for(NormInfo exAct : exActs){
                if(!hisActIds.contains(exAct.getActivityId())){
                    MatchInfo mci=new MatchInfo();
                    ActivityMatchInfo aim=new ActivityMatchInfo();
                    aim.setAppId(actData.get(0).getAppId());
                    aim.setSlotId(actData.get(0).getSlotId());
                    aim.setActivityId(exAct.getActivityId());
                    aim.setnPvClick(exAct.getnPvClick());
                    aim.setnPvCost(exAct.getnPvCost());
                    aim.setnCvrForm(exAct.getnFormCvr());
                    aim.setnCvrNotForm(exAct.getnNotFormCvr());
                    aim.setnCvr(exAct.getnCvr());
                    aim.setScore(exAct.getScore());
                    aim.setExcellent(true);
                    mci.score=exAct.getScore();
                    mci.act=aim;
                    normCandis.add(mci);
                }
            }

            for (int i=0; i < exTopN && i < normCandis.size(); i++) {
                MatchInfo candisInfo= normCandis.poll();
                if(idset.contains(candisInfo.act.activityId))
                    continue;
                result.add(candisInfo.act);
                idset.add(candisInfo.act.activityId);
                topn--;
            }
        }

        return result;
    }

    /**
     * @author: lijingzhe
     * @date: 2019/11/18
     * @methodParameters:
     * @methodReturnType:
     * @description: 计算全局置信值
     */
    private static double getMatchScore(ActivityVal num, ActivityVal den, Double thr1, Double thr2, Boolean isUC) {
        double slotScore=WilsonInterval.wilsonCalc(num.slotVal, den.slotVal).lowerBound;
        double globalScore=WilsonInterval.wilsonCalc(num.globalVal, den.globalVal).lowerBound;
        double appScore=WilsonInterval.wilsonCalc(num.appVal, den.appVal).lowerBound;
        if(isUC){
            slotScore=WilsonInterval.wilsonCalc(num.slotVal, den.slotVal/100).lowerBound;
            globalScore=WilsonInterval.wilsonCalc(num.globalVal, den.globalVal/100).lowerBound;
            appScore=WilsonInterval.wilsonCalc(num.appVal, den.appVal/100).lowerBound;
        }

        double sconfidence=Math.min(den.slotVal / thr1, 1);
        double aconfidence=Math.min(den.appVal / thr1, 1);
        double gconfidence=Math.min(den.globalVal / thr2, 1);

        double matchscore=sconfidence * slotScore
                +(1 - sconfidence) * aconfidence * appScore * 0.9
                +(1 - sconfidence - (1 - sconfidence) * aconfidence) * globalScore * Math.max(0.5, gconfidence);

        return matchscore;
    }

    /**
     * @author: lijingzhe
     * @date: 2019/11/13
     * @methodParameters:
     * @methodReturnType:
     * @description: 4.5优选，在4.2基础上增加二次约束和组合推荐
     */
    public static RankResult select(List<ActivityRankInfo> actModel, List<ActivityMatchInfo> actData, List<SlotAlgRecInfo> saris) {
        RankResult rankResult=new RankResult();
        Map<Long,ActivityMatchInfo> maps = new HashMap<>();
        for (ActivityMatchInfo ami : actData) {
            maps.put(ami.activityId,ami);
        }

        //判断列表长度，不一致抛异常
        if (actModel.size() == 0 || actData.size() == 0) {
            return null;
        }

        if (actModel.size() != actData.size()) {
            rankResult.activityModel=actModel.get(0);
            rankResult.similarCostActivitiesInfo=null;
            return rankResult; //todo 抛异常
        }
        //活动id倒排
        List<ActivityRankInfo> actModelCopy =
                actModel.stream()
                        .sorted(Comparator.comparing(ActivityRankInfo::getActivityId).reversed())
                        .collect(Collectors.toList());
        List<ActivityMatchInfo> actDataCopy =
                actData.stream()
                        .sorted(Comparator.comparing(ActivityMatchInfo::getActivityId).reversed())
                        .collect(Collectors.toList());

        //活动 id 对不上返回空
        for (int i=0; i < actModelCopy.size(); ++i) {
            long activityIdMode=actModelCopy.get(i).activityId;
            long activityIdData=actDataCopy.get(i).activityId;
            if (activityIdMode != activityIdData) {
                rankResult.activityModel=actModel.get(0);
                rankResult.similarCostActivitiesInfo=null;
                return rankResult; // todo 抛异常
            }

            actModelCopy.get(i).changeScore=actDataCopy.get(i).changeScore;
            actModelCopy.get(i).isExcellent=actDataCopy.get(i).isExcellent;
            actModelCopy.get(i).nPvClick=actDataCopy.get(i).nPvClick;
            actModelCopy.get(i).nPvCost=actDataCopy.get(i).nPvCost;
            actModelCopy.get(i).score=actDataCopy.get(i).score;
            actModelCopy.get(i).subType=actDataCopy.get(i).subType;
            actModelCopy.get(i).cvrBias = actDataCopy.get(i).cvrBias;
            actModelCopy.get(i).uvCost = actDataCopy.get(i).uvCost;
            actModelCopy.get(i).uvClick = actDataCopy.get(i).uvClick;
            actModelCopy.get(i).cvr = actDataCopy.get(i).cvr;
        }

        Map<String,Double> rates = getSlotAlgStatus(actData,saris,maps);

        List<SelectPoolInfo>  sinfos=new ArrayList<>();
        List<ActivityRankInfo> candiList = new ArrayList<>();
        ActivityRankInfo result=new ActivityRankInfo();

        try {
            // uv消耗二次分布
            candiList = getSingleOptCandis(sinfos, actDataCopy, actModelCopy, Constant.ALG_UC,100,100);

            // 二次约束和组合推荐
            SelectInfoDetail selectInfoDetail =
                    selectMultiOptInfo(sinfos, rates, actDataCopy, actModelCopy, result);

            int selectedActivityIndex=selectInfoDetail.index;
            result=candiList.get(selectedActivityIndex);
            result.strategyType=selectInfoDetail.strategyType;
            double maxChangeScore=selectInfoDetail.maxChangeScore;

            List<SimilarCostActivitiesInfo> sameCostActivitiesInfoList=new ArrayList<>();
            //消耗分相近的活动信息
            ArrayList<SelectPoolInfo> sameCostcondi=selectInfoDetail.condi;
            if(sameCostcondi.size() > 0){
                double selectedActivityReward=sameCostcondi.get(0).getReward();
                int sameCostActivityCnt=sameCostcondi.size();

                for (SelectPoolInfo selectInfo:sameCostcondi){

                    SimilarCostActivitiesInfo sameCostActivitiesInfo=new SimilarCostActivitiesInfo();

                    Long activityId=selectInfo.getActivityId();
                    Long activityType=selectInfo.getActivityType();

                    int ifSelected=0;
                    if (selectInfo.getIndex() == selectedActivityIndex){
                        ifSelected=1;
                    }

                    int index=selectInfo.getIndex();

                    double mabReward=selectInfo.getReward();
                    double changeScore=selectInfo.getChangeScore();
                    double hrpm=selectInfo.getHrpm();
                    double arpm=selectInfo.getArpm();
                    double grpm=selectInfo.getGrpm();
                    int subType=selectInfo.getSubType();
                    boolean excellent=selectInfo.isExcellent();

                    //计算与最优的差异
                    double diffReward= selectInfo.getReward()-selectedActivityReward;
                    double diffChangeScore=selectInfo.getChangeScore() - maxChangeScore;

                    //封装
                    sameCostActivitiesInfo.activityId=activityId;
                    sameCostActivitiesInfo.sameActivityCnt=sameCostActivityCnt;
                    sameCostActivitiesInfo.activityType=activityType;
                    sameCostActivitiesInfo.ifSelected=ifSelected;
                    sameCostActivitiesInfo.mabReward=mabReward;
                    sameCostActivitiesInfo.changeScore=changeScore;
                    sameCostActivitiesInfo.index=index;
                    sameCostActivitiesInfo.hrpm=hrpm;
                    sameCostActivitiesInfo.arpm=arpm;
                    sameCostActivitiesInfo.grpm=grpm;
                    sameCostActivitiesInfo.diffReward=diffReward;
                    sameCostActivitiesInfo.diffChangeScore=diffChangeScore;
                    sameCostActivitiesInfo.subType=subType;
                    sameCostActivitiesInfo.isExcellent=excellent;
                    sameCostActivitiesInfoList.add(sameCostActivitiesInfo);
                }
            }

            rankResult.activityModel=result;
            rankResult.similarCostActivitiesInfo=sameCostActivitiesInfoList;

            return rankResult;
        }catch (Exception e)
        {
            e.printStackTrace();
            logger.error(e.getMessage(), e);
            logger.error("error, size:{},candi:{},list:{},", 0, JSON.toJSONString(candiList), JSON.toJSONString(result));
        }
        return null;
    }

    private static Map<String, Double> getSlotAlgStatus(List<ActivityMatchInfo> actData, List<SlotAlgRecInfo> saris,Map<Long,ActivityMatchInfo> maps) {
        Map<String,Double> rates = new HashMap();

        double algCnt = 0;
        double algCost = 0;
        double algClick = 0;
        double algEffect = 0;
        // 广告位算法推荐状态
        for (SlotAlgRecInfo sari : saris) {
            algCnt += sari.algCnt;
            algCost += maps.getOrDefault(sari.activityId, new ActivityMatchInfo()).uvCost * algCnt;
            algClick += maps.getOrDefault(sari.activityId, new ActivityMatchInfo()).uvClick * algCnt;
            algEffect += algCnt * maps.getOrDefault(sari.activityId, new ActivityMatchInfo()).uvClick * maps.getOrDefault(sari.activityId, new ActivityMatchInfo()).cvr;
        }
        // 广告位算法推荐组合状态参数(algCnt,algUC,algUCK,algCVR,algUR)——组合推荐状态
        double algUC = algCost/algCnt;
        double algUCK = algClick/algCnt;
        double algCVR = algEffect/algClick;
        double algUR = algEffect/algCnt;

        double baseUC = actData.get(0).uvCost;
        double baseUCK = actData.get(0).uvCost;
        double baseCVR = actData.get(0).cvr;
        double baseUR = baseUCK * baseCVR;
        rates.put(Constant.ALG_UC, algUC/baseUC);
        rates.put(Constant.ALG_UCK, algUCK/baseUCK);
        rates.put(Constant.ALG_CVR, algCVR/baseCVR);
        rates.put(Constant.ALG_UR, algUR/baseUR);
        return rates;
    }

    /**
     * @author: lijingzhe
     * @date: 2019/11/19
     * @methodParameters:
     * @methodReturnType:
     * @description: 单目标收益二项分布
     */
    private static List<ActivityRankInfo> getSingleOptCandis(List<SelectPoolInfo> sinfos, List<ActivityMatchInfo> actDataCopy, List<ActivityRankInfo> actModelCopy, String oppALg, int hthr, int athr) {
        // 活动访问pv
        ArrayList<Double> requests=new ArrayList<>();
        // 每pv消耗收益
        ArrayList<Double> rewards=new ArrayList<>();
        ArrayList<Double> counts=new ArrayList<>();
        // 贝塔分布alpha和beta
        ArrayList<Double> alphas=new ArrayList<>();
        ArrayList<Double> betas=new ArrayList<>();
        // 转化分
        ArrayList<Double> changeScores=new ArrayList<>();
        // 广告位置信cvr
        ArrayList<Double> cvrBiass = new ArrayList<>();
        ArrayList<Long> activityIds=new ArrayList<>();
        ArrayList<Long> activityTypes=new ArrayList<>();
        // 每pv消耗
        ArrayList<Double> grpms=new ArrayList<>();
        ArrayList<Double> arpms=new ArrayList<>();
        ArrayList<Double> hrpms=new ArrayList<>();
        // 每uv券点击
        ArrayList<Double> uvClicks=new ArrayList<>();
        // cvr
        ArrayList<Double> cvrs=new ArrayList<>();
        // 是否是优质活动
        ArrayList<Boolean> excellents=new ArrayList<>();
        ArrayList<Integer> subTypes=new ArrayList<>();
        // 归一化score
        ArrayList<Double> scores=new ArrayList<>();

        List<ActivityRankInfo> candiList = new ArrayList();

        HashMap<Long, RankInfo> mMap=new HashMap();
        double maxG= Constant.MIN_REWARD, maxH= Constant.MIN_REWARD, maxA= Constant.MIN_REWARD;
        double minG=0, minH= Constant.MIN_REWARD, minA= Constant.MIN_REWARD;
        double oppH=0,oppA=0,oppG=0;

        for (int i=0;i<actDataCopy.size();++i) {
            RankInfo info=mMap.containsKey(actDataCopy.get(i).activityId) ? mMap.get(actDataCopy.get(i).activityId) : new RankInfo();
            if(oppALg.equals(Constant.ALG_UC)){
                oppG=actDataCopy.get(i).hisRequest.globalVal > 0 ? actDataCopy.get(i).hisCost.globalVal / actDataCopy.get(i).hisRequest.globalVal : 0;
                info.grpm=oppG;
                info.gexp=actDataCopy.get(i).hisRequest.globalVal;
                maxG=Math.max(oppG, maxG);

                oppA=actDataCopy.get(i).hisRequest.appVal > 0 ? actDataCopy.get(i).hisCost.appVal / actDataCopy.get(i).hisRequest.appVal : 0;
                info.arpm=oppA;
                info.aexp=actDataCopy.get(i).hisRequest.appVal;
                if(info.aexp > athr){
                    maxA=Math.max(oppA, maxA);
                }
                minA=Math.min(oppA, minA);

               oppH=actDataCopy.get(i).hisRequest.slotVal > 0 ? actDataCopy.get(i).hisCost.slotVal / actDataCopy.get(i).hisRequest.slotVal : 0;
               info.hrpm=oppH;
               info.hexp=actDataCopy.get(i).hisRequest.slotVal;
               if(info.hexp > hthr){
                   maxH=Math.max(oppH,maxH);
               }
               minH=Math.min(oppH,minH);
            }else if(oppALg.equals(Constant.ALG_UCK)){
                oppG=actDataCopy.get(i).hisRequest.globalVal > 0 ? actDataCopy.get(i).hisClick.globalVal / actDataCopy.get(i).hisRequest.globalVal : 0;
                info.grpm=oppG;
                info.gexp=actDataCopy.get(i).hisRequest.globalVal;
                maxG=Math.max(oppG, maxG);

                oppA=actDataCopy.get(i).hisRequest.appVal > 0 ? actDataCopy.get(i).hisClick.appVal / actDataCopy.get(i).hisRequest.appVal : 0;
                info.arpm=oppA;
                info.aexp=actDataCopy.get(i).hisRequest.appVal;
                if(info.aexp > athr){
                    maxA=Math.max(oppA, maxA);
                }
                minA=Math.min(oppA, minA);

                oppH=actDataCopy.get(i).hisRequest.slotVal > 0 ? actDataCopy.get(i).hisClick.slotVal / actDataCopy.get(i).hisRequest.slotVal : 0;
                info.hrpm=oppH;
                info.hexp=actDataCopy.get(i).hisRequest.slotVal;
                if(info.hexp > hthr){
                    maxH=Math.max(oppH,maxH);
                }
                minH=Math.min(oppH,minH);
            }else if(oppALg.equals(Constant.ALG_CVR)){
                oppG=actDataCopy.get(i).hisClick.globalVal > 0 ? actDataCopy.get(i).hisEffect.globalVal / actDataCopy.get(i).hisClick.globalVal : 0;
                info.grpm=oppG;
                info.gexp=actDataCopy.get(i).hisClick.globalVal;
                maxG=Math.max(oppG, maxG);

                oppA=actDataCopy.get(i).hisClick.appVal > 0 ? actDataCopy.get(i).hisEffect.appVal / actDataCopy.get(i).hisClick.appVal : 0;
                info.arpm=oppA;
                info.aexp=actDataCopy.get(i).hisClick.appVal;
                if(info.aexp > athr){
                    maxA=Math.max(oppA, maxA);
                }
                minA=Math.min(oppA, minA);

                oppH=actDataCopy.get(i).hisClick.slotVal > 0 ? actDataCopy.get(i).hisEffect.slotVal / actDataCopy.get(i).hisClick.slotVal : 0;
                info.hrpm=oppH;
                info.hexp=actDataCopy.get(i).hisRequest.slotVal;
                if(info.hexp > hthr){
                    maxH=Math.max(oppH,maxH);
                }
                minH=Math.min(oppH,minH);
            }else if(oppALg.equals(Constant.ALG_UR)){
                oppG=actDataCopy.get(i).hisRequest.globalVal > 0 ? actDataCopy.get(i).hisEffect.globalVal / actDataCopy.get(i).hisRequest.globalVal : 0;
                info.grpm=oppG;
                info.gexp=actDataCopy.get(i).hisRequest.globalVal;
                maxG=Math.max(oppG, maxG);

                oppA=actDataCopy.get(i).hisRequest.appVal > 0 ? actDataCopy.get(i).hisEffect.appVal / actDataCopy.get(i).hisRequest.appVal : 0;
                info.arpm=oppA;
                info.aexp=actDataCopy.get(i).hisRequest.appVal;
                if(info.aexp > athr){
                    maxA=Math.max(oppA, maxA);
                }
                minA=Math.min(oppA, minA);

                oppH=actDataCopy.get(i).hisRequest.slotVal > 0 ? actDataCopy.get(i).hisEffect.slotVal / actDataCopy.get(i).hisRequest.slotVal : 0;
                info.hrpm=oppH;
                info.hexp=actDataCopy.get(i).hisRequest.slotVal;
                if(info.hexp > hthr){
                    maxH=Math.max(oppH,maxH);
                }
                minH=Math.min(oppH,minH);
            }
            mMap.put(actDataCopy.get(i).activityId, info);
        }

        for (int i=0;i<actDataCopy.size();++i) {
            double sUvClick = actDataCopy.get(i).hisRequest.slotVal > 0 ? actDataCopy.get(i).hisClick.slotVal/actDataCopy.get(i).hisRequest.slotVal : 0;
            double sCvr = actDataCopy.get(i).hisClick.slotVal > 0 ? actDataCopy.get(i).hisEffect.slotVal/actDataCopy.get(i).hisClick.slotVal : 0;
            uvClicks.add(sUvClick);
            cvrs.add(sCvr);
            requests.add(actDataCopy.get(i).hisRequest.slotVal);
            double hrpm=mMap.get(actDataCopy.get(i).activityId).hrpm;
            double arpm=mMap.get(actDataCopy.get(i).activityId).arpm;
            double grpm=mMap.get(actDataCopy.get(i).activityId).grpm;
            hrpms.add(hrpm);
            arpms.add(arpm);
            grpms.add(grpm);
            excellents.add(actDataCopy.get(i).isExcellent);
            subTypes.add(actDataCopy.get(i).subType);
            scores.add(actDataCopy.get(i).score);
            cvrBiass.add(actDataCopy.get(i).cvrBias);

            double reward= Constant.MIN_REWARD;
            double sconfidence=Math.min(actDataCopy.get(i).hisRequest.slotVal / hthr, 1);
            double aconfidence=Math.min(actDataCopy.get(i).hisRequest.appVal / athr, 1);

            reward=sconfidence * normlize( hrpm * 0.8, maxH, 0.8) +
                    (1 - sconfidence) * aconfidence * normlize(arpm * 0.7, maxA, 0.7) +
                    (1 - sconfidence - (1 - sconfidence) * aconfidence) * normlize(grpm * 0.5, maxG, 0.6);

            reward=reward * reward;

            reward=Math.max(reward, 0);

            actModelCopy.get(i).reward=actModelCopy.get(i).reward * Constant.DECAY+reward;
            actModelCopy.get(i).count=actModelCopy.get(i).count * Constant.DECAY+1.0;
            actModelCopy.get(i).alpha=1.5+actModelCopy.get(i).reward;
            actModelCopy.get(i).beta=2.0+(actModelCopy.get(i).count - actModelCopy.get(i).reward);

            rewards.add(actModelCopy.get(i).reward);
            counts.add(actModelCopy.get(i).count);
            alphas.add(actModelCopy.get(i).alpha);
            betas.add(actModelCopy.get(i).beta);
            activityIds.add(actModelCopy.get(i).activityId);
            activityTypes.add(actModelCopy.get(i).activityType);
            changeScores.add(actModelCopy.get(i).changeScore);
            candiList.add(actModelCopy.get(i));
        }
        mMap.clear();
        SelectInfoDetail selectInfoDetail=new SelectInfoDetail();
        getSinfos(sinfos, candiList.size(),alphas,betas,activityTypes,activityIds,changeScores,hrpms,arpms,grpms,requests,excellents,scores,subTypes,cvrBiass,uvClicks,cvrs);
        return candiList;
    }


    /**
     *功能描述 融合表单类和非表单类cvr
     *
     * @author lijingzhe
     * @date 2019/9/9
     * @param ad
     * @return cn.com.duiba.nezha.alg.common.model.activityrecommend.ActivityCvrInfo
     */
    public static ActivityCvrInfo calChangeScoreMergeAbsolute(ActivityInfoAd ad,double cvr) {
        double changeScoreBias = 0.05; //统计腰部+尾部流量(pv>100)的均值cvr
        double changeScoreMerge=0.0;

        double changeScoreForms = 0.0;  //该活动下
        double changeScoreNotForms = 0.0;

        // 该活动表单类置信cvr
        if(ad.sumClickForms > ad.sumEffectPVForms){
            changeScoreForms = WilsonInterval.wilsonCalc(ad.sumEffectPVForms, ad.sumClickForms).lowerBound;
        }else{
            changeScoreForms = WilsonInterval.wilsonCalc(ad.sumClickForms, ad.sumClickForms).lowerBound;
        }

        // 该活动非表单类置信cvr
        if(ad.sumClickNotForms > ad.sumEffectPVNotForms){
            changeScoreNotForms = WilsonInterval.wilsonCalc(ad.sumEffectPVNotForms, ad.sumClickNotForms).lowerBound;
        }else{
            changeScoreNotForms = WilsonInterval.wilsonCalc(ad.sumClickNotForms, ad.sumClickNotForms).lowerBound;
        }

        double wcvr = cvr > 0 ? changeScoreBias/cvr :1;
        changeScoreMerge = changeScoreForms + changeScoreNotForms * 0.2 * Math.min(2.5, wcvr);

        //输出 转化得分明细
        ActivityCvrInfo  activityChangeVal= new ActivityCvrInfo() ;

        activityChangeVal.sumClickForms=ad.sumClickForms;
        activityChangeVal.sumEffectPVForms=ad.sumEffectPVForms;

        activityChangeVal.sumClickNotForms=ad.sumClickNotForms;
        activityChangeVal.sumEffectPVNotForms=ad.sumEffectPVNotForms;

        activityChangeVal.changeScoreForms= changeScoreForms;
        activityChangeVal.changeScoreNotForms= changeScoreNotForms;

        activityChangeVal.changeScoreMerge=changeScoreMerge;
        //结果输出
        return activityChangeVal ;

    }

    public static double normlize(double val, double max, double limit) {
        double norm=Math.min(val * limit / max, limit);
        return norm;
    }

    /**
     *功能描述 标准化到区间[lower,upper]
     * @author lijingzhe
     * @date 2019/8/22
     * @param val, min, max, lower, upper
     * @return double
     */
    public static double normlize(double val, double min, double max,
                                  double lower, double upper) {
        double norm =
                upper > lower && upper <= 1 && lower >= 0 ?
                        (upper - lower) * (val - min) / (max - min)+lower :
                        0.6 * (val - min) / (max - min)+0.2 ;
        return norm;
    }

    /**
     * @author: lijingzhe
     * @date: 2019/11/20
     * @methodParameters:
     * @methodReturnType:
     * @description: 兼容+二次约束+组合推荐策略
     */
    private static SelectInfoDetail selectMultiOptInfo(List<SelectPoolInfo> sinfos, Map<String,Double> rates, List<ActivityMatchInfo> actDataCopy, List<ActivityRankInfo> actModelCopy, ActivityRankInfo result) {
        SelectInfoDetail selectInfoDetail = new SelectInfoDetail();
        if(sinfos.size() > 0){
            // 非卡包活动转化分最高活动
            List<SelectPoolInfo> chs = sinfos.stream()
                    .sorted(Comparator.comparing(SelectPoolInfo::getChangeScore).reversed())
                    .collect(Collectors.toList());
            double maxChangeScore = chs.get(0).getChangeScore();
            double midChangeScore = chs.get(Math.min(chs.size()/2+1, 10)).getChangeScore();

            List<SelectPoolInfo> ranks = sinfos.stream()
                    .sorted(Comparator.comparing(SelectPoolInfo::getReward).reversed())
                    .collect(Collectors.toList());

            // 投放次数>=100 && 投放次数<=500的活动，优质活动兜底
            for (int i = 0; i < ranks.size(); i++) {
                if(ranks.get(i).getRequest() < 100){
                    if(ranks.get(i).isExcellent() && Math.random() < 0.02){
                        selectInfoDetail.condi = new ArrayList<>(ranks.subList(0, Math.min(ranks.size(),5)));
                        selectInfoDetail.index = ranks.get(i).getIndex();
                        selectInfoDetail.maxChangeScore = maxChangeScore;
                        selectInfoDetail.strategyType = 1;
                        break;
                    }
                    // 转化差的活动低概率投放
                    if(ranks.get(i).getChangeScore() < 0.8 * ranks.get(i).getCvrBias() && Math.random() < 0.001){
                        selectInfoDetail.condi = new ArrayList<>(ranks.subList(0, Math.min(ranks.size(),5)));
                        selectInfoDetail.index = ranks.get(i).getIndex();
                        selectInfoDetail.maxChangeScore = maxChangeScore;
                        selectInfoDetail.strategyType = -1;
                        break;
                    }
                }else if(ranks.get(i).getRequest() >= 100 && ranks.get(i).getRequest() <= 500){
                    // 排名top 2-5 每个活动以1%概率随机投放
                    if(ranks.get(i).isExcellent() == true && i > 0 && i < 5 && Math.random() < 0.01){
                        selectInfoDetail.condi=
                                new ArrayList<>(ranks.subList(0, Math.min(ranks.size(),5)));
                        selectInfoDetail.index=ranks.get(i).getIndex();
                        selectInfoDetail.maxChangeScore=maxChangeScore;
                        selectInfoDetail.strategyType = 2;
                        break;
                        // 排名top 5-20 每个活动以0.1%概率投放
                    }else if(ranks.get(i).isExcellent() == true && i < 20 && Math.random() < 0.001) {
                        selectInfoDetail.condi =
                                new ArrayList<>(ranks.subList(0, Math.min(ranks.size(), 5)));
                        selectInfoDetail.index = ranks.get(i).getIndex();
                        selectInfoDetail.maxChangeScore = maxChangeScore;
                        selectInfoDetail.strategyType = 3;
                        break;
                    }

                    if(ranks.get(i).getChangeScore() <0.8 * ranks.get(i).getCvrBias() && Math.random() < 0.0002){
                        selectInfoDetail.condi =
                                new ArrayList<>(ranks.subList(0, Math.min(ranks.size(), 5)));
                        selectInfoDetail.index = ranks.get(i).getIndex();
                        selectInfoDetail.maxChangeScore = maxChangeScore;
                        selectInfoDetail.strategyType = -2;
                        break;
                    }
                }

                // 真实cvr约束+真实uv券点击*真实cvr二次约束
                if(ranks.get(i).getsCvr() >= 0.8 * ranks.get(i).getCvr() && ranks.get(i).getChangeScore() >= 0.6 * ranks.get(i).getCvr()){
                    if(ranks.get(i).getHrpm() >= ranks.get(i).getUvCost()){
                        if(ranks.get(i).getsUvClick() * ranks.get(i).getsCvr() >= ranks.get(i).getUvClick() * ranks.get(i).getCvr()){
                            selectInfoDetail.condi =
                                    new ArrayList<>(ranks.subList(0, Math.min(ranks.size(), 5)));
                            //最优活动索引
                            selectInfoDetail.index = ranks.get(i).getIndex();
                            //最大转化分
                            selectInfoDetail.maxChangeScore = maxChangeScore;
                            selectInfoDetail.strategyType = 4;
                            break;
                        }
                    }
                }else{
                    // 组合推荐
                      selectInfoDetail = getPairWise(actDataCopy, actModelCopy, ranks, maxChangeScore, rates);
                }
            }

            if(selectInfoDetail==null || selectInfoDetail.strategyType == 0){
                List<SelectPoolInfo> rkf = ranks.stream()
                        .filter(e->e.getChangeScore() >= midChangeScore)
                        .collect(Collectors.toList());
                selectInfoDetail.index = ranks.get(0).getIndex();
                selectInfoDetail.maxChangeScore = ranks.get(0).getChangeScore();
                selectInfoDetail.condi = new ArrayList<>(ranks.subList(0, Math.min(ranks.size(), 5)));
                selectInfoDetail.strategyType = 9;
            }
        }else{
            selectInfoDetail.index = 0;
            selectInfoDetail.maxChangeScore = sinfos.get(0).getChangeScore();
            selectInfoDetail.condi = new ArrayList<>();
            selectInfoDetail.strategyType = 0;
        }
        return selectInfoDetail;
    }

    /**
     * @author: lijingzhe
     * @date: 2019/11/20
     * @methodParameters:
     * @methodReturnType:
     * @description: 过滤卡包，计算收益并封装对象
     */
    private static void getSinfos(List<SelectPoolInfo> sinfos, int numMachines, List<Double> alphas, List<Double> betas, List<Long> activityTypes, List<Long> activityIds, List<Double> changeScores, List<Double> hrpms, List<Double> arpms, List<Double> grpms, List<Double> requests, List<Boolean> excellents, List<Double> scores, List<Integer> subTypes, List<Double> cvrBiass, List<Double> uvClicks, List<Double> cvrs) {
        for (int i=0; i < numMachines; i++) {
            if(activityTypes.get(i) != 21){
                double theta=BetaDistribution.BetaDist(alphas.get(i), betas.get(i));
                SelectPoolInfo sinfo=new SelectPoolInfo();
                sinfo.setReward(theta);
                sinfo.setActivityType(activityTypes.get(i));
                sinfo.setIndex(i);
                sinfo.setActivityId(activityIds.get(i));
                sinfo.setChangeScore(changeScores.get(i));
                sinfo.setHrpm(hrpms.get(i));
                sinfo.setArpm(arpms.get(i));
                sinfo.setGrpm(grpms.get(i));
                sinfo.setRequest(requests.get(i));
                sinfo.setExcellent(excellents.get(i));
                sinfo.setScores(scores.get(i));
                sinfo.setSubType(subTypes.get(i));
                sinfo.setCvrBias(cvrBiass.get(i));
                sinfo.setsUvClick(uvClicks.get(i));
                sinfo.setsCvr(cvrs.get(i));
                sinfos.add(sinfo);
            }
        }
    }

    /**
     * @author: lijingzhe
     * @date: 2019/11/19
     * @methodParameters:
     * @methodReturnType:
     * @description: 组合对冲策略
     * 1. 获取广告位算法推荐状态参数(algUC, algUCK, algCVR, algUR)
     * 2. 计算目标系中min(algX/X)对应的系Y——Y1
     * 3. 筛选Y in (!X) > baseY的Y2
     * 4. Y2 = null ? max(Y1) : max(Y2)
     */
    private static SelectInfoDetail getPairWise(List<ActivityMatchInfo> actDataCopy, List<ActivityRankInfo> actModelCopy, List<SelectPoolInfo> ranks,Double maxChangeScore, Map<String,Double> rates) {
        SelectInfoDetail selectInfoDetail = new SelectInfoDetail();

        List<Map.Entry<String,Double>> list = new ArrayList<Map.Entry<String,Double>>(rates.entrySet());
        Collections.sort(list, new Comparator<Map.Entry<String, Double>>() {
            public int compare(Map.Entry<String, Double> r1,Map.Entry<String, Double> r2) {
                return (int)(r1.getValue() - r2.getValue() > 0 ? 1 : -1);
            }
        });
        // 获取key
        String oppAlg = list.get(0).getKey();
        if(oppAlg.equals(Constant.ALG_UC)){
            // 每uv消耗补救
            selectInfoDetail.condi =
                    new ArrayList<>(ranks.subList(0, Math.min(ranks.size(), 5)));
            // 挑选消耗最高活动
            selectInfoDetail.index = ranks.get(0).getIndex();
            //最大转化分
            selectInfoDetail.maxChangeScore = maxChangeScore;
            selectInfoDetail.strategyType = 5;
        }else if(oppAlg.equals(Constant.ALG_UCK)){
            List<SelectPoolInfo> sinfos = new ArrayList<>();
            // 每uv券点击补救
            List<ActivityRankInfo> aris = getSingleOptCandis(sinfos, actDataCopy, actModelCopy, Constant.ALG_UCK,100,100);
            if(sinfos.size() > 0){
                List<SelectPoolInfo> uck_ranks = sinfos.stream()
                        .sorted(Comparator.comparing(SelectPoolInfo::getReward).reversed())
                        .collect(Collectors.toList());
                selectInfoDetail.condi = new ArrayList<>(uck_ranks.subList(0, Math.min(uck_ranks.size(), 5)));
                selectInfoDetail.index = uck_ranks.get(0).getIndex();
                selectInfoDetail.maxChangeScore = maxChangeScore;
                selectInfoDetail.strategyType = 6;
            }
        }else if(oppAlg.equals(Constant.ALG_CVR)){
            List<SelectPoolInfo> sinfos = new ArrayList<>();
            // cvr补救
            List<ActivityRankInfo> aris = getSingleOptCandis(sinfos, actDataCopy, actModelCopy, Constant.ALG_CVR,500,500);
            if(sinfos.size() > 0) {
                List<SelectPoolInfo> cvr_ranks = sinfos.stream()
                        .sorted(Comparator.comparing(SelectPoolInfo::getReward).reversed())
                        .collect(Collectors.toList());
                selectInfoDetail.condi = new ArrayList<>(cvr_ranks.subList(0, Math.min(cvr_ranks.size(), 5)));
                selectInfoDetail.index = cvr_ranks.get(0).getIndex();
                selectInfoDetail.maxChangeScore = maxChangeScore;
                selectInfoDetail.strategyType = 7;
            }
        }else if(oppAlg.equals(Constant.ALG_UR)){
            List<SelectPoolInfo> sinfos = new ArrayList<>();
            // 每uv券点击*cvr补救
            List<ActivityRankInfo> aris = getSingleOptCandis(sinfos, actDataCopy, actModelCopy, Constant.ALG_UR,500,500);
            if(sinfos.size() > 0) {
                List<SelectPoolInfo> ur_ranks = sinfos.stream()
                        .sorted(Comparator.comparing(SelectPoolInfo::getReward).reversed())
                        .collect(Collectors.toList());
                selectInfoDetail.condi = new ArrayList<>(ur_ranks.subList(0, Math.min(ur_ranks.size(), 5)));
                selectInfoDetail.index = ur_ranks.get(0).getIndex();
                selectInfoDetail.maxChangeScore = maxChangeScore;
                selectInfoDetail.strategyType = 8;
            }
        }
        return selectInfoDetail;
    }

    /**
     *功能描述 封装海选数据
     * @author lijingzhe
     * @date 2019/8/23
     * @param act, actIds, exHashMap
     * @return cn.com.duiba.nezha.alg.common.model.activityrecommend.ActivityMatchInfo
     */
    private static ActivityMatchInfo fillData(ActivityInfoData act, Set<Long> actIds, Map<Long, NormInfo> exHashMap){
        ActivityMatchInfo actM=new ActivityMatchInfo();
        actM.activityId=act.activityId;
        actM.slotId=act.slotId;
        actM.appId=act.appId;

        actM.hisClick=new ActivityVal();
        actM.hisClick.globalVal=act.click.globalVal;
        actM.hisClick.appVal=act.click.appVal;
        actM.hisClick.slotVal=act.click.slotVal;

        actM.hisCost=new ActivityVal();
        actM.hisCost.globalVal=act.cost.globalVal;
        actM.hisCost.appVal=act.cost.appVal;
        actM.hisCost.slotVal=act.cost.slotVal;

        actM.hisRequest=new ActivityVal();
        actM.hisRequest.globalVal=act.request.globalVal;
        actM.hisRequest.appVal=act.request.appVal;
        actM.hisRequest.slotVal=act.request.slotVal;

        actM.hisEffect=new ActivityVal();
        actM.hisEffect.globalVal=act.effect.globalVal;
        actM.hisEffect.appVal=act.effect.appVal;
        actM.hisEffect.slotVal=act.effect.slotVal;

        actM.subType=act.subType;
        actM.isExcellent=actIds.contains(act.activityId);
        actM.nPvClick=exHashMap.getOrDefault(act.activityId,new NormInfo()).getnPvClick();
        actM.nPvCost=exHashMap.getOrDefault(act.activityId, new NormInfo()).getnPvCost();
        actM.nCvrForm=exHashMap.getOrDefault(act.activityId, new NormInfo()).getnFormCvr();
        actM.nCvrNotForm=exHashMap.getOrDefault(act.activityId, new NormInfo()).getnNotFormCvr();
        actM.nCvr=exHashMap.getOrDefault(act.activityId, new NormInfo()).getnCvr();
        actM.score=exHashMap.getOrDefault(act.activityId, new NormInfo()).getScore();

        return actM;
    }
}

