package cn.com.duiba.nezha.alg.feature.parse;

import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import cn.com.duiba.nezha.alg.feature.vo.DPAActFeatureDo;
import org.slf4j.LoggerFactory;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

/**
 * @author lijingzhe
 * @description DPA活动解析
 * @date 2020/6/17
 */
public class DPAActFeatureParse {
    public static int F_MAX_SIZE = 64;
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(DPAActFeatureParse.class);

    public static Integer[] HUABEI_REGION = {80, 11, 12, 13, 14, 15};
    public static Integer[] DONGBEI_REGION = {81, 21, 22, 23};
    public static Integer[] HUADONG_REGION = {82, 31, 32, 33, 34, 35, 36, 37};
    public static Integer[] HUAZHONG_REGION = {83, 41, 42, 43};
    public static Integer[] HUANAN_REGION = {84, 44, 45, 46};
    public static Integer[] XINAN_REGION = {85, 50, 51, 52, 53, 54};
    public static Integer[] XIBEI_REGION = {86, 61, 62, 63, 64, 65};
    public static Integer[] HMT_REGION = {87, 71, 8100, 8200};


    public static Integer[] FIRST_TIER_CITY = {11, 12, 31, 50, 2101, 3201, 3202, 3205, 3301, 3302, 3702, 4101, 4201, 4301, 4401, 4403, 4419, 5101, 6101};
    public static Integer[] SECOND_TIER_CITY = {1301, 1306, 1401, 2102, 2201, 2301, 3203, 3204, 3206, 3303, 3304, 3306, 3307, 3310, 3401, 3501, 3502, 3505, 3601, 3701, 3706, 3707, 4406, 4413, 4420, 4501, 5201, 5301, 6201, 6501};
    public static Integer[] THIRD_TIER_CITY = {1309,1303,1502,2202,3207,3209,3305,3412,3509,3611,3708,3703,4114,4110,4210,4307,4306,4407,4408,4601,5107,6104,1304,1302,1501,2306,3212,3210,3403,3407,3503,3607,3609,3713,4103,4107,4108,4206,4304,4302,4452,4412,4503,4602,5203,6301,1310,1305,2103,3208,3213,3211,3411,3402,3506,3604,3717,3710,4113,4115,4211,4205,4303,4451,4405,4404,4502,5106,5303,6401};



    public static Map<String, String> generateFeatureMapStatic(DPAActFeatureDo dpa) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (dpa != null) {
                retMap.put("f201001",DataUtil.Long2String(dpa.getAppId()));
                retMap.put("f108002",DataUtil.Long2String(dpa.getSlotId()));
                retMap.put("f501001", dpa.getUa());
                retMap.put("f505001", dpa.getPriceSection());
                retMap.put("f4010030",DataUtil.Long2String(dpa.getAreaCode()));

                /**
                 * 地域解析
                 */
                Long regionLevel = getRegionLevel(dpa.getAreaCode());
                retMap.put("f4010031", DataUtil.Long2String(regionLevel)); // 地区级别（华北/东北/华东/华中/华南/西南/西北/港澳）
                Long tierCityLevel = getTierCityLevel(dpa.getAreaCode());
                retMap.put("f4010032", DataUtil.Long2String(tierCityLevel)); // 城市级别（一线/二线/三线）


                Integer hour= LocalDateUtil.getHour(LocalDateTime.now());
                retMap.put("f502001", DataUtil.Integer2String(hour)); // 竞价请求创建时间-小时

                Integer weekDay = LocalDateUtil.getDayOfWeek(LocalDateTime.now());
                retMap.put("f502002", DataUtil.Integer2String(weekDay)); // 竞价请求创建时间-星期几
            }

        } catch (Exception e) {
            logger.error("DPAActFeatureParse.generateFeatureMapStatic error:" + e);
        }

        return retMap;
    }


    public static Map<String, String> generateFeatureMapDynamicAct(DPAActFeatureDo dpa) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (dpa != null) {
                retMap.put("f301001", DataUtil.Long2String(dpa.getPackageIdDo().getActivityId()));
            }
        }catch (Exception e) {
            logger.error("DPAActFeatureParse generateFeatureMapDynamicTitle error:", e);

        }
        return retMap;
    }

    public static Map<String, String> generateFeatureMapDynamicPackage(DPAActFeatureDo dpa) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (dpa != null) {
                retMap.put("f331001", DataUtil.Long2String(dpa.getPackageIdDo().getSkinId()));
                retMap.put("f331002", DataUtil.Long2String(dpa.getPackageIdDo().getTitleId()));
                retMap.put("f331003", DataUtil.Long2String(dpa.getPackageIdDo().getSubTitleId()));
                retMap.put("f331004", DataUtil.Long2String(dpa.getPackageIdDo().getPrizeId()));
                retMap.put("f331005", DataUtil.Long2String(dpa.getPackageIdDo().getSkinTagId()));
//                retMap.put("f331006", dpa.getPackageIdDo().getTitleTagId());
//                retMap.put("f331007", dpa.getPackageIdDo().getSubTitleTagId());
                retMap.put("f331008", DataUtil.Long2String(dpa.getPackageIdDo().getPrizeTagId()));
            }
        }catch (Exception e) {
            logger.error("DPAActFeatureParse generateFeatureMapDynamicPrize error:", e);

        }
        return retMap;
    }

    //根据areaCode解析地区级别（华北/东北/华东/华中/华南/西南/西北/港澳） f4010031
    public static Long getRegionLevel(Long areaCode) {

        Long ret = null;

        if (AssertUtil.isNotEmpty(areaCode)) {

            int area = DataUtil.toInt(areaCode);
            if ( areaCode >= 1000L && areaCode != 8100L && areaCode != 8200L) {
                area =  DataUtil.toInt(Long.valueOf(areaCode / 100));
            }

            if (Arrays.asList(HUABEI_REGION).contains(area)) { ret = 80L; }
            else if (Arrays.asList(DONGBEI_REGION).contains(area)) { ret = 81L; }
            else if (Arrays.asList(HUADONG_REGION).contains(area)) { ret = 82L; }
            else if (Arrays.asList(HUAZHONG_REGION).contains(area)) { ret = 83L; }
            else if (Arrays.asList(HUANAN_REGION).contains(area)) { ret = 84L; }
            else if (Arrays.asList(XINAN_REGION).contains(area)) { ret = 85L; }
            else if (Arrays.asList(XIBEI_REGION).contains(area)) { ret = 86L; }
            else if (Arrays.asList(HMT_REGION).contains(area)) { ret = 87L; }

        }

        return ret;
    }



    //根据areaCode解析城市级别（一线/二线/三线） f4010032
    public static Long getTierCityLevel(Long areaCode) {

        Long ret = null;

        if (AssertUtil.isNotEmpty(areaCode)) {

            int area = DataUtil.toInt(areaCode);

            if (Arrays.asList(FIRST_TIER_CITY).contains(area)) { ret = 1L; }
            else if (Arrays.asList(SECOND_TIER_CITY).contains(area)) { ret = 2L; }
            else if (Arrays.asList(THIRD_TIER_CITY).contains(area)) { ret = 3L; }

        }

        return ret;
    }
}

