package cn.com.duiba.nezha.alg.feature.parse;

import cn.com.duiba.nezha.alg.common.enums.DateStyle;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import cn.com.duiba.nezha.alg.feature.vo.*;
import com.alibaba.fastjson.JSON;
import org.slf4j.LoggerFactory;


import java.time.LocalDateTime;
import java.util.*;

/**
 * Created by pc on 2019/2/21.
 */
public class AdxFeatureParse {

    public static int F_MAX_SIZE = 64;
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(AdxFeatureParse.class);

    public static Integer[] HUABEI_REGION = {80, 11, 12, 13, 14, 15};
    public static Integer[] DONGBEI_REGION = {81, 21, 22, 23};
    public static Integer[] HUADONG_REGION = {82, 31, 32, 33, 34, 35, 36, 37};
    public static Integer[] HUAZHONG_REGION = {83, 41, 42, 43};
    public static Integer[] HUANAN_REGION = {84, 44, 45, 46};
    public static Integer[] XINAN_REGION = {85, 50, 51, 52, 53, 54};
    public static Integer[] XIBEI_REGION = {86, 61, 62, 63, 64, 65};
    public static Integer[] HMT_REGION = {87, 71, 8100, 8200};

    public static Integer[] FIRST_TIER_CITY = {11, 12, 31, 50, 2101, 3201, 3202, 3205, 3301, 3302, 3702, 4101, 4201, 4301, 4401, 4403, 4419, 5101, 6101};
    public static Integer[] SECOND_TIER_CITY = {1301, 1306, 1401, 2102, 2201, 2301, 3203, 3204, 3206, 3303, 3304, 3306, 3307, 3310, 3401, 3501, 3502, 3505, 3601, 3701, 3706, 3707, 4406, 4413, 4420, 4501, 5201, 5301, 6201, 6501};
    public static Integer[] THIRD_TIER_CITY = {1309, 1303, 1502, 2202, 3207, 3209, 3305, 3412, 3509, 3611, 3708, 3703, 4114, 4110, 4210, 4307, 4306, 4407, 4408, 4601, 5107, 6104, 1304, 1302, 1501, 2306, 3212, 3210, 3403, 3407, 3503, 3607, 3609, 3713, 4103, 4107, 4108, 4206, 4304, 4302, 4452, 4412, 4503, 4602, 5203, 6301, 1310, 1305, 2103, 3208, 3213, 3211, 3411, 3402, 3506, 3604, 3717, 3710, 4113, 4115, 4211, 4205, 4303, 4451, 4405, 4404, 4502, 5106, 5303, 6401};

    public static int[] cntBucket = {0, 1, 3, 5, 10, 15, 20};
    private static double[] CTR_BUCKET = {0.001, 0.002, 0.003, 0.005, 0.007, 0.01, 0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.10, 0.15, 0.20, 0.30, 0.40, 0.50, 0.60, 0.70, 0.80, 0.90, 1.0}; // ctr区间划分
    private static double[] CVA_BUCKET = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 25, 30, 35, 40, 45, 50, 60, 70, 80, 90, 100}; // click-value区间划分

    // 视频频道编码videoChannel
    public static Map<String, Integer> videoChannelCodes = new HashMap<String, Integer>(){
        {
            put("aiqiyi--1", 0);
            put("aiqiyi-0", 1); put("aiqiyi-1", 2); put("aiqiyi-2", 3); put("aiqiyi-3", 4); put("aiqiyi-4", 5);
            put("aiqiyi-5", 6); put("aiqiyi-6", 7); put("aiqiyi-7", 8); put("aiqiyi-8", 9); put("aiqiyi-9", 10);
            put("aiqiyi-10", 11); put("aiqiyi-11", 12); put("aiqiyi-12", 13); put("aiqiyi-13", 14); put("aiqiyi-14", 15);
            put("aiqiyi-15", 16); put("aiqiyi-16", 17); put("aiqiyi-17", 18); put("aiqiyi-18", 19); put("aiqiyi-19", 20);
            put("aiqiyi-20", 21); put("aiqiyi-21", 22); put("aiqiyi-22", 23); put("aiqiyi-23", 24); put("aiqiyi-24", 25);
            put("aiqiyi-25", 26); put("aiqiyi-26", 27); put("aiqiyi-27", 28); put("aiqiyi-28", 29); put("aiqiyi-29", 30);
            put("aiqiyi-30", 31); put("aiqiyi-31", 32); put("aiqiyi-32", 33); put("aiqiyi-33", 34); put("aiqiyi-34", 35);
            put("aiqiyi-89", 36); put("aiqiyi-91", 37); put("aiqiyi-92", 38); put("aiqiyi-96", 39); put("aiqiyi-97", 40);
            put("aiqiyi-99", 41); put("aiqiyi-8198", 42); put("aiqiyi-30001", 43); put("aiqiyi-30002", 44); put("aiqiyi-30003", 45);
            put("aiqiyi-30004", 46); put("aiqiyi-30005", 47); put("aiqiyi-30006", 48); put("aiqiyi-30007", 49); put("aiqiyi-30008", 50);
            put("aiqiyi-30009", 51); put("aiqiyi-30010", 52); put("aiqiyi-30011", 53); put("aiqiyi-30012", 54); put("aiqiyi-30013", 55);
            put("aiqiyi-30014", 56); put("aiqiyi-30015", 57); put("aiqiyi-30016", 58); put("aiqiyi-30017", 59); put("aiqiyi-30018", 60);
            put("aiqiyi-30019", 61); put("aiqiyi-30020", 62); put("aiqiyi-30021", 63); put("aiqiyi-30022", 64); put("aiqiyi-30030", 65);
            put("aiqiyi-40000", 66); put("aiqiyi-40001", 67); put("aiqiyi-40002", 68); put("aiqiyi-40004", 69); put("aiqiyi-49999", 70);
            put("aiqiyi-50000", 71); put("aiqiyi-50001", 72); put("aiqiyi-70001", 73); put("aiqiyi-70002", 74); put("aiqiyi-70003", 75);

            put("mango-0", 100); put("mango-1", 101); put("mango-2", 102); put("mango-3", 103); put("mango-20", 104);
            put("mango-50", 105); put("mango-51", 106); put("mango-99", 107); put("mango-104", 108); put("mango-105", 109);
            put("mango-106", 110); put("mango-111", 111); put("mango-115", 112); put("mango-116", 113); put("mango-117", 114);
            put("mango-118", 115); put("mango-120", 116);

        }
    };


    public static Map<String, String> generateFeatureMapStatic(AdxFeatureDo cf) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (cf != null) {

                cf.setCurrentCreatTime(getStringToTimeStamp());

                //推广组
                retMap.put("f1010010", DataUtil.Long2String(cf.getGroupId())); // 推广组ID
                retMap.put("f1010020", DataUtil.Integer2String(cf.getAdxType())); // 平台类型
                retMap.put("f1010030", cf.getCat()); // 推广组对应类别
                retMap.put("f1010031", cf.getTuiaCat()); // 推啊平台的推广组对应类别(标准)
                retMap.put("f1010040", cf.getSectionCat()); // 推广组当前部分的类别
                retMap.put("f1010041", cf.getTuiaSectionCat()); // 推啊平台的推广组当前部分的类别(标准)
                retMap.put("f1010050", cf.getKeywords()); // 推广组关键词


                //资源位
                retMap.put("f2010010", DataUtil.Long2String(cf.getResourceId())); // 资源位ID
                retMap.put("f2010020", DataUtil.Integer2String(cf.getAdvertType())); // 广告类型
                retMap.put("f2010030", DataUtil.Integer2String(cf.getPosType())); // 广告位类型
                retMap.put("f2010040", DataUtil.Integer2String(cf.getPosStyle())); // 广告位样式
                retMap.put("f2010050", cf.getSpecCode()); // 广告类型编码


                //PD特征
                retMap.put("f3020000", DataUtil.Integer2String(cf.getPmpType())); // 流量类型pmpType：0-RTB 1-PD 2-PDB
                retMap.put("f3020010", DataUtil.Long2String(cf.getDealId())); // PD deal ID
                retMap.put("f3020020", DataUtil.Long2String(cf.getPrice())); // PD ldeal出价


                //上下文
                retMap.put("f4010010", cf.getUa()); // ua
                retMap.put("f4010020", cf.getIp()); // ip
                retMap.put("f4010030", DataUtil.Long2String(cf.getAreaCode())); // 地域

                Long regionLevel = getRegionLevel(cf.getAreaCode());
                retMap.put("f4010031", DataUtil.Long2String(regionLevel)); // 地区级别（华北/东北/华东/华中/华南/西南/西北/港澳）
                Long tierCityLevel = getTierCityLevel(cf.getAreaCode());
                retMap.put("f4010032", DataUtil.Long2String(tierCityLevel)); // 城市级别（一线/二线/三线）

                retMap.put("f4010040", DataUtil.Integer2String(cf.getDeviceType())); // 设备类型
                retMap.put("f4010041", DataUtil.Integer2String(cf.getTuiaDeviceType())); // 推啊平台的设备类型(标准)
                retMap.put("f4010050", cf.getMake()); // 设备厂商
                retMap.put("f4010060", cf.getModel()); // 设备型号
                retMap.put("f4010070", cf.getOs()); // 操作系统
                retMap.put("f4010080", cf.getCarrier()); // 运营商
                retMap.put("f4010090", DataUtil.Integer2String(cf.getConnectionType())); // 联网方式
                retMap.put("f4010091", DataUtil.Integer2String(cf.getTuiaConnectionType())); // 推啊平台的联网方式(标准)
                retMap.put("f4010100", getFloat2String(cf.getLat())); // 纬度
                retMap.put("f4010110", getFloat2String(cf.getLon())); // 经度

                Integer hour = getTimeStampToHour(cf.getCurrentCreatTime());
                retMap.put("f4010121", DataUtil.Integer2String(hour)); // 竞价请求创建时间-小时
                Integer weekDay = getTimeStampToWeekDay(cf.getCurrentCreatTime());
                retMap.put("f4010122", DataUtil.Integer2String(weekDay)); // 竞价请求创建时间-星期几


                //用户
                //retMap.put("f5010010", cf.getGender()); // 性别
                retMap.put("f5010020", cf.getInterestKeyWords()); // 用户兴趣
                //retMap.put("f5010030", cf.getYob()); // 出生年月
                retMap.put("f5010050", cf.getImei()); // 设备号


                //上下文-新增
                retMap.put("f6010010", cf.getAppName()); // 应用名称
                retMap.put("f6010020", cf.getAppBundle()); // 包名
                retMap.put("f6010030", cf.getAppPageCat()); // 应用当前页类型
                retMap.put("f6010040", DataUtil.Integer2String(cf.getAppPaid())); // 0-免费 1-付费
                retMap.put("f6010050", cf.getGeoCountry()); // 设备位置信息-国家
                retMap.put("f6010060", cf.getGeoRegion()); // 设备位置信息-地区
                retMap.put("f6010070", cf.getGeoCity()); // 设备位置信息-城市
                retMap.put("f6010080", DataUtil.Integer2String(cf.getGeoType())); // 设备位置信息-定位类型
                retMap.put("f6010090", cf.getTagId()); // adx资源位，竞价广告位ID（adx方）
                retMap.put("f6010100", DataUtil.Integer2String(cf.getTagIndex())); // adx资源位，竞价广告位顺序索引（adx方）
                retMap.put("f6010110", cf.getAppIndustryId());//app所属行业id


                if (cf.getStyleStandard() != null) {
                    retMap.put("f3010130", DataUtil.Integer2String(cf.getStyleStandard())); // 广告样式规格
                }


                // adx特定数据特征
                AiQiYiFeature aiQiYiFeature = Optional.ofNullable(cf.getAiQiYiFeature()).orElse(new AiQiYiFeature());
                MgTvFeature mgTvFeature = Optional.ofNullable(cf.getMgTvFeature()).orElse(new MgTvFeature());
                BaiDuFeature baiDuFeature = Optional.ofNullable(cf.getBaiDuFeature()).orElse(new BaiDuFeature());

                retMap.put("f6020010", FeatureBaseParse.getAnyValue(cf.getAdxType(),
                        aiQiYiFeature.getTitle(), mgTvFeature.getVideoName(), baiDuFeature.getTitle())); //videoTitle 视频标题名称
                retMap.put("f6020020", FeatureBaseParse.getAnyValue(cf.getAdxType(),
                        aiQiYiFeature.getUrl(), mgTvFeature.getVideoUrl(), baiDuFeature.getUrl())); // videoUrl视频网址
                retMap.put("f6020030", FeatureBaseParse.getAnyValue(cf.getAdxType(),
                        aiQiYiFeature.getLen(), mgTvFeature.getDuration(), baiDuFeature.getContentLength())); // videoDuration视频时长，单位秒
                retMap.put("f6020040", FeatureBaseParse.getCrossAnyValue(cf.getAdxType(), cf.getGroupId(),
                        aiQiYiFeature.getChannelId(), mgTvFeature.getChannelId(), baiDuFeature.getChannelId())); // channelId频道信息id（推广组id&频道信息id）
                retMap.put("f6020050", FeatureBaseParse.getCrossAnyValue(cf.getAdxType(), cf.getGroupId(),
                        aiQiYiFeature.getAdTypeId(), mgTvFeature.getLocation(), baiDuFeature.getAdSlotType())); // adTypeId广告类型，创意类型id（推广组id&广告类型id）

                retMap.put("f6020060", FeatureBaseParse.getCrossAnyValue(cf.getAdxType(), cf.getGroupId(),
                        aiQiYiFeature.getCountry(), null, null)); // country国家id
                retMap.put("f6020070", FeatureBaseParse.getCrossAnyValuePlus(cf.getAdxType(), cf.getGroupId(),
                        aiQiYiFeature.getMetro(), null, baiDuFeature.getProvince())); // province省份id
                retMap.put("f6020080", FeatureBaseParse.getCrossAnyValuePlus(cf.getAdxType(), cf.getGroupId(),
                        aiQiYiFeature.getCity(), mgTvFeature.getCityCode(), baiDuFeature.getCity())); // city城市id
                retMap.put("f6020090", FeatureBaseParse.getCrossAnyValuePlus(cf.getAdxType(), cf.getGroupId(),
                        null, null, baiDuFeature.getDistrict())); // district地区id
                retMap.put("f6020100", FeatureBaseParse.getCrossAnyValuePlus(cf.getAdxType(), cf.getGroupId(),
                        null, null, baiDuFeature.getStreet())); // street街道id

                //爱奇艺特定数据
                retMap.put("f6030010", DataUtil.Integer2String(aiQiYiFeature.getAlbumId())); // albumId剧目信息
                retMap.put("f6030020", DataUtil.Integer2String(aiQiYiFeature.getPlatformId())); // platformId投放平台信息id

                //芒果特定数据
                retMap.put("f6040010", DataUtil.Integer2String(mgTvFeature.getVideoId())); // videoId视频id
                retMap.put("f6040020", DataUtil.Integer2String(mgTvFeature.getCollectionId())); // collectionId视频所述合集id
                retMap.put("f6040030", mgTvFeature.getCollectionName()); // collectionName视频所述合集名称
                retMap.put("f6040040", mgTvFeature.getItemIds()); // itemIds视频二级分类id
                retMap.put("f6040050", mgTvFeature.getItemNames()); // itemNames视频二级分类名称
                retMap.put("f6040060", DataUtil.Integer2String(mgTvFeature.getAreaId())); // videoAreaId视频所属地区id
                retMap.put("f6040070", mgTvFeature.getAreaName()); // videoAreaName视频所属地区名称
                retMap.put("f6040080", DataUtil.Integer2String(mgTvFeature.getYear())); // videoYear视频出品年代
                retMap.put("f6040090", DataUtil.Integer2String(mgTvFeature.getVideoType())); // videoType片源，1-正片，2-短片

                //百度联盟特定数据
                retMap.put("f6050010", DataUtil.Integer2String(baiDuFeature.getPageType())); // pageType页面类型
                retMap.put("f6050020", baiDuFeature.getUserTagIdList()); // 百度用户意图词信息

                //资源位+分媒体行为特征
                String resoAppId = FeatureBaseParse.getIdCrossValue(cf.getResourceId(), cf.getAppName());
                retMap.put("f6060010", resoAppId); //创意Id&appId
                Long resoAppExpCntDay = cf.getResoAppExpCntDay();
                retMap.put("f6060011", DataUtil.Long2String(resoAppExpCntDay)); //资源位+百度appId 近1h曝光次数
                Long resoAppClickCntDay = cf.getResoAppClickCntDay();
                retMap.put("f6060012", DataUtil.Long2String(resoAppClickCntDay)); //资源位+百度appId 近1h点击次数
                Long resoAppAdCostDay = cf.getResoAppAdCostDay();
                retMap.put("f6060013", DataUtil.Long2String(resoAppAdCostDay)); //资源位+百度appId 近1h广告消耗
                Double resoAppCtr = FeatureBaseParse.getWilsonScore(resoAppClickCntDay, resoAppExpCntDay);
                Long resoAppCtrLevel = FeatureBaseParse.bucket(resoAppCtr, CTR_BUCKET);
                retMap.put("f6060014", DataUtil.Long2String(resoAppCtrLevel)); //资源位+百度appId  近1h ctr分桶值
                Double resoAppCva = DataUtil.division(resoAppAdCostDay, resoAppClickCntDay);
                Long resoAppCvaLevel = FeatureBaseParse.bucket(resoAppCva, CVA_BUCKET);
                retMap.put("f6060015", DataUtil.Long2String(resoAppCvaLevel)); //资源位+百度appId 近1h 点击价值分桶值


                // dmp数据
                DmpFeature dmpFeature = Optional.ofNullable(cf.getDmpFeature()).orElse(new DmpFeature());

                // dmp数据--adx用户画像
                retMap.put("f5010010", FeatureBaseParse.getDmpSexData(dmpFeature.getSex(), cf.getGender())); //性别
                retMap.put("f5010030", FeatureBaseParse.getDmpData(dmpFeature.getAge(), cf.getYob())); //年龄
                retMap.put("f5010031", dmpFeature.getMarry()); //婚恋

                // dmp数据--adx用户当天行为画像
                Long imeiDayBidTimes = getImeiDayTimes(cf.getImei(), 50L, cf.getImeiDayBidTimes());
                Long imeiDayBid = FeatureBaseParse.getCntLevel(dmpFeature.getImeiDayBidTimes(), 50L);
                retMap.put("f5010060", DataUtil.Long2String(FeatureBaseParse.getDmpData(imeiDayBid, imeiDayBidTimes))); // imei 当天竞价次数


                Long imeiDayExpTimes = getImeiDayTimes(cf.getImei(), 7L, cf.getImeiDayExpTimes());
                Long imeiDayExp = FeatureBaseParse.getCntLevel(dmpFeature.getImeiDayExpTimes(), 7L);
                retMap.put("f5010070", DataUtil.Long2String(FeatureBaseParse.getDmpData(imeiDayExp, imeiDayExpTimes))); // imei 当天曝光次数

                Long resoDayBidTimes = getResoDayTimes(cf.getResourceId(), cf.getImei(), 50L, cf.getResoDayBidTimes());
                Long resoDayBid = FeatureBaseParse.getCntLevel(dmpFeature.getResoDayBidTimes(), 50L);
                retMap.put("f5010080", DataUtil.Long2String(FeatureBaseParse.getDmpData(resoDayBid, resoDayBidTimes))); // imei+资源位 当天竞价次数

                Long resoDayExpTimes = getResoDayTimes(cf.getResourceId(), cf.getImei(), 7L, cf.getResoDayExpTimes());
                Long resoDayExp = FeatureBaseParse.getCntLevel(dmpFeature.getResoDayExpTimes(), 7L);
                retMap.put("f5010090", DataUtil.Long2String(FeatureBaseParse.getDmpData(resoDayExp, resoDayExpTimes))); // imei+资源位 当天曝光次数

                retMap.put("f5010100", cf.getNewSex()); // (爱奇艺)性别标签

                // dmp数据--adx用户历史行为画像
                retMap.put("f5020010", FeatureBaseParse.getLTimesLevel(dmpFeature.getHistoryGroBid())); // imei+推广组 历史7天 竞价次数
                retMap.put("f5020020", FeatureBaseParse.getMTimesLevel(dmpFeature.getHistoryGroExp())); // imei+推广组 历史7天 曝光次数
                retMap.put("f5020030", FeatureBaseParse.getSTimesLevel(dmpFeature.getHistoryGroClick())); // imei+推广组 历史7天 点击次数

                retMap.put("f5020040", FeatureBaseParse.getLTimesLevel(dmpFeature.getHistoryResoBid())); // imei+资源位 历史7天 竞价次数
                retMap.put("f5020050", FeatureBaseParse.getMTimesLevel(dmpFeature.getHistoryResoExp())); // imei+资源位 历史7天 曝光次数
                retMap.put("f5020060", FeatureBaseParse.getSTimesLevel(dmpFeature.getHistoryResoClick())); // imei+资源位 历史7天 点击次数

                retMap.put("f5020070", FeatureBaseParse.getLTimesLevel(dmpFeature.getHistoryImeiBid())); // imei 历史7天 竞价次数
                retMap.put("f5020080", FeatureBaseParse.getMTimesLevel(dmpFeature.getHistoryImeiExp())); // imei 历史7天 曝光次数
                retMap.put("f5020090", FeatureBaseParse.getSTimesLevel(dmpFeature.getHistoryImeiClick())); // imei 历史7天 点击次数

                //adx用户历史交叉行为
                retMap.put("f5020101", FeatureBaseParse.getCrossLevel(dmpFeature.getHistoryImeiBid(), dmpFeature.getHistoryImeiExp())); // imei 历史7天 竞价&曝光 交叉
                retMap.put("f5020102", FeatureBaseParse.getCrossLevel(dmpFeature.getHistoryImeiBid(), dmpFeature.getHistoryImeiClick())); // imei 历史7天 竞价&点击 交叉
                retMap.put("f5020103", FeatureBaseParse.getCrossLevel(dmpFeature.getHistoryImeiExp(), dmpFeature.getHistoryImeiClick())); // imei 历史7天 曝光&点击 交叉

                retMap.put("f5020104", FeatureBaseParse.getExpClickCrossValue(dmpFeature.getHistoryGroClick(), dmpFeature.getHistoryGroExp())); // imei+推广组 历史7天 曝光&点击 交叉值
                retMap.put("f5020105", FeatureBaseParse.getExpClickCrossValue(dmpFeature.getHistoryResoClick(), dmpFeature.getHistoryResoExp())); // imei+资源位 历史7天 曝光&点击 交叉值
                retMap.put("f5020106", FeatureBaseParse.getExpClickCrossValue(dmpFeature.getHistoryImeiClick(), dmpFeature.getHistoryImeiExp())); // imei 历史7天 曝光&点击 交叉值

                //推广组和历史交叉行为组合
                retMap.put("f5030101", FeatureBaseParse.getComCrossValue(cf.getGroupId(), dmpFeature.getHistoryImeiClick(), dmpFeature.getHistoryImeiExp())); // 推广组id +&+ imei历史7天曝光&点击交叉值
                retMap.put("f5030102", FeatureBaseParse.getComCrossValue(cf.getResourceId(), dmpFeature.getHistoryImeiClick(), dmpFeature.getHistoryImeiExp())); // 资源位id +&+ imei历史7天曝光&点击交叉值
                retMap.put("f5030103", FeatureBaseParse.getComCrossValue(cf.getIdeaId(), dmpFeature.getHistoryImeiClick(), dmpFeature.getHistoryImeiExp())); // 创意id +&+ imei历史7天曝光&点击交叉值


                // dmp数据--用户标签列表
                String userBasicTags = FeatureBaseParse.getDmpData(dmpFeature.getUserBasicTags(), cf.getUserTagsList());
                retMap.put("f7010010", userBasicTags);
                retMap.put("f7010011", FeatureBaseParse.getLengthLevel(userBasicTags,50,5)); // 用户标签列表-长度
                retMap.put("f7010013", dmpFeature.getTagCategory()); // 每个类目下用户标签数

                String tagCategoryLevel = FeatureBaseParse.getCategoryLevel(dmpFeature.getTagCategory(),10,1);
                retMap.put("f7010014", tagCategoryLevel); // 每个类目下用户标签数--分桶


                // dmp数据--app安装列表
                String appList = FeatureBaseParse.getDmpData(dmpFeature.getAppList(), cf.getAppList());
                retMap.put("f7020010", appList);
                retMap.put("f7020011", FeatureBaseParse.getLengthLevel(appList,5,100,5)); // app安装列表长度
                retMap.put("f7020013", dmpFeature.getValidAppList()); // 有效app安装列表
                retMap.put("f7020014", dmpFeature.getTopAppList()); // top app安装列表300
                retMap.put("f7020015", dmpFeature.getTopAppList2()); // top app安装列表500
                retMap.put("f7020016", dmpFeature.getTopAppList3()); // top app安装列表1000
                retMap.put("f7020017", dmpFeature.getMidAppList()); // 中长尾app安装列表
                retMap.put("f7020018", dmpFeature.getInstallAppList());  // 近期安装app
                retMap.put("f7020019", dmpFeature.getUninstallAppList()); // 近期卸载app
                retMap.put("f7020020", dmpFeature.getAppCategory()); // 每个类目下的app数
                retMap.put("f7020021", dmpFeature.getTradeAppList()); // 广告行业对应的app
                retMap.put("f7020022", dmpFeature.getAppVector()); // app向量

                String appCategoryLevel = FeatureBaseParse.getCategoryLevel(dmpFeature.getAppCategory(),10,1);
                retMap.put("f7020023", appCategoryLevel); // 每个类目app数--分桶


                // dmp数据--短信签名列表
                String sigList = FeatureBaseParse.getDmpData(dmpFeature.getSigList(), cf.getMessageNameList());
                retMap.put("f7030010", sigList);
                retMap.put("f7030011", FeatureBaseParse.getLengthLevel(sigList, 15,1)); // 短信签名列表长度
                retMap.put("f7030013", dmpFeature.getMsgCategory()); // 每个类目下的短信签名数
                retMap.put("f7030014", dmpFeature.getMsgKeyword()); // 短信签名关键字

                //String msgCategoryLevel = FeatureBaseParse.getCategoryLevel(dmpFeature.getMsgCategory(),10,1);
                //retMap.put("f7030015", msgCategoryLevel); // 每个类目短信签名数--分桶
                retMap.put("f7030016", dmpFeature.getMsgVector()); // 短信签名向量

                //dmp数据--adx用户画像行为
                Map<String, Integer> recentVideoCatePrefMap = DataUtil.stringToMap(dmpFeature.getRecentVideoCatePref(), 0);
                retMap.put("f7040010", FeatureParse.getUserCategoryActLevelByBucket(recentVideoCatePrefMap, FeatureParse.videoCateCodes, cntBucket));//短期视频二级类目偏好（近14天）
                Map<String, Integer> recentVideoChannelPrefMap = DataUtil.stringToMap(dmpFeature.getRecentVideoChannelPref(), 0);
                retMap.put("f7040020", FeatureParse.getUserCategoryActLevelByBucket(recentVideoChannelPrefMap, videoChannelCodes, cntBucket));//短期视频频道偏好（近14天）
                retMap.put("f7040030", dmpFeature.getLastVideoCategory());//上一次观看视频二级分类id
                retMap.put("f7040040", dmpFeature.getLastVideoChannel());//上一次观看视频频道
                retMap.put("f7040050", FeatureParse.bucket(dmpFeature.getWeekActivityRequest(), cntBucket).toString()); //一周内活动访问次数
                retMap.put("f7040060", FeatureParse.bucket(dmpFeature.getDayActivityRequest(), cntBucket).toString()); //当日活动访问次数
                retMap.put("f7040070", dmpFeature.getPeriodActivityRequest()); //历史活动访问频繁时段（凌晨/早上/上午/中午/下午/晚上/深夜）
                retMap.put("f7040080", FeatureParse.bucket(dmpFeature.getWeekActivityJoin(), cntBucket).toString()); //一周内活动参与次数
                retMap.put("f7040090", FeatureParse.bucket(dmpFeature.getDayActivityJoin(), cntBucket).toString()); //当日活动参与次数
                retMap.put("f7040100", dmpFeature.getPeriodActivityJoin()); //历史活动参与频繁时段
                retMap.put("f7040110", dmpFeature.getPeriodLaunchTrade()); //历史发券频繁时段
                retMap.put("f7040120", dmpFeature.getClickLaunchTrade() == null ? null : dmpFeature.getClickLaunchTrade().toString()); //上一次发券是否点击
                retMap.put("f7040130", dmpFeature.getPeriodClickTrade()); //历史券点击频繁时段
                retMap.put("f7040140", dmpFeature.getEffectClickTrade() == null ? null : dmpFeature.getEffectClickTrade().toString()); //上一次券点击是否转化


                //用户+素材行为特征
                retMap.put("f7050010", cf.getExpDayMaterials()); // 用户当天曝光的素材序列（曝光降序）TOP30
                retMap.put("f7050020", cf.getClickDayMaterials()); // 用户当天点击的素材序列（点击降序）TOP30
                retMap.put("f7050030", cf.getExpWeekMaterials()); // 用户近7天曝光的素材序列 (曝光降序) TOP30
                retMap.put("f7050040", cf.getClickWeekMaterials()); // 用户近7天点击的素材序列（点击降序）TOP30

                //用户+资源位+素材行为特征
                retMap.put("f7060010", cf.getResoExpDayMaterials()); // 用户+资源位，当天曝光的素材序列（曝光降序）TOP30
                retMap.put("f7060020", cf.getResoClickDayMaterials()); // 用户+资源位，当天点击的素材序列（点击降序）TOP30


            }

        } catch (Exception e) {

            logger.error("AdxFeatureParse.generateFeatureMapStatic error:" + e);
        }

        return retMap;
    }


    public static Map<String, String> generateFeatureMapDynamic(AdxFeatureDo cf, AdxFeatureDo staticCf) {
        Map<String, String> retMap = new HashMap<>(F_MAX_SIZE);
        try {
            if (cf != null) {

                //创意（素材）
                retMap.put("f3010010", DataUtil.Long2String(cf.getIdeaId())); // 创意ID
                retMap.put("f3010020", cf.getButtonText()); // 按钮文字
                retMap.put("f3010030", DataUtil.Integer2String(cf.getIconH(), "0")); // 创意图标高
                retMap.put("f3010040", DataUtil.Integer2String(cf.getIconW())); // 创意图标宽
                retMap.put("f3010050", DataUtil.Integer2String(cf.getMjPicType())); // 规格类型(适用墨迹)
                retMap.put("f3010060", DataUtil.Integer2String(cf.getIdeaPicH1())); // 创意图片高1
                retMap.put("f3010070", DataUtil.Integer2String(cf.getIdeaPicW1())); // 创意图片宽1
                retMap.put("f3010080", DataUtil.Integer2String(cf.getIdeaPicH2())); // 创意图片高2
                retMap.put("f3010090", DataUtil.Integer2String(cf.getIdeaPicW2())); // 创意图片宽2
                retMap.put("f3010100", DataUtil.Integer2String(cf.getIdeaPicH3())); // 创意图片高3
                retMap.put("f3010110", DataUtil.Integer2String(cf.getIdeaPicW3())); // 创意图片宽3
                retMap.put("f3010120", DataUtil.Integer2String(cf.getIdeaPicNum())); // 创意图片个数
                retMap.put("f3010130", DataUtil.Integer2String(cf.getStyleStandard())); // 广告样式规格
                retMap.put("f3010140", DataUtil.Integer2String(cf.getIdeaType())); // 互动创意类型(0-普通静态创意，1-动态创意)

                Long ideaExpTimes = getIdeaExpTime(cf.getIdeaId(), staticCf.getExpTimeMap());
                retMap.put("f5010040", DataUtil.Long2String(ideaExpTimes)); // 该创意+设备号 当天曝光次数

                retMap.put("f3030000", DataUtil.Integer2String(cf.getPictureType())); // 素材图片类型（1=图片 3=视频）

                // 图片素材标签
                List<AdxIdeaMaterialDo> pictureList = cf.getPictureMaterialDo();
                String picMaterialId = FeatureBaseParse.getMaterialFeature(pictureList, "materialId");
                retMap.put("f3030010", picMaterialId); //素材id
                retMap.put("f3030020", FeatureBaseParse.getMaterialFeature(pictureList, "effectType")); //素材效果分类（动态/静态/视频）
                retMap.put("f3030030", FeatureBaseParse.getMaterialFeature(pictureList, "figureType")); //素材类型（单图/组图）
                retMap.put("f3030040", FeatureBaseParse.getMaterialFeature(pictureList, "customType")); //是否定制
                retMap.put("f3030050", FeatureBaseParse.getMaterialFeature(pictureList, "materialIndustryTag")); //素材行业标签
                retMap.put("f3030060", FeatureBaseParse.getMaterialFeature(pictureList, "slotType")); //广告位性质
                retMap.put("f3030070", FeatureBaseParse.getMaterialFeature(pictureList, "materialStrategy")); //优质素材策略
                retMap.put("f3030080", FeatureBaseParse.getMaterialFeature(pictureList, "prizeElement")); //奖品元素
                retMap.put("f3030090", FeatureBaseParse.getMaterialFeature(pictureList, "designColor")); //设计色调
                retMap.put("f3030100", FeatureBaseParse.getMaterialFeature(pictureList, "designStyle")); //设计风格
                retMap.put("f3030110", FeatureBaseParse.getMaterialFeature(pictureList, "dynamicEffect")); //设计动效
                retMap.put("f3030120", FeatureBaseParse.getMaterialFeature(pictureList, "textElement")); //文案元素
                retMap.put("f3030130", FeatureBaseParse.getMaterialFeature(pictureList, "activityTool")); //活动工具
                retMap.put("f3030140", FeatureBaseParse.getMaterialFeature(pictureList, "season")); //季节标签
                retMap.put("f3030150", FeatureBaseParse.getMaterialFeature(pictureList, "festival")); //节日标签
                retMap.put("f3030160", FeatureBaseParse.getMaterialFeature(pictureList, "timing")); //时间标签（周末/月初/月中/月末）
                retMap.put("f3030170", FeatureBaseParse.getMaterialFeature(pictureList, "scene")); //使用场景
                retMap.put("f3030180", FeatureBaseParse.getMaterialFeature(pictureList, "adxRatioType")); //素材尺寸比例区间ID

                // 视频素材标签
                List<AdxIdeaMaterialDo> videoList = cf.getVideoMaterialDo();
                String videoMaterialId = FeatureBaseParse.getMaterialFeature(videoList, "materialId");
                retMap.put("f3030011", videoMaterialId); //素材id
                retMap.put("f3030021", FeatureBaseParse.getMaterialFeature(videoList, "effectType")); //素材效果分类（动态/静态/视频）
                retMap.put("f3030031", FeatureBaseParse.getMaterialFeature(videoList, "figureType")); //素材类型（单图/组图）
                retMap.put("f3030041", FeatureBaseParse.getMaterialFeature(videoList, "customType")); //是否定制
                retMap.put("f3030051", FeatureBaseParse.getMaterialFeature(videoList, "materialIndustryTag")); //素材行业标签
                retMap.put("f3030061", FeatureBaseParse.getMaterialFeature(videoList, "slotType")); //广告位性质
                retMap.put("f3030071", FeatureBaseParse.getMaterialFeature(videoList, "materialStrategy")); //优质素材策略
                retMap.put("f3030081", FeatureBaseParse.getMaterialFeature(videoList, "prizeElement")); //奖品元素
                retMap.put("f3030091", FeatureBaseParse.getMaterialFeature(videoList, "designColor")); //设计色调
                retMap.put("f3030101", FeatureBaseParse.getMaterialFeature(videoList, "designStyle")); //设计风格
                retMap.put("f3030111", FeatureBaseParse.getMaterialFeature(videoList, "dynamicEffect")); //设计动效
                retMap.put("f3030121", FeatureBaseParse.getMaterialFeature(videoList, "textElement")); //文案元素
                retMap.put("f3030131", FeatureBaseParse.getMaterialFeature(videoList, "activityTool")); //活动工具
                retMap.put("f3030141", FeatureBaseParse.getMaterialFeature(videoList, "season")); //季节标签
                retMap.put("f3030151", FeatureBaseParse.getMaterialFeature(videoList, "festival")); //节日标签
                retMap.put("f3030161", FeatureBaseParse.getMaterialFeature(videoList, "timing")); //时间标签（周末/月初/月中/月末）
                retMap.put("f3030171", FeatureBaseParse.getMaterialFeature(videoList, "scene")); //使用场景
                retMap.put("f3030181", FeatureBaseParse.getMaterialFeature(videoList, "adxRatioType")); //素材尺寸比例区间ID


                //素材行为特征
                retMap.put("f3040010", FeatureBaseParse.getMaterialFeature(pictureList, "maExpCntDay")); //素材id 当天曝光次数
                retMap.put("f3040011", FeatureBaseParse.getMaterialFeature(videoList, "maExpCntDay"));
                retMap.put("f3040012", FeatureBaseParse.getMaterialCntLevel(pictureList, "maExpCntDay")); //素材id 当天曝光次数level
                retMap.put("f3040013", FeatureBaseParse.getMaterialCntLevel(videoList, "maExpCntDay"));

                retMap.put("f3040020", FeatureBaseParse.getMaterialFeature(pictureList, "maClickCntDay")); //素材id 当天点击次数
                retMap.put("f3040021", FeatureBaseParse.getMaterialFeature(videoList, "maClickCntDay"));
                retMap.put("f3040022", FeatureBaseParse.getMaterialCntLevel(pictureList, "maClickCntDay")); //素材id 当天点击次数level
                retMap.put("f3040023", FeatureBaseParse.getMaterialCntLevel(videoList, "maClickCntDay"));

                retMap.put("f3040030", FeatureBaseParse.getMaterialFeature(pictureList, "maExpCnt3Day")); //素材id 近3天曝光次数
                retMap.put("f3040031", FeatureBaseParse.getMaterialFeature(videoList, "maExpCnt3Day"));
                retMap.put("f3040032", FeatureBaseParse.getMaterialCntLevel(pictureList, "maExpCnt3Day")); //素材id 近3天曝光次数level
                retMap.put("f3040033", FeatureBaseParse.getMaterialCntLevel(videoList, "maExpCnt3Day"));

                retMap.put("f3040040", FeatureBaseParse.getMaterialFeature(pictureList, "maClickCnt3Day")); //素材id 近3天点击次数
                retMap.put("f3040041", FeatureBaseParse.getMaterialFeature(videoList, "maClickCnt3Day"));
                retMap.put("f3040042", FeatureBaseParse.getMaterialCntLevel(pictureList, "maClickCnt3Day")); //素材id 近3天点击次数level
                retMap.put("f3040043", FeatureBaseParse.getMaterialCntLevel(videoList, "maClickCnt3Day"));


                //素材+资源位行为特征
                retMap.put("f3050010", FeatureBaseParse.getMaterialFeature(pictureList, "maResoExpCntDay")); //素材id+资源位id 当天曝光次数
                retMap.put("f3050011", FeatureBaseParse.getMaterialFeature(videoList, "maResoExpCntDay"));
                retMap.put("f3050012", FeatureBaseParse.getMaterialCntLevel(pictureList, "maResoExpCntDay")); //素材id+资源位id 当天曝光次数level
                retMap.put("f3050013", FeatureBaseParse.getMaterialCntLevel(videoList, "maResoExpCntDay"));

                retMap.put("f3050020", FeatureBaseParse.getMaterialFeature(pictureList, "maResoClickCntDay")); //素材id+资源位id 当天点击次数
                retMap.put("f3050021", FeatureBaseParse.getMaterialFeature(videoList, "maResoClickCntDay"));
                retMap.put("f3050022", FeatureBaseParse.getMaterialCntLevel(pictureList, "maResoClickCntDay")); //素材id+资源位id 当天点击次数level
                retMap.put("f3050023", FeatureBaseParse.getMaterialCntLevel(videoList, "maResoClickCntDay"));

                retMap.put("f3050030", FeatureBaseParse.getMaterialFeature(pictureList, "maResoExpCnt3Day")); //素材id+资源位id 近3天曝光次数
                retMap.put("f3050031", FeatureBaseParse.getMaterialFeature(videoList, "maResoExpCnt3Day"));
                retMap.put("f3050032", FeatureBaseParse.getMaterialCntLevel(pictureList, "maResoExpCnt3Day")); //素材id+资源位id 近3天曝光次数level
                retMap.put("f3050033", FeatureBaseParse.getMaterialCntLevel(videoList, "maResoExpCnt3Day"));

                retMap.put("f3050040", FeatureBaseParse.getMaterialFeature(pictureList, "maResoClickCnt3Day")); //素材id+资源位id 近3天点击次数
                retMap.put("f3050041", FeatureBaseParse.getMaterialFeature(videoList, "maResoClickCnt3Day"));
                retMap.put("f3050042", FeatureBaseParse.getMaterialCntLevel(pictureList, "maResoClickCnt3Day")); //素材id+资源位id 近3天点击次数level
                retMap.put("f3050043", FeatureBaseParse.getMaterialCntLevel(videoList, "maResoClickCnt3Day"));

                //创意+分媒体行为特征
                String ideaAppId = FeatureBaseParse.getIdCrossValue(cf.getIdeaId(), staticCf.getAppName());
                retMap.put("f3060010", ideaAppId); //创意Id&appId
                Long ideaAppExpCntDay = cf.getIdeaAppExpCntDay();
                retMap.put("f3060011", DataUtil.Long2String(ideaAppExpCntDay)); //创意+百度appId 当天曝光次数
                Long ideaAppClickCntDay = cf.getIdeaAppClickCntDay();
                retMap.put("f3060012", DataUtil.Long2String(ideaAppClickCntDay)); //创意+百度appId 当天点击次数
                Long ideaAppAdCostDay = cf.getIdeaAppAdCostDay();
                retMap.put("f3060013", DataUtil.Long2String(ideaAppAdCostDay)); //创意+百度appId 当天广告消耗
                Double ideaAppCtr = FeatureBaseParse.getWilsonScore(ideaAppClickCntDay, ideaAppExpCntDay);
                Long ideaAppCtrLevel = FeatureBaseParse.bucket(ideaAppCtr, CTR_BUCKET);
                retMap.put("f3060014", DataUtil.Long2String(ideaAppCtrLevel)); //创意+百度appId 当天ctr分桶值
                Double ideaAppCva = DataUtil.division(ideaAppAdCostDay, ideaAppClickCntDay);
                Long ideaAppCvaLevel = FeatureBaseParse.bucket(ideaAppCva, CVA_BUCKET);
                retMap.put("f3060015", DataUtil.Long2String(ideaAppCvaLevel)); //创意+百度appId 当天点击价值分桶值


                //用户+资源位+素材行为特征
                String lastExpMaterialId = DataUtil.Long2String(staticCf.getLastExpMaterialId());
                retMap.put("f7060030", lastExpMaterialId); //用户+资源位，上次曝光的素材id
                String lastExpEqualStatus = getLastEqualStatus(picMaterialId, videoMaterialId, lastExpMaterialId);
                retMap.put("f7060031", lastExpEqualStatus); //用户+资源位，上次曝光的素材id 与当前素材id是否相同
                String lastClickMaterialId = DataUtil.Long2String(staticCf.getLastClickMaterialId());
                retMap.put("f7060040", lastClickMaterialId); //用户+资源位，上次点击的素材id
                String lastClickEqualStatus = getLastEqualStatus(picMaterialId, videoMaterialId, lastClickMaterialId);
                retMap.put("f7060041", lastClickEqualStatus); //用户+资源位，上次点击的素材id 与当前素材id是否相同

                //创意活动特征


            }
        } catch (Exception e) {
            logger.error("AdxFeatureParse.generateFeatureMapDynamic error:", e);

        }
        return retMap;
    }


    public static Integer getTimeStampToHour(Long timeStamp) {
        Integer ret = null;
        if (timeStamp != null) {
            Date date = new Date(timeStamp);

            if (date != null) {
                LocalDateTime localDate = LocalDateUtil.dateToLocal(date);

                if (localDate != null) {
                    ret = LocalDateUtil.getHour(localDate);

                }
            }
        }
        return ret;
    }


    public static Integer getTimeStampToWeekDay(Long timeStamp) {
        Integer ret = null;
        if (timeStamp != null) {
            Date date = new Date(timeStamp);

            if (date != null) {
                LocalDateTime localDate = LocalDateUtil.dateToLocal(date);

                if (localDate != null) {
                    ret = LocalDateUtil.getDayOfWeek(localDate);

                }
            }
        }
        return ret;
    }


    public static Long getStringToTimeStamp() {
        Long ret = null;
        String currentTime = LocalDateUtil.getCurrentLocalDateTime(DateStyle.YYYY_MM_DD_HH_MM_SS.getValue());

        if (currentTime != null) {
            ret = LocalDateUtil.getTimeMillis(currentTime, DateStyle.YYYY_MM_DD_HH_MM_SS.getValue());

        }
        return ret;
    }


    public static String getFloat2String(Float src) {
        String ret = null;
        if (src != null) {
            ret = Float.toString(src);
        }
        return ret;

    }


    public static Long getObject2Long(Object src) {
        Long ret = null;
        if (AssertUtil.isNotEmpty(src)) {
            ret = DataUtil.string2Long(String.valueOf(src));
        }
        return ret;

    }


    //该创意+设备号的当天累计曝光次数 f5010040
    public static Long getIdeaExpTime(Long ideaId, Map<Long, Object> expTimeMap) {

        Long ret = null;

        if (AssertUtil.isAllNotEmpty(ideaId, expTimeMap)) {

            Long cnt = getObject2Long(expTimeMap.get(ideaId));
            if (cnt == null || cnt < 1) {
                ret = 0L;
            } else if (cnt <= 7) {
                ret = cnt;
            } else {
                ret = -1L;
            }
        }
        return ret;
    }


    //根据areaCode解析地区级别（华北/东北/华东/华中/华南/西南/西北/港澳） f4010031
    public static Long getRegionLevel(Long areaCode) {

        Long ret = null;

        if (AssertUtil.isNotEmpty(areaCode)) {

            int area = DataUtil.toInt(areaCode);
            if (areaCode >= 1000L && areaCode != 8100L && areaCode != 8200L) {
                area = DataUtil.toInt(Long.valueOf(areaCode / 100));
            }

            if (Arrays.asList(HUABEI_REGION).contains(area)) {
                ret = 80L;
            } else if (Arrays.asList(DONGBEI_REGION).contains(area)) {
                ret = 81L;
            } else if (Arrays.asList(HUADONG_REGION).contains(area)) {
                ret = 82L;
            } else if (Arrays.asList(HUAZHONG_REGION).contains(area)) {
                ret = 83L;
            } else if (Arrays.asList(HUANAN_REGION).contains(area)) {
                ret = 84L;
            } else if (Arrays.asList(XINAN_REGION).contains(area)) {
                ret = 85L;
            } else if (Arrays.asList(XIBEI_REGION).contains(area)) {
                ret = 86L;
            } else if (Arrays.asList(HMT_REGION).contains(area)) {
                ret = 87L;
            }

        }

        return ret;
    }


    //根据areaCode解析城市级别（一线/二线/三线） f4010032
    public static Long getTierCityLevel(Long areaCode) {

        Long ret = null;

        if (AssertUtil.isNotEmpty(areaCode)) {

            int area = DataUtil.toInt(areaCode);

            if (Arrays.asList(FIRST_TIER_CITY).contains(area)) {
                ret = 1L;
            } else if (Arrays.asList(SECOND_TIER_CITY).contains(area)) {
                ret = 2L;
            } else if (Arrays.asList(THIRD_TIER_CITY).contains(area)) {
                ret = 3L;
            }

        }

        return ret;
    }


    // 设备号 当天竞价次数/当天曝光次数
    public static Long getImeiDayTimes(String imei, Long timesLimit, Map<String, Long> timesMap) {

        Long ret = 0L;

        if (AssertUtil.isAllNotEmpty(imei, timesMap)) {

            Long cnt = timesMap.get(imei);

            if (cnt == null || cnt < 1) {
                ret = 0L;
            } else if (cnt <= timesLimit) {
                ret = cnt;
            } else {
                ret = 99L;
            }
        }
        return ret;
    }


    // 资源位id+设备号  当天竞价次数/当天曝光次数
    public static Long getResoDayTimes(Long resoId, String imei, Long timesLimit, Map<String, Long> timesMap) {

        Long ret = 0L;

        if (AssertUtil.isAllNotEmpty(imei, resoId, timesMap)) {

            String statKey = DataUtil.Long2String(resoId) + "_" + imei;
            Long cnt = timesMap.get(statKey);

            if (cnt == null || cnt < 1) {
                ret = 0L;
            } else if (cnt <= timesLimit) {
                ret = cnt;
            } else {
                ret = 99L;
            }
        }
        return ret;
    }


    //判断当前id与上一次id是否相同
    public static String getLastEqualStatus(String currentId1, String currentId2, String lastId) {
        Integer ret = 0;
        String currentId = null;
        if (AssertUtil.isNotEmpty(currentId1) && AssertUtil.isEmpty(currentId2)) {currentId = currentId1;}
        else if (AssertUtil.isNotEmpty(currentId2) && AssertUtil.isEmpty(currentId1)) {currentId = currentId2;}
        else if (AssertUtil.isAllNotEmpty(currentId1, currentId2)) {currentId = currentId1 + "," + currentId2;}

        if (AssertUtil.isAllNotEmpty(currentId, lastId)) {
            ret = currentId.equals(lastId) ? 1 : 0;

        } else if (AssertUtil.isAllEmpty(currentId, lastId)){
            ret = null;
        }
        return DataUtil.Integer2String(ret);
    }



    //测试
    public static void main(String[] args) {

        Long currentGmtCreateTime1 = new Date().getTime();
        System.out.println("time:" + currentGmtCreateTime1);

        Integer hour = getTimeStampToHour(currentGmtCreateTime1);
        System.out.println("hour:" + DataUtil.Integer2String(hour));

        Integer weekDay = getTimeStampToWeekDay(currentGmtCreateTime1);
        System.out.println("weekday:" + DataUtil.Integer2String(weekDay));

        Long currentTime1 = getStringToTimeStamp();
        System.out.println("time1:" + currentTime1);

        System.out.println("ret1:" + getFloat2String(null));
        System.out.println("ret2:" + getObject2Long(null));

        System.out.println("getRegionLevel:" + getRegionLevel(4690L));
        System.out.println("getTierCityLevel:" + getTierCityLevel(null));

        Map<String, Long> timesMap1 = new HashMap<>();
        timesMap1.put("123_A", 10L);
        System.out.println("getImeiDayTimes:" + getImeiDayTimes("A", 7L, timesMap1));
        System.out.println("getResoDayTimes:" + getResoDayTimes(123L, "A", 50L, timesMap1));

        AdxFeatureDo adxFeatureDo1 = new AdxFeatureDo();
        adxFeatureDo1.setCurrentCreatTime(new Date().getTime());
        Map<String, String> staticMap1 = new HashMap();
        staticMap1 = AdxFeatureParse.generateFeatureMapStatic(adxFeatureDo1);
        System.out.println("staticMap1=" + JSON.toJSONString(staticMap1));

        System.out.println("getLength=" + FeatureBaseParse.getLengthLevel("1,2,3,4,5,6,7,8,9,10",5,10,5));
        System.out.println("getCategoryLevel=" + FeatureBaseParse.getCategoryLevel("11,5,10,0,9",10,1));
        System.out.println("getExpClickCrossValue=" + FeatureBaseParse.getExpClickCrossValue(null, null));

        AdxIdeaMaterialDo ma1 = new AdxIdeaMaterialDo();
        ma1.setMaterialId(111L);
        AdxIdeaMaterialDo ma2 = new AdxIdeaMaterialDo();
        ma2.setMaterialId(112L);
        List<AdxIdeaMaterialDo> materialList = new ArrayList<>();
        materialList.add(0,ma1);
        materialList.add(1,null);
        materialList.add(2,ma2);
        System.out.println("getMaterialFeature="+FeatureBaseParse.getMaterialFeature(materialList,"materialId"));
    }

}