package cn.com.duiba.nezha.alg.model;

import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.model.enums.MutModelType;
import cn.com.duiba.nezha.alg.model.enums.PredictResultType;
import cn.com.duiba.nezha.alg.model.tf.TFServingClient;
import com.alibaba.fastjson.JSON;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ESMM implements Serializable {
    private static final long serialVersionUID = -316102112618444131L;

    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(ESMM.class);

    private int PB_MAX_SIZE = 128;

    /**
     * 点击率预估模型
     */
    private IModel ctrModel;

    /**
     * 转化率预估模型
     */
    private IModel cvrModel;


    /**
     * 降级点击率预估模型
     */
    private IModel ctrFusingModel;

    /**
     * 降级转化率预估模型
     */
    private IModel cvrFusingModel;

    /**
     * 后端预估模型
     */
    private IModel bCvrModel;


    /**
     * git
     * 点击率预估深度学习TF-S-Client
     */
    private TFServingClient ctrTFServingClient;

    /**
     * 转化率预估深度学习TF-S-Client
     */
    private TFServingClient cvrTFServingClient;

    /**
     * 模型类型 ESMM  UN_ESMM
     */
    private MutModelType mutModelType;


    public IModel getCtrModel() {
        return ctrModel;
    }

    public void setCtrModel(IModel ctrModel) {
        this.ctrModel = ctrModel;
    }

    public IModel getCvrModel() {
        return cvrModel;
    }

    public void setCvrModel(IModel cvrModel) {
        this.cvrModel = cvrModel;
    }

    public IModel getBCvrModel() {
        return bCvrModel;
    }

    public void setBCvrModel(IModel bCvrModel) {
        this.bCvrModel = bCvrModel;
    }


    public TFServingClient getCtrTFServingClient() {
        return ctrTFServingClient;
    }

    public void setCtrTFServingClient(TFServingClient ctrTFServingClient) {
        this.ctrTFServingClient = ctrTFServingClient;
    }

    public TFServingClient getCvrTFServingClient() {
        return cvrTFServingClient;
    }

    public void setCvrTFServingClient(TFServingClient cvrTFServingClient) {
        this.cvrTFServingClient = cvrTFServingClient;
    }

    public MutModelType getMutModelType() {
        return mutModelType;
    }

    public void setMutModelType(MutModelType mutModelType) {
        this.mutModelType = mutModelType;
    }

    public IModel getCtrFusingModel() {
        return ctrFusingModel;
    }

    public void setCtrFusingModel(IModel ctrFusingModel) {
        this.ctrFusingModel = ctrFusingModel;
    }

    public IModel getCvrFusingModel() {
        return cvrFusingModel;
    }

    public void setCvrFusingModel(IModel cvrFusingModel) {
        this.cvrFusingModel = cvrFusingModel;
    }

    /**
     * @param ctrModel
     * @param cvrModel
     * @param bCvrModel
     * @param ctrTFServingClient
     * @param cvrTFServingClient
     * @param mutModelType
     */
    public ESMM(IModel ctrModel, IModel cvrModel, IModel ctrFusingModel, IModel cvrFusingModel, IModel bCvrModel, TFServingClient ctrTFServingClient, TFServingClient cvrTFServingClient, MutModelType mutModelType) {
        this.ctrModel = ctrModel;
        this.cvrModel = cvrModel;
        this.ctrFusingModel = ctrFusingModel;
        this.cvrFusingModel = cvrFusingModel;
        this.bCvrModel = bCvrModel;
        this.ctrTFServingClient = ctrTFServingClient;
        this.cvrTFServingClient = cvrTFServingClient;
        this.mutModelType = mutModelType;
    }

    /**
     * @param ctrModel
     * @param cvrModel
     * @param bCvrModel
     * @param ctrTFServingClient
     * @param cvrTFServingClient
     * @param mutModelType
     */
    public ESMM(IModel ctrModel, IModel cvrModel, IModel bCvrModel, TFServingClient ctrTFServingClient, TFServingClient cvrTFServingClient, MutModelType mutModelType) {
        this.ctrModel = ctrModel;
        this.cvrModel = cvrModel;
        this.bCvrModel = bCvrModel;
        this.ctrTFServingClient = ctrTFServingClient;
        this.cvrTFServingClient = cvrTFServingClient;
        this.mutModelType = mutModelType;
    }

    /**
     * 预估主接口
     * <p>
     * <p>
     * <p>
     * <p>
     * <p>
     * 点击率预估 && 转化率预估  &&  深度学习
     *
     * @param featureMap
     * @param <T>
     * @return
     * @throws Exception
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsWithTF(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = null;


        if (MutModelType.CTR == mutModelType) {
            ret = predictCTRs(featureMap);
        } else {
            ret = predictCTRsAndCVRsWithTFDEEP(featureMap);
        }
        return ret;
    }


    /**
     * CTR预估
     *
     * @param featureMap
     * @param <T>
     * @return
     * @throws Exception
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCTRs(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = ctrModel.predictWithTF(featureMap, ctrTFServingClient);

        if (AssertUtil.isAllNotEmpty(ctrMap)) {
            ret.put(PredictResultType.CTR, ctrMap);
        } else {
            ret = null;
        }
        return ret;
    }


    /**
     * CVR预估
     *
     * @param featureMap
     * @param <T>
     * @return
     * @throws Exception
     */
    public <T> Map<PredictResultType, Map<T, Double>> predictCVRs(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = cvrModel.predictWithTF(featureMap, cvrTFServingClient);

        if (AssertUtil.isAllNotEmpty(ctrMap)) {
            ret.put(PredictResultType.CVR, ctrMap);
        } else {
            ret = null;
        }

        return ret;

    }

    public <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRsWithTFDEEP(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = null;

        if (MutModelType.DEEP_ESMM == mutModelType) {
            /**
             * esmm 深度
             */

            ret = ctrTFServingClient.predictMut(getParams(featureMap));

        } else {
            /**
             * 普通 深度
             *
             */
            ret = predictCTRsAndCVRs(featureMap);
        }


        if (ret == null) {
            ret = predictFusingCTRsAndCVRs(featureMap);

            if (ret != null) {

                for (Map.Entry<PredictResultType, Map<T, Double>> entry : ret.entrySet()) {
                    PredictResultType key = entry.getKey();
                    if (key == PredictResultType.CTR) {
                        for (Map.Entry<T, Double> subEntry : entry.getValue().entrySet()) {
                            Double predictCtr = subEntry.getValue();

                            if (predictCtr == 0) {
                                String logInfo = ctrTFServingClient.modelName + " predictFusing warn preCtr=0";
                                logger.warn(logInfo);
                            }
                        }
                    }
                }
            } else {
                String logInfo = ctrTFServingClient.modelName + " predictFusing res=null";
                logger.warn(logInfo);
            }

        }


        return ret;

    }

    /**
     * 后端预估
     *
     * @param featureMap
     * @param <T>
     * @return
     * @throws Exception
     */
    public <T> Map<T, Double> predictBCVRs(Map<T, Map<String, String>> featureMap) throws Exception {
        return bCvrModel.predicts(featureMap);
    }


    /**
     * CTR && CVR预估
     *
     * @param featureMap
     * @param <T>
     * @return
     * @throws Exception
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictCTRsAndCVRs(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        Map<T, Double> ctrMap = ctrModel.predictWithTF(featureMap, ctrTFServingClient);

        Map<T, Double> cvrMap = cvrModel.predictWithTF(featureMap, cvrTFServingClient);
        if (AssertUtil.isAllNotEmpty(ctrMap, cvrMap)) {

            ret.put(PredictResultType.CTR, ctrMap);
            ret.put(PredictResultType.CVR, cvrMap);
        } else {
            ret = null;
        }

        return ret;

    }

    /**
     * 降级 CTR && CVR预估
     *
     * @param featureMap
     * @param <T>
     * @return
     * @throws Exception
     */
    private <T> Map<PredictResultType, Map<T, Double>> predictFusingCTRsAndCVRs(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<PredictResultType, Map<T, Double>> ret = new HashMap<>(PB_MAX_SIZE);

        ret.put(PredictResultType.CTR, ctrFusingModel.predictWithTF(featureMap, null));
        ret.put(PredictResultType.CVR, cvrFusingModel.predictWithTF(featureMap, null));
        return ret;

    }


    /**
     * @param featureMap
     * @return
     * @throws Exception
     */
    private <T> Map<T, List<Float>> getParams(Map<T, Map<String, String>> featureMap) throws Exception {
        Map<T, List<Float>> ret = new HashMap<>(PB_MAX_SIZE);

        if (AssertUtil.isNotEmpty(featureMap)) {
            for (Map.Entry<T, Map<String, String>> entry : featureMap.entrySet()) {
                List<Float> ctrP = ctrModel.getParam(entry.getValue());
                List<Float> cvrP = cvrModel.getParam(entry.getValue());
                ret.put(entry.getKey(), paramsSplicing(ctrP, cvrP));
            }
        }
        return ret;

    }


    private List<Float> paramsSplicing(List<Float> ctrP, List<Float> cvrP) {
        ctrP.addAll(cvrP);
        return ctrP;
    }


}