package cn.com.duiba.nezha.compute.common.model.activityselect;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Comparator;
import java.util.Queue;


public class ActivitySelector {

    private static final Logger logger = LoggerFactory.getLogger(ActivitySelector.class);

    static class Constant {
        static double MIN_REWARD = 0.1;
        static long DISCOUNT = 2;
        static  int MAX_HIS_VAL = 10000;
        static double DECAY = 0.99;  //100次以前的观察，无效
    }

    static class RankInfo {
        double grpm;
        double gexp;
        double hrpm;
        double hexp;
        double arpm;
        double aexp;
    }

    static class MatchInfo {
        double score;
        ActivityInfo act;
    }

    public static  int getCoef(double hisRequest, double request, double hisSend, double send)
    {
        int coef = 1;
        int i = 1,j=1;
        if(request < 0)
        {
            for(i = 1;i<144;i++)
            {
                request += hisRequest/144;
                if(request > 0)
                    break;
            }
        }
        if(send < 0)
        {
            for(j = 1;j<144;j++)
            {
                send += hisSend/144;
                if(send > 0)
                    break;
            }
        }
        coef = coef + Math.max(i,j);
        return coef;
    }

    public static Comparator<MatchInfo> iComparator = new Comparator<MatchInfo>() {

        @Override
        public int compare(MatchInfo c1, MatchInfo c2) {
            return (int) (c2.score - c1.score >= 0 ? 1 : -1);
        }
    };

    //recall
    public static List<ActivityInfo> match(List<ActivityInfo> activityInfos, int topn) {

        List<ActivityInfo> result = new ArrayList<>();

        Queue<MatchInfo> candis = new PriorityQueue<>(activityInfos.size(), iComparator);
        Queue<MatchInfo> slotcandis = new PriorityQueue<>(activityInfos.size(), iComparator);

        for (ActivityInfo activity : activityInfos) {

            //update history
            double request = 0;
            double send = 0;
            int co = 1;

            request = activity.request.globalVal - activity.lastRequest.globalVal;  //half-hour
            send = activity.send.globalVal - activity.lastSend.globalVal;
            co = getCoef(activity.request.globalVal,request,activity.send.globalVal,send);
            request = request + co * activity.request.globalVal / 144;
            send = send + co * activity.send.globalVal / 144;

            double lastVal = activity.hisSend.globalVal;
            activity.hisSend.globalVal = lastVal > Constant.MAX_HIS_VAL ? (activity.hisSend.globalVal + send) / Constant.DISCOUNT : activity.hisSend.globalVal + send;
            activity.hisRequest.globalVal = lastVal > Constant.MAX_HIS_VAL ? (activity.hisRequest.globalVal+request) / Constant.DISCOUNT : activity.hisRequest.globalVal + request;

            request = activity.request.appVal - activity.lastRequest.appVal;
            send = activity.send.appVal - activity.lastSend.appVal;
            co = getCoef(activity.request.appVal,request,activity.send.appVal,send);
            request = request + co * activity.request.appVal / 144;
            send = send + co * activity.send.appVal / 144;
            lastVal = activity.hisSend.appVal;
            activity.hisSend.appVal = lastVal > Constant.MAX_HIS_VAL ? (activity.hisSend.appVal+send) / Constant.DISCOUNT : activity.hisSend.appVal + send;
            activity.hisRequest.appVal = lastVal > Constant.MAX_HIS_VAL ? (activity.hisRequest.appVal+request) / Constant.DISCOUNT : activity.hisRequest.appVal + request;

            request = activity.request.slotVal - activity.lastRequest.slotVal;
            send = activity.send.slotVal - activity.lastSend.slotVal;
            co = getCoef(activity.request.slotVal,request,activity.send.slotVal,send);
            request = request + co * activity.request.slotVal / 144;
            send = send + co * activity.send.slotVal / 144;
            lastVal = activity.hisSend.slotVal;
            activity.hisSend.slotVal = lastVal > Constant.MAX_HIS_VAL ? (activity.hisSend.slotVal+send) / Constant.DISCOUNT : activity.hisSend.slotVal + send;
            activity.hisRequest.slotVal = lastVal > Constant.MAX_HIS_VAL ? (activity.hisRequest.slotVal+request) / Constant.DISCOUNT : activity.hisRequest.slotVal + request;

            //新活动试投
            int limit = topn;
            if (activity.hisRequest.globalVal < 100)
            {
                if (activityInfos.size() > limit) {
                    if(Math.random() < 0.0001)
                    {
                        result.add(activity);
                        topn--;
                    }
                }
                else
                {
                    result.add(activity);
                    topn--;
                }
            }
            else {
                //计算matchscore
                double slotScore = WilsonInterval.wilsonCalc((long) activity.hisSend.slotVal / 3, (long) activity.hisRequest.slotVal * 3).lowerBound;
                double globalScore = WilsonInterval.wilsonCalc((long) activity.hisSend.globalVal / 3, (long) activity.hisRequest.globalVal * 3).lowerBound;
                double appScore = WilsonInterval.wilsonCalc((long) activity.hisSend.appVal / 3, (long) activity.hisRequest.appVal * 3).lowerBound;

                double coef = 0, matchscore = 0;

                double sconfidence = Math.min(activity.hisSend.slotVal / 100, 1);
                double aconfidence = Math.min(activity.hisSend.appVal / 100, 1);
                double gconfidence = Math.min(activity.hisSend.globalVal / 1000, 1);

                matchscore = sconfidence * slotScore + (1 - sconfidence) * aconfidence * appScore * 0.7 + (1 - sconfidence-(1 - sconfidence) * aconfidence) * globalScore * 0.3 * Math.max(0.5,gconfidence);

                MatchInfo info = new MatchInfo();
                info.act = activity;
                info.score = matchscore;
                candis.add(info);

                if(sconfidence > 0.5)
                {
                    MatchInfo info2 = new MatchInfo();
                    info2.act = activity;
                    info2.score = slotScore;
                    slotcandis.add(info2);
                }
            }

        }

        int size1 = slotcandis.size();
        for (int i = 0; i < 20 && i < size1; i++) {
            result.add(slotcandis.poll().act);
        }

        int size = candis.size();
        for (int i = 0; i < topn && i < size; i++) {
            result.add(candis.poll().act);
        }

        return result;
    }

    //100 - 10 - 1
    public static ActivityInfo select(List<ActivityInfo> activityInfos) {


        //1、init
        ArrayList<Double> rewards = new ArrayList<>();
        ArrayList<Double> counts = new ArrayList<>();
        ArrayList<Double> alphas = new ArrayList<>();
        ArrayList<Double> betas = new ArrayList<>();

        double decay = Constant.DECAY;

        HashMap<Long, RankInfo> mMap = new HashMap();
        double maxG = 0.1, maxH = 0.1, maxA = 0.1;

        int size = activityInfos.size();

        //2、match
        activityInfos = match(activityInfos, 45);

        //3、rank
        //3.1 get info
        for (ActivityInfo act : activityInfos) {
            //get global data
            RankInfo info = mMap.containsKey(act.activityId) ? mMap.get(act.activityId) : new RankInfo();
            double grpm = act.hisRequest.globalVal > 0 ? act.hisSend.globalVal / act.hisRequest.globalVal : 0;
            info.grpm = grpm;
            info.gexp = act.hisRequest.globalVal;
            maxG = Math.max(grpm, maxG);

            //System.out.println(act.hisSend.globalVal+" "+act.hisRequest.globalVal+" "+grpm);

            //get app data
            double arpm = act.hisRequest.appVal > 0 ? act.hisSend.appVal / act.hisRequest.appVal : 0;
            info.arpm = arpm;
            info.aexp = act.hisRequest.appVal;
            if (info.hexp > 50)
                maxA = Math.max(arpm, maxA);

            //get slot data
            double hrpm = act.hisRequest.slotVal > 0 ? act.hisSend.slotVal / act.hisRequest.slotVal : 0;
            info.hrpm = hrpm;
            info.hexp = act.hisRequest.slotVal;
            if (info.hexp > 50)
                maxH = Math.max(hrpm, maxH);

            //System.out.println(hrpm+"\t"+maxH);

            mMap.put(act.activityId, info);
        }

        List<ActivityInfo> candiList = new ArrayList<>();

        for (ActivityInfo act : activityInfos) {
            if (act.request.globalVal > 0) {
                double reward = Constant.MIN_REWARD;
                double sconfidence = Math.min(act.hisRequest.slotVal / 100,1);
                double aconfidence = Math.min(act.hisRequest.appVal / 100,1);

                reward = sconfidence * normlize(mMap.get(act.activityId).hrpm * 0.8, maxH, 0.8) +
                        (1 - sconfidence) * aconfidence * normlize(mMap.get(act.activityId).arpm * 0.7, maxA, 0.7)+
                        (1 - sconfidence - (1 - sconfidence) * aconfidence) * normlize(mMap.get(act.activityId).grpm * 0.6, maxG, 0.6);
                reward = Math.max(reward, Constant.MIN_REWARD);

                act.reward = act.reward * decay + reward;
                act.count = act.count * decay + 1.0;

                if(sconfidence > 0.1 && act.count < 20) //减少试投成本
                {
                    act.reward = 20 * reward;
                    act.count = 20;
                }

                long oneday = 24 * 60 * 60 * 1000;
                if((sconfidence > 0.99 || System.currentTimeMillis() - act.updateTime > oneday))    //临时脏数据修复代码
                {
                    if(act.reward / act.count < 0.6 * reward || act.reward / act.count > 1.6 * reward)
                    {
                        act.reward = 10 * reward;
                        act.count = 10;
                    }
                }

                act.alpha = 1.5 + act.reward;
                act.beta = 2.0 + (act.count - act.reward);
            }

            rewards.add(act.reward);
            counts.add(act.count);
            alphas.add(act.alpha);
            betas.add(act.beta);
            candiList.add(act);
        }

        //4、select
        int numMachines = candiList.size();

        ActivityInfo result = candiList.get(selectMachine(alphas, betas, numMachines));


        //System.out.println(rewards.toString()+" "+alphas.toString()+" "+betas.toString()+" "+numMachines+" "+selectMachine(alphas, betas, numMachines));

        if(System.currentTimeMillis() - result.updateTime > 60 * 1000) //1分钟更新一次
        {
            result.isUpdate = true;
        }
        mMap.clear();

        //System.out.println(result.activityId);

        result.lastRequest.appVal = result.request.appVal;
        result.lastRequest.slotVal = result.request.slotVal;
        result.lastRequest.globalVal = result.request.globalVal;

        result.lastSend.appVal = result.send.appVal;
        result.lastSend.slotVal = result.send.slotVal;
        result.lastSend.globalVal = result.send.globalVal;

        //print(result);

        return result;
    }


    public static  void print(ActivityInfo act)
    {
        System.out.println("-----------"+act.activityId+"-----------");
        System.out.println("request = "+act.request.slotVal+"\t"+act.request.appVal+"\t"+act.request.globalVal);
        System.out.println("send = "+act.send.slotVal+"\t"+act.send.appVal+"\t"+act.send.globalVal);
        System.out.println("lastRequest = "+act.lastRequest.slotVal+"\t"+act.lastRequest.appVal+"\t"+act.lastRequest.globalVal);
        System.out.println("lastSend = "+act.lastSend.slotVal+"\t"+act.lastSend.appVal+"\t"+act.lastSend.globalVal);
        System.out.println("hisRequest = "+act.hisRequest.slotVal+"\t"+act.hisRequest.appVal+"\t"+act.hisRequest.globalVal);
        System.out.println("hisSend = "+act.hisSend.slotVal+"\t"+act.hisSend.appVal+"\t"+act.hisSend.globalVal);
    }

    public static double normlize(double val, double max, double limit) {
        double norm = Math.min(val * limit / max, limit);
        return norm;
    }


    public ActivitySelector() { //下周

    }

    private static int selectMachine(List<Double> alphas, List<Double> betas, int numMachines) {
        int selectMachine = 0;
        double maxTheta = 0;
        for (int i = 0; i < numMachines; i++) {
            double theta = BetaDistribution.BetaDist(alphas.get(i), betas.get(i));
            if (theta > maxTheta) {
                maxTheta = theta;
                selectMachine = i;
            }
        }
        return selectMachine;
    }

    private double getCtr(double exp, double clk) {
        return exp > 0 ? clk / exp : 0;
    }

    private double sum(List<Long> list) {
        double sum = 0;
        for (Long val : list) {
            sum += val;
        }
        return sum;
    }

    public static double wilsonRoofLeft(Double ratio, Double num) {
        if (ratio == null || num == null)
            return 0;
        long fenzi = (long) (ratio * num);
        long fenmu = num.longValue();
        WilsonPair pair = WilsonInterval.wilsonCalc(fenzi, fenmu);
        return pair.upperBound;
    }

    public double wilsonRoofRight(Double num, Double ratio) {
        if (ratio == null || num == null)
            return 0;
        if (ratio == 0) {
            ratio = 0.00000001;
        }
        long fenmu = (long) (num / ratio);
        long fenzi = num.longValue();
        WilsonPair pair = WilsonInterval.wilsonCalc(fenzi, fenmu);
        return pair.upperBound;
    }

    public double wilsonBottom(Long fenzi, Long fenmu) {
        if (fenzi == null || fenmu == null)
            return 0;
        WilsonPair pair = WilsonInterval.wilsonCalc(fenzi.longValue(), fenmu.longValue());
        return pair.lowerBound;
    }

    public static class SingletonHolder {
        private static final ActivitySelector instance = new ActivitySelector();
    }

    private static ActivitySelector getInstance() {
        return SingletonHolder.instance;
    }
}