package cn.com.duiba.nezha.engine.biz.service.advert;

import cn.com.duiba.nezha.alg.alg.vo.AutoBiddingDo;
import cn.com.duiba.nezha.engine.api.enums.FlowTag;
import cn.com.duiba.nezha.engine.api.support.RecommendEngineException;
import cn.com.duiba.nezha.engine.biz.constant.GlobalConstant;
import cn.com.duiba.nezha.engine.biz.domain.AutoBiddingFactorIndex;
import cn.com.duiba.nezha.engine.biz.domain.StatisticData;
import cn.com.duiba.nezha.engine.biz.domain.advert.Advert;
import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.biz.domain.mergeData.MergeData;
import cn.com.duiba.nezha.engine.biz.enums.AdvertType;
import cn.com.duiba.nezha.engine.biz.enums.BackendDefaultCVR;
import cn.com.duiba.nezha.engine.biz.enums.MergeDataType;
import cn.com.duiba.nezha.engine.biz.service.advert.ctr.AdvertMergeStatService;
import cn.com.duiba.nezha.engine.biz.service.advert.ctr.AdvertStatService;
import cn.com.duiba.nezha.engine.biz.service.advert.ctr.TagStatAssociationService;
import cn.com.duiba.nezha.engine.biz.vo.advert.AdvertRecommendRequestVo;
import cn.com.duiba.nezha.engine.common.utils.Pair;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

@Service
public class DataHandleBo {

    protected Logger logger = LoggerFactory.getLogger(getClass());
    @Autowired
    private AdvertMergeStatService advertMergeStatService;
    @Autowired
    private AdvertStatService advertStatService;
    @Autowired
    private OrientationPackageAdjustPriceFactorService orientationPackageAdjustPriceFactorService;
    @Autowired
    private TagStatAssociationService tagStatAssociationService;
    @Autowired
    private AdvertSmartShopService advertSmartShopService;
    @Autowired
    private TradeRepetitionLaunchService tradeRepetitionLaunchService;

    // 处理融合数据
    public void handleMergeData(Collection<Advert> adverts, Long appId) {
        try {
            DBTimeProfile.enter("handleMergeData");

            // 获取 广告的全局融合数据 和 广告在媒体上 的融合数据
            Map<MergeDataType, List<MergeData>> advertAndAppData = advertMergeStatService.getAdvertAndAppData(adverts, appId);

            Map<Long, MergeData> appMergeDataMap = advertAndAppData
                    .getOrDefault(MergeDataType.APP, new ArrayList<>())
                    .stream()
                    .collect(toMap(MergeData::getAdvertId, identity()));

            Map<Long, MergeData> globalMergeDataMap = advertAndAppData
                    .getOrDefault(MergeDataType.GLOBAL, new ArrayList<>())
                    .stream()
                    .collect(toMap(MergeData::getAdvertId, identity()));

            // 设置融合数据,计算统计的ctr和统计的cvr
            adverts.forEach(advert -> {
                MergeData appMergeData = appMergeDataMap.get(advert.getId());
                MergeData globalMergeData = globalMergeDataMap.get(advert.getId());
                advert.setAppMergeData(appMergeData);
                advert.setGlobalMergeData(globalMergeData);
                this.handleStatCtrAndCvr(advert);
            });
        } finally {
            DBTimeProfile.release();
        }
    }


    // 处理广告的统计ctr和统计cvr
    private void handleStatCtrAndCvr(Advert advert) {

        Optional<MergeData> globalMergeData = advert.getGlobalMergeData();

        advert.getOrientationPackages().forEach(orientationPackage -> {

            if (advert.getType().equals(AdvertType.SHOW)) {// 展示广告
                orientationPackage.setStatCtr(globalMergeData.map(MergeData::getCtr).orElse(GlobalConstant.SHOW_DEFAULT_CTR));
                orientationPackage.setStatCvr(globalMergeData.map(MergeData::getCvr).orElse(GlobalConstant.SHOW_DEFAULT_CVR));
                // 如果媒体的统计ctr和cvr存在.则使用媒体上的
                Optional<MergeData> appMergeData = advert.getAppMergeData();
                appMergeData.map(MergeData::getCtr).ifPresent(orientationPackage::setStatCtr);
                appMergeData.map(MergeData::getCvr).ifPresent(orientationPackage::setStatCvr);

            } else {// 互动广告 + 增值广告
                Integer cvrType = orientationPackage.getCvrType();

                orientationPackage.setStatCtr(globalMergeData.map(MergeData::getCtr).orElse(GlobalConstant.INTERACT_DEFAULT_CTR));
                orientationPackage.setStatCvr(globalMergeData.map(mergeData -> mergeData.getCvr(cvrType)).orElse(BackendDefaultCVR.getCvr(cvrType)));

                // 如果媒体的统计ctr和cvr存在.则使用媒体上的
                Optional<MergeData> appMergeData = advert.getAppMergeData();
                appMergeData.map(MergeData::getCtr).ifPresent(orientationPackage::setStatCtr);
                appMergeData.map(mergeData -> mergeData.getCvr(cvrType)).ifPresent(orientationPackage::setStatCvr);
            }
        });
    }

    // 处理调价因子
    public void handleAdjustPriceFactor(Set<OrientationPackage> orientationPackages, Long appId, Long slotId) {
        orientationPackageAdjustPriceFactorService.handle(orientationPackages, appId, slotId);
    }

    // 处理统计数据
    public void handleStatisticData(Collection<Advert> adverts, Set<OrientationPackage> orientationPackages, Long appId) {
        try {
            DBTimeProfile.enter("handleStatisticData");

            Set<AdvertStatService.Query> queries = new HashSet<>();
            Map<Long, Pair<AdvertStatService.Query>> advertQueryMap = new HashMap<>();
            Map<OrientationPackage, Pair<AdvertStatService.Query>> packageQueryMap = new HashMap<>();

            orientationPackages.forEach(orientationPackage -> {
                Long advertId = orientationPackage.getAdvertId();
                Long packageId = orientationPackage.getId();
                AdvertStatService.Query advertQuery = AdvertStatService.Query.newBuilder().advertId(advertId).build();
                AdvertStatService.Query advertAppQuery = AdvertStatService.Query.newBuilder().advertId(advertId).appId(appId).build();

                AdvertStatService.Query packageQuery = AdvertStatService.Query.newBuilder().advertId(advertId).packageId(packageId).build();
                AdvertStatService.Query packageAppQuery = AdvertStatService.Query.newBuilder().advertId(advertId).packageId(packageId).appId(appId).build();

                queries.add(advertQuery);
                queries.add(advertAppQuery);
                queries.add(packageQuery);
                queries.add(packageAppQuery);
                advertQueryMap.put(advertId, Pair.of(advertQuery, advertAppQuery));
                packageQueryMap.put(orientationPackage, Pair.of(packageQuery, packageAppQuery));

            });
            //当前小时
            Map<AdvertStatService.Query, StatisticData> currentHourStatMap = advertStatService.getCurrentHourStat(queries);

            //当天
            Map<AdvertStatService.Query, StatisticData> currentDayStatMap = advertStatService.getCurrentDayStat(queries);

            //7天
            Map<AdvertStatService.Query, StatisticData> recently7DayStatMap = advertStatService.get7DayStat(queries);
            adverts.forEach(advert -> {
                Pair<AdvertStatService.Query> queryPair = advertQueryMap.get(advert.getId());
                Optional<AdvertStatService.Query> advertQuery = queryPair.getLeft();
                Optional<AdvertStatService.Query> advertAppQuery = queryPair.getRight();
                advertQuery.map(currentHourStatMap::get).ifPresent(advert::setHourStatisticData);
                advertQuery.map(currentDayStatMap::get).ifPresent(advert::setTodayStatisticData);
                advertQuery.map(recently7DayStatMap::get).ifPresent(advert::setRecently7DayStatisticData);
                advertAppQuery.map(currentHourStatMap::get).ifPresent(advert::setHourAppStatisticData);
                advertAppQuery.map(currentDayStatMap::get).ifPresent(advert::setTodayAppStatisticData);
                advertAppQuery.map(recently7DayStatMap::get).ifPresent(advert::setRecently7DayAppStatisticData);
            });

            orientationPackages.forEach(orientationPackage -> {
                Pair<AdvertStatService.Query> queryPair = packageQueryMap.get(orientationPackage);
                Optional<AdvertStatService.Query> packageQuery = queryPair.getLeft();
                Optional<AdvertStatService.Query> packageAppQuery = queryPair.getRight();
                packageQuery.map(currentHourStatMap::get).ifPresent(orientationPackage::setHourStatisticData);
                packageQuery.map(currentDayStatMap::get).ifPresent(orientationPackage::setTodayStatisticData);
                packageQuery.map(recently7DayStatMap::get).ifPresent(orientationPackage::setRecently7DayStatisticData);
                packageAppQuery.map(currentHourStatMap::get).ifPresent(orientationPackage::setHourAppStatisticData);
                packageAppQuery.map(currentDayStatMap::get).ifPresent(orientationPackage::setTodayAppStatisticData);
                packageAppQuery.map(recently7DayStatMap::get).ifPresent(orientationPackage::setRecently7DayAppStatisticData);
            });
        } finally {
            DBTimeProfile.release();
        }
    }

    public void handleSmartShopData(Set<OrientationPackage> orientationPackages) {
        try {
            DBTimeProfile.enter("handleSmartShopData");
            Map<OrientationPackage, AdvertStatService.Query> orientPackage2QueryMap = orientationPackages.stream()
                    .collect(toMap(Function.identity(), orientPackage -> AdvertStatService.Query.newBuilder()
                            .advertId(orientPackage.getAdvertId())
                            .packageId(orientPackage.getId())
                            .tag(FlowTag.GOOD.getTag())
                            .build()));
            // 取出 广告+ 配置 优质流量数据
            Map<AdvertStatService.Query, StatisticData> currentDayStat = advertStatService.getCurrentDayStat(orientPackage2QueryMap.values());

            orientationPackages.forEach(orientationPackage ->
                    orientationPackage.setSmartShopStatisticData(currentDayStat.get(orientPackage2QueryMap.get(orientationPackage))));

        } finally {
            DBTimeProfile.release();
        }
    }

    public void handleSmartShopWhiteBlackData(Set<OrientationPackage> orientationPackages, Long appId, Long slotId, Long activityId) {

        try {
            DBTimeProfile.enter("handleSmartShopWhiteBlackData");
            advertSmartShopService.handlePackageQualityLevelLevel(orientationPackages, appId, slotId, activityId);
        } finally {
            DBTimeProfile.release();
        }
    }

    // 处理时段数据
    public void handleHourlyData(Collection<Advert> adverts, Long appId) {
        try {
            DBTimeProfile.enter("handleHourlyData");
            Map<Advert, AdvertStatService.Query> advertQueryMap = adverts
                    .stream()
                    .collect(toMap(identity(),
                            advert -> AdvertStatService.Query.newBuilder().appId(appId).advertId(advert.getId()).build()));

            Map<AdvertStatService.Query, List<StatisticData>> todayHourlyStatMap = advertStatService.getTodayHourlyStat(advertQueryMap.values());

            adverts.forEach(advert -> {
                List<StatisticData> statisticDataList = todayHourlyStatMap.getOrDefault(advertQueryMap.get(advert), new ArrayList<>());

                advert.getOrientationPackages().forEach(orientationPackage -> {
                    List<Double> hourlyCtr = new ArrayList<>();
                    List<Double> hourlyCvr = new ArrayList<>();
                    List<Double> hourlyClick = new ArrayList<>();
                    List<Double> hourlyLaunch = new ArrayList<>();
                    List<Double> hourlyConsume = new ArrayList<>();

                    statisticDataList.forEach(statisticData -> {
                        hourlyCtr.add(statisticData.getCtr());
                        hourlyCvr.add(statisticData.getCvr(orientationPackage.getCvrType()));
                        hourlyClick.add(statisticData.getChargeClickCount().doubleValue());
                        hourlyLaunch.add(statisticData.getLaunchCount().doubleValue());
                        hourlyConsume.add(statisticData.getTotalConsume().doubleValue());

                    });

                    orientationPackage.setHourlyCtr(hourlyCtr);
                    orientationPackage.setHourlyCvr(hourlyCvr);
                    orientationPackage.setHourlyClick(hourlyClick);
                    orientationPackage.setHourlyLaunch(hourlyLaunch);
                    orientationPackage.setHourlyConsume(hourlyConsume);
                });


            });


        } catch (Exception e) {
            throw new RecommendEngineException("getTodayHourlyStat error", e);
        } finally {
            DBTimeProfile.release();
        }
    }

    // 处理标签数据
    public void handleTagData(Collection<Advert> adverts, Long appId) {
        try {
            DBTimeProfile.enter("handleTagData");
            tagStatAssociationService.handleTagStat(adverts, appId);
        } finally {
            DBTimeProfile.release();
        }

    }

    /**
     * hbase查询行业重复发全 衍生特征的数据
     *
     * @param advertRecommendRequestVo
     */
    public void handTradeRepetitionLaunch(AdvertRecommendRequestVo advertRecommendRequestVo) {

        try {
            DBTimeProfile.enter("handTradeRepetitionLaunch");
            //  处理用户 综合行业 以及发券次数、上次发券综合行业
            CatUtils.executeInCatTransaction(() -> {
                        tradeRepetitionLaunchService.handTradeRepetitionLaunchStat(advertRecommendRequestVo);
                        return null;
                    },"Hbase", "handTradeRepetitionLaunchStat");
            //  处理 用户偏好画像 数据
            tradeRepetitionLaunchService.handInterestPortrayalStat(advertRecommendRequestVo);
        } catch (Throwable throwable) {
          logger.error("handTradeRepetitionLaunch error !", throwable);
        } finally {
            DBTimeProfile.release();
        }
    }
}
