package cn.com.duiba.nezha.engine.biz.service.advert.merge;

import cn.com.duiba.nezha.alg.alg.basepricecontrol.BasePriceControl;
import cn.com.duiba.nezha.alg.alg.basepricecontrol.BasePriceResult;
import cn.com.duiba.nezha.alg.common.model.pacing.*;
import cn.com.duiba.nezha.alg.common.model.qscore.Qscore;
import cn.com.duiba.nezha.alg.common.model.qscore.QualityInfo;
import cn.com.duiba.nezha.engine.api.support.RecommendEngineException;
import cn.com.duiba.nezha.engine.biz.domain.*;
import cn.com.duiba.nezha.engine.biz.domain.advert.Advert;
import cn.com.duiba.nezha.engine.biz.domain.advert.OrientationPackage;
import cn.com.duiba.nezha.engine.biz.domain.advert.StatisticDataGetter;
import cn.com.duiba.nezha.engine.biz.entity.nezha.advert.AdvertStatisticMergeEntity;
import cn.com.duiba.nezha.engine.biz.service.CacheService;
import cn.com.duiba.nezha.engine.biz.service.advert.DataHandleBo;
import cn.com.duiba.nezha.engine.biz.service.advert.TrusteeshipRecommendService;
import cn.com.duiba.nezha.engine.biz.service.advert.floorprice.FloorPriceService;
import cn.com.duiba.nezha.engine.biz.service.advert.support.AdvertSupportService;
import cn.com.duiba.nezha.engine.biz.vo.advert.AdvertRecommendRequestVo;
import cn.com.duiba.nezha.engine.biz.vo.advert.AdvertStatDimWeightVo;
import cn.com.duiba.nezha.engine.biz.vo.advert.PacingResult;
import cn.com.duiba.nezha.engine.common.utils.FormatUtil;
import cn.com.duiba.nezha.engine.common.utils.RedisKeyUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.alibaba.fastjson.JSON;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.hazelcast.util.CollectionUtil;
import com.hazelcast.util.MapUtil;
import org.apache.commons.lang.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import static cn.com.duiba.nezha.engine.api.enums.ChargeTypeEnum.CPA;
import static java.util.stream.Collectors.*;

/**
 * Created by pc on 2016/10/18.
 */
@Service
public class AdvertDataMergeService extends CacheService {
    private static final Logger logger = LoggerFactory.getLogger(AdvertDataMergeService.class);

    //系统托管包类型
    private static final Integer SYSTEM_MANAGED = 2;

    @Autowired
    private DataHandleBo dataHandleBo;

    @Autowired
    private AdvertSupportService advertSupportService;

    @Autowired
    private TrusteeshipRecommendService trusteeshipRecommendService;
    @Autowired
    private FloorPriceService floorPriceService;

    public List<OrientationPackage> dataMerge(AdvertRecommendRequestVo advertRecommendRequestVo) {

        try {

            // 自动托管,过滤掉一批广告
            Set<OrientationPackage> orientationPackages = advertRecommendRequestVo.getAdvertOrientationPackages();
            DBTimeProfile.enter("dataMerge: " + orientationPackages.size());
            Set<OrientationPackage> finalOrientationPackages = orientationPackages;
            orientationPackages = CatUtils.executeInCatTransaction(() -> trusteeshipRecommendService.recommend(finalOrientationPackages, advertRecommendRequestVo),"dataMerge", "advertDataMergeService#dataMerge");


            // 扶持广告,过滤掉需要熔断的广告 主要去除一部分 配置
            AppDo appDo = advertRecommendRequestVo.getAppDo();
            Long slotId = appDo.getSlotId();
            Set<OrientationPackage> finalOrientationPackages1 = orientationPackages;
            CatUtils.executeInCatTransaction(() -> {
                advertSupportService.support(finalOrientationPackages1, slotId);
                return null;
            },"dataMerge", "advertSupportService#support");

            advertRecommendRequestVo.setAdvertOrientationPackages(orientationPackages);

            // 处理智能匹配数据
            Set<OrientationPackage> finalOrientationPackages2 = orientationPackages;
            CatUtils.executeInCatTransaction(() -> {
                dataHandleBo.handleSmartShopData(finalOrientationPackages2);
                return null;
            },"dataMerge", "dataHandleBo#handleSmartShopData");


            // 处理智能匹配黑白名单数据
            Long appId = appDo.getId();
            Long activityId = advertRecommendRequestVo.getActivityDo().getOperatingId();
            dataHandleBo.handleSmartShopWhiteBlackData(orientationPackages, appId, slotId, activityId);


            Map<Long, Advert> advertMap = advertRecommendRequestVo.getAdvertMap();

            // 统计数据权重对象
            AdvertStatDimWeightVo advertStatDimWeightVo = advertRecommendRequestVo.getAdvertStatDimWeightVo();
            double statCtrWeight = advertStatDimWeightVo.getStatCtrWeight();
            double preCtrWeight = advertStatDimWeightVo.getPreCtrWeight();
            double statCvrWeight = advertStatDimWeightVo.getStatCvrWeight();
            double preCvrWeight = advertStatDimWeightVo.getPreCvrWeight();


            Map<FeatureIndex, Double> predictCtr = advertRecommendRequestVo.getPredictCtr();
            Map<FeatureIndex, Double> predictCvr = advertRecommendRequestVo.getPredictCvr();

            Map<Long, Integer> userAdvertBehaviorMap = advertRecommendRequestVo.getUserAdvertBehaviorMap();
            orientationPackages.forEach(orientationPackage -> {
                Long advertId = orientationPackage.getAdvertId();
                Advert advert = advertMap.get(advertId);
                Long packageId = orientationPackage.getId();

                FeatureIndex featureIndex = orientationPackage
                        .getMaterials()
                        .stream()
                        .findAny()
                        .map(material -> new FeatureIndex(advertId, packageId, material.getId()))
                        .orElse(new FeatureIndex(advertId, packageId));

                Double statCtr = orientationPackage.getStatCtr();
                Double preCtr = predictCtr.getOrDefault(featureIndex, 0.0);
                Double ctr = this.getAdvertCtr(statCtr, statCtrWeight, preCtr, preCtrWeight, advert.isNew());

                Double statCvr = orientationPackage.getStatCvr();
                Double preCvr = predictCvr.getOrDefault(featureIndex, 0.0);
                Double cvr = this.getAdvertCvr(statCvr, statCvrWeight, preCvr, preCvrWeight, advert.isNew());

                Long clickFee = orientationPackage.getClickFee();
                Long convertCost = orientationPackage.getConvertCost();
                Double factor = orientationPackage.getAdjustPriceFactor();
                if (orientationPackage.getFeeWeightFactor() != null) {
                    factor = factor * orientationPackage.getFeeWeightFactor();
                }

                Long finalFee = this.calculateFinalFee(orientationPackage.isCpa(), factor, cvr, clickFee, convertCost);

                orientationPackage.setCtr(ctr);
                orientationPackage.setCvr(cvr);
                orientationPackage.setPreCtr(preCtr);
                orientationPackage.setPreCvr(preCvr);
                orientationPackage.setFinalFee(finalFee);
                orientationPackage.setUserBehavior(userAdvertBehaviorMap.getOrDefault(advertId, 0));

                PacingResult pacing = this.pacing(advert, orientationPackage);
                orientationPackage.setGiveUp(pacing.getGiveUp());
                orientationPackage.setFlowTag(pacing.getTag());

            });

            //托管自救放弃流量
            Set<OrientationPackage> finalOrientationPackages3 = orientationPackages;
            Set<OrientationPackage> giveUpRescurePackage = CatUtils.executeInCatTransaction(() -> trusteeshipRecommendService.giveUpRescurePackage(finalOrientationPackages3)
            ,"dataMerge", "trusteeshipRecommendService#giveUpRescurePackage");
            orientationPackages.removeAll(giveUpRescurePackage);

            //底价放弃
            Set<OrientationPackage> finalOrientationPackages4 = orientationPackages;
            Set<OrientationPackage> floorPriceGiveUpSet = CatUtils.executeInCatTransaction(() -> floorPriceGiveUp(advertRecommendRequestVo, finalOrientationPackages4)
                    ,"dataMerge", "advertDataMergeService#floorPriceGiveUp");
            orientationPackages.removeAll(floorPriceGiveUpSet);

            //维稳因子重置
            Map<Long, Double> ctrReconstructionFactorMap = advertRecommendRequestVo.getCtrReconstructionFactorMap();
            Map<Long, Double> cvrReconstructionFactorMap = advertRecommendRequestVo.getCvrReconstructionFactorMap();
            Boolean advertMultiDimScoreEffective = advertRecommendRequestVo.getAdvertMultiDimScoreEffective();

            //加载 arpu调整因子
            Double alphaFactor = arpuOptimizeFactorCache.get(RedisKeyUtil.getArpuOptimizeFactorKey());
            advertRecommendRequestVo.setAlphaFactor(alphaFactor);

            // 计算rankScore
            DBTimeProfile.enter("calculate rankScore");

            return orientationPackages.stream()
                    .peek(orientationPackage -> {
                        Long advertId = orientationPackage.getAdvertId();
                        Integer chargeType = orientationPackage.getChargeType();
                        Advert advert = advertMap.get(advertId);
                        Double ctr = orientationPackage.getCtr();
                        Double cvr = orientationPackage.getCvr();
                        Long finalFee = orientationPackage.getFinalFee();
                        Double tagWeight = advert.getWeight();
                        Double discountRate = advert.getDiscountRate();
                        Double ctrReconstructionFactor = ctrReconstructionFactorMap.getOrDefault(advertId, 1D);
                        Double cvrReconstructionFactor = cvrReconstructionFactorMap.getOrDefault(advertId, 1D);
                        double qScore = this.getScore(finalFee, chargeType, ctr, cvr, advertMultiDimScoreEffective);

                        if(advertRecommendRequestVo.getArpuOptimizeON()){
                            //特殊逻辑 qScore的 α 次方后保留6位小数处理
                            qScore = BigDecimal.valueOf(Math.pow(qScore, alphaFactor)).setScale(6,BigDecimal.ROUND_HALF_UP).doubleValue();
                        }

                        double rankScore = this.getRankScore(ctrReconstructionFactor, cvrReconstructionFactor, finalFee, tagWeight, qScore, discountRate);

                        Double supportWeight = orientationPackage.getSupportWeight();
                        rankScore = rankScore * supportWeight;

                        orientationPackage.setSupportWeight(supportWeight);
                        orientationPackage.setSupportSuccess(!supportWeight.equals(1.0D));

                        orientationPackage.setqScore(qScore);
                        orientationPackage.setRankScore(rankScore);
                    })
                    .collect(toList());
        } catch (Throwable throwable) {
            logger.error("dataMerge error !", throwable);
            return Lists.newArrayList();
        } finally {
            DBTimeProfile.release();
            DBTimeProfile.release();
        }
    }

    /**
     * 底价调整，获取需要放弃配置的
     *
     * @param advertRecommendRequestVo
     * @param orientationPackages
     */
    public Set<OrientationPackage> floorPriceGiveUp(AdvertRecommendRequestVo advertRecommendRequestVo, Set<OrientationPackage> orientationPackages) {

        Set<OrientationPackage> resultGiveUpSet = Sets.newHashSet();
        try {
            // 开启
            RequestDo requestDo = advertRecommendRequestVo.getRequestDo();
            AppDo appDo = advertRecommendRequestVo.getAppDo();

            Boolean floorPriceWhiteListOff = requestDo.getFloorPriceWhiteListOff();

            // 过滤所有 系统托管 的 cpa计费的 配置 ocpc
            List<OrientationPackage> packages = orientationPackages.stream()
                    .filter(orientationPackage ->
                            CPA.getValue().equals(orientationPackage.getChargeType()))
                    .filter(orientationPackage ->
                            SYSTEM_MANAGED.equals(orientationPackage.getPackageType()))
                    .collect(toList());


            List<OrientationPackage> floorPricePackages = packages;

            //如果 开启了白名单 则只保留 白名单里面的 广告 否则 全部广告都生效
            if (floorPriceWhiteListOff) {
                floorPricePackages = packages.stream().filter(OrientationPackage::getFloorPriceWhilteAdvert).collect(toList());
            }
            if (CollectionUtil.isEmpty(floorPricePackages)) {
                return resultGiveUpSet;
            }
            //广告+配置+媒体+广告位 ->finaFee
            Map<FloorPriceFeatureIndex, Long> finalFeeMap = floorPricePackages.stream().collect(toMap(orientationPackage -> FloorPriceFeatureIndex.conver(orientationPackage, appDo), OrientationPackage::getFinalFee));

            // OrientationPackage - > FloorPriceFeatureIndex
            Map<OrientationPackage, FloorPriceFeatureIndex> indexOrientationPackageMap = floorPricePackages.stream().collect(toMap(Function.identity(), orientationPackage -> FloorPriceFeatureIndex.conver(orientationPackage, appDo)));

            // 读取缓存的 底价调整对象
            Map<FloorPriceFeatureIndex, BasePriceResult> basePriceResultMap = floorPriceService.getFloorPriceFactor(appDo, floorPricePackages);

            //将本次 放弃结果比例打印
            setGiveUpProbsToLog(orientationPackages, indexOrientationPackageMap, basePriceResultMap);

            //本次竞价列表长度
            Integer biddingAdvertNum = orientationPackages.size();
            // 计算本次 请求 底价放弃配置
            Map<FloorPriceFeatureIndex, Boolean> giveUpControl = BasePriceControl.giveUpControl(basePriceResultMap, finalFeeMap, biddingAdvertNum);
            // 移除底价放弃配置
            if (MapUtil.isNullOrEmpty(giveUpControl)) {
                return resultGiveUpSet;
            }

            //过滤得到所有需要放弃的配置
            resultGiveUpSet = orientationPackages.stream().filter(orientationPackage -> {
                FloorPriceFeatureIndex floorPriceFeatureIndex = indexOrientationPackageMap.get(orientationPackage);
                if (floorPriceFeatureIndex != null) {
                    return giveUpControl.getOrDefault(floorPriceFeatureIndex, false);
                }
                return false;
            }).collect(toSet());

        } catch (Exception e) {
            logger.warn("floorPrice giveUp error!", e);
        }
        return resultGiveUpSet;
    }

    /**
     * 设置 本次 放弃结果比例
     *
     * @param orientationPackages
     * @param indexOrientationPackageMap
     * @param basePriceResultMap
     */
    private void setGiveUpProbsToLog(Set<OrientationPackage> orientationPackages, Map<OrientationPackage, FloorPriceFeatureIndex> indexOrientationPackageMap, Map<FloorPriceFeatureIndex, BasePriceResult> basePriceResultMap) {
        orientationPackages.forEach(orientationPackage -> {
            FloorPriceFeatureIndex floorPriceFeatureIndex = indexOrientationPackageMap.get(orientationPackage);
            if (floorPriceFeatureIndex != null) {
                BasePriceResult result = basePriceResultMap.get(floorPriceFeatureIndex);
                if (result != null) {
                    orientationPackage.setGiveUpProb5(result.getGiveUpProb5());
                    orientationPackage.setGiveUpProb10(result.getGiveUpProb10());
                }
            }
        });
    }

    private Long calculateFinalFee(Boolean cpa, Double factor, double cvr, Long clickFee, Long convertCost) {
        if (!cpa) {
            return clickFee;
        }

        Long finalFee = Math.round(convertCost * cvr * factor);
        // 设定保底值1分
        if (finalFee.equals(0L)) {
            finalFee = 1L;
        }
        return finalFee;
    }


    private double getRankScore(double ctrReconstructionFactor,
                                double cvrReconstructionFactor,
                                Long fee,
                                double tagWeight,
                                double qScore,
                                double discountRate) {

        return fee * qScore * tagWeight * discountRate * ctrReconstructionFactor * cvrReconstructionFactor;
    }


    private PacingResult pacing(Advert advert, OrientationPackage orientationPackage) {

        try {
            Integer cvrType = orientationPackage.getCvrType();

            // 时段信息
            TimeInfo timeInfo = new TimeInfo();
            timeInfo.setHourBudgetFee(orientationPackage.getHourlyBudgetFees());
            timeInfo.setHourBudgetExp(orientationPackage.getHourlyBudgetCounts());
            timeInfo.setPackageBudget(Optional.ofNullable(orientationPackage.getBudget()).map(Long::doubleValue).orElse(-1D));
            timeInfo.setHourCtr(orientationPackage.getHourlyCtr());
            timeInfo.setHourCvr(orientationPackage.getHourlyCvr());
            timeInfo.setHourClk(orientationPackage.getHourlyClick());
            timeInfo.setHourExp(orientationPackage.getHourlyLaunch());
            timeInfo.setHourFee(orientationPackage.getHourlyConsume());

            // 广告ctr数据
            StatInfo adCtrStatInfo = this.convert(advert, StatisticData::getCtr);

            // 广告cvr数据
            StatInfo adCvrStatInfo = this.convert(advert, statisticData -> statisticData.getCvr(cvrType));

            // 广告点击量数据
            StatInfo adClickStatInfo = this.convert(advert, statisticData -> statisticData.getChargeClickCount().doubleValue());

            // 配置包ctr数据
            StatInfo pkgCtrStatInfo = this.convert(orientationPackage, StatisticData::getCtr);

            // 配置包cvr数据
            StatInfo pkgCvrStatInfo = this.convert(orientationPackage, statisticData -> statisticData.getCvr(cvrType));

            // 配置包点击量数据
            StatInfo pkgClickStatInfo = this.convert(orientationPackage, statisticData -> statisticData.getChargeClickCount().doubleValue());


            AdvertStatisticMergeEntity tagStatisticData = orientationPackage.getTagStatisticData();
            StatInfo tagCvrStatInfo = this.getStatInfo(tagStatisticData);

            // ctr数据
            CtrInfo ctrInfo = new CtrInfo();
            ctrInfo.setCtr(orientationPackage.getCtr());
            ctrInfo.setAdCtrInfo(adCtrStatInfo);
            ctrInfo.setOrientCtrInfo(pkgCtrStatInfo);

            // cvr数据
            CvrInfo cvrInfo = new CvrInfo();
            cvrInfo.setCvr(orientationPackage.getCvr());
            cvrInfo.setAdCvrInfo(adCvrStatInfo);
            cvrInfo.setOrientCvrInfo(pkgCvrStatInfo);
            cvrInfo.setCompeterCvrInfo(tagCvrStatInfo);

            OrientInfo orientInfo = new OrientInfo();
            orientInfo.setAdvertId(advert.getId());
            orientInfo.setOrientId(orientationPackage.getId());
            orientInfo.setManagered(orientationPackage.isWeakTarget());
            orientInfo.setChargeType(orientationPackage.getChargeType());
            orientInfo.setTarget(orientationPackage.getConvertCost());
            orientInfo.setFee(orientationPackage.getFinalFee());
            orientInfo.setFactor(orientationPackage.getAdjustPriceFactor());
            orientInfo.setQuailityLevel(orientationPackage.getQualityLevel());

            OrientInfo.Type type = orientInfo.new Type();
            type.setChargeType(orientationPackage.getChargeType());
            int pid = 0;
            if (orientationPackage.getSmartShop()) {
                pid |= 0b100;
            }
            type.setPid(pid);
            type.setPackageType(orientationPackage.getUserBehavior());
            orientInfo.setType(type);


            AutoMatchInfo autoMatchInfo = new AutoMatchInfo();
            Optional<StatisticData> optionalStatDo = orientationPackage.getSmartShopStatisticData();
            Long convert = optionalStatDo.map(statisticData -> statisticData.getConvertCount(cvrType)).orElse(0L);
            Long cost = optionalStatDo.map(StatisticData::getTotalConsume).orElse(0L);
            Long launch = optionalStatDo.map(StatisticData::getLaunchCount).orElse(0L);
            Long click = optionalStatDo.map(StatisticData::getChargeClickCount).orElse(0L);
            autoMatchInfo.setConvertCost(orientationPackage.getConvertCost());
            autoMatchInfo.setConvert(convert);
            autoMatchInfo.setClick(click);
            autoMatchInfo.setCost(cost);
            autoMatchInfo.setLaunch(launch);
            orientInfo.setAutoMatchInfo(autoMatchInfo);
            Optional.ofNullable(orientationPackage.getTodayStatisticData())
                    .map(StatisticData::getTotalConsume)
                    .map(Long::doubleValue)
                    .ifPresent(orientInfo::setOrientCostG1d);
            boolean needGiveUp = new Pacing().pacing(orientInfo, cvrInfo, ctrInfo, adClickStatInfo, pkgClickStatInfo, timeInfo);
            PacingResult pacingResult = PacingResult.newBuilder().giveUp(needGiveUp).tag(0L).build();

            Optional.ofNullable(orientInfo.getAutoMatchInfo()).map(AutoMatchInfo::getEffectLog)
                    .map(AutoMatchInfo.EffectLog::getTag)
                    .ifPresent(pacingResult::setTag);

            return pacingResult;

        } catch (Exception e) {
            logger.warn("pacing error:{}", e);
            return PacingResult.DEFAULT;
        }

    }

    private StatInfo convert(StatisticDataGetter statisticDataGetter,
                             Function<StatisticData, Double> function) {

        StatInfo statInfo = new StatInfo();
        statInfo.setApp1d(function.apply(statisticDataGetter.getTodayAppStatisticData()));
        statInfo.setApp1h(function.apply(statisticDataGetter.getHourAppStatisticData()));
        statInfo.setApp7d(function.apply(statisticDataGetter.getRecently7DayAppStatisticData()));
        statInfo.setG1d(function.apply(statisticDataGetter.getTodayStatisticData()));
        statInfo.setG1h(function.apply(statisticDataGetter.getHourStatisticData()));
        statInfo.setG7d(function.apply(statisticDataGetter.getRecently7DayStatisticData()));
        return statInfo;

    }

    /**
     * 融合CTR
     * 综合CTR = （统计CTR*统计CTR权重 + 预估CTR*预估CTR权重）,注释：权重和为1
     * 统计CTR = 统计维度系数*统计维度CTR
     *
     * @param statCtr       统计维度CTR
     * @param statCtrWeight 统计CTR权重
     * @param preCtr        预估CTR
     * @param preCtrWeight  预估CTR权重
     */
    private Double getAdvertCtr(double statCtr,
                                double statCtrWeight,
                                double preCtr,
                                double preCtrWeight,
                                boolean newAddAdStatus) {
        double ret;
        if (preCtr < 0.0001) {
            ret = statCtr;
        } else {
            if (newAddAdStatus) {
                ret = 0.5 * preCtr + 0.5 * statCtr;
            } else {
                ret = (statCtrWeight * statCtr +
                        preCtrWeight * preCtr) / (statCtrWeight + preCtrWeight);
            }
        }
        return FormatUtil.doubleFormat(ret, 6);
    }


    /**
     * 融合CVR
     * 综合CVR = （统计CVR*统计CVR权重 + 预估CVR*预估CVR权重）,注释：权重和为1
     * 统计CVR = 统计维度系数*统计维度CVR
     *
     * @param statCvr       统计维度CVR
     * @param statCvrWeight 统计CVR权重
     * @param preCvr        预估CVR
     * @param preCvrWeight  预估CVR权重
     */
    private Double getAdvertCvr(double statCvr,
                                double statCvrWeight,
                                double preCvr,
                                double preCvrWeight,
                                boolean newAddAdStatus) {
        double ret;
        if (preCvr < 0.0001) {
            ret = statCvr;
        } else {
            if (newAddAdStatus) {
                ret = 0.5 * preCvr + 0.5 * statCvr;
            } else {
                ret = (statCvrWeight * statCvr +
                        preCvrWeight * preCvr) / (statCvrWeight + preCvrWeight);
                ret = Math.min(ret, statCvr * 1.5);
            }
        }
        return FormatUtil.doubleFormat(ret, 6);
    }

    private double getScore(Long finalFee, Integer chargeType, Double ctr, Double cvr, Boolean advertMultiDimScoreEffective) {

        double score = ctr;

        // 如果不为true（包括false和null）,则返回ctr
        if (!BooleanUtils.isTrue(advertMultiDimScoreEffective)) {
            return score;
        }

        try {
            Qscore qscore = new Qscore();
            QualityInfo qualityInfo = new QualityInfo();
            qualityInfo.setType(chargeType);
            qualityInfo.setCtr(ctr);
            qualityInfo.setCvr(cvr);
            qualityInfo.setTarget(finalFee.doubleValue());

            score = qscore.getQscore(qualityInfo);
        } catch (Exception e) {
            logger.warn("calculate qScore error :{}", e);
        }

        return score;

    }


    private StatInfo getStatInfo(AdvertStatisticMergeEntity advertStatisticMergeEntity) {
        StatInfo statInfo = new StatInfo();
        statInfo.setApp1h(advertStatisticMergeEntity.getAppCurrentlyHour());
        statInfo.setApp1d(advertStatisticMergeEntity.getAppCurrentlyDay());
        statInfo.setApp7d(advertStatisticMergeEntity.getAppRecently7Day());
        statInfo.setG1h(advertStatisticMergeEntity.getGlobalCurrentlyHour());
        statInfo.setG1d(advertStatisticMergeEntity.getGlobalCurrentlyDay());
        statInfo.setG7d(advertStatisticMergeEntity.getGlobalRecently7Day());
        return statInfo;
    }


    private LoadingCache<String, Double> arpuOptimizeFactorCache = CacheBuilder.newBuilder()
            .expireAfterWrite(10L, TimeUnit.MINUTES)
            .build(new CacheLoader<String, Double>() {
                @Override
                public Double load(String key) throws Exception {
                    return loadArpuOptimizeFactorCacheCache(key).orElse(1D);
                }

                // 如果加载不到新值 返回 老值
                @Override
                public ListenableFuture<Double> reload(String key, Double oldValue) throws Exception {
                    ListenableFutureTask<Double> task = ListenableFutureTask.create(() -> loadArpuOptimizeFactorCacheCache(key).orElse(oldValue));
                    executorService.submit(task);
                    return task;
                }
            });

    /**
     * 加载 算法 底价放弃 调整参数
     * @param key
     * @return
     */
    private Optional<Double> loadArpuOptimizeFactorCacheCache(String key) {
        try {
            DBTimeProfile.enter("loadArpuOptimizeFactorCacheCache");
            return  Optional.ofNullable(key)
                    .map(modelKey -> nezhaStringRedisTemplate.opsForValue().get(key))
                    .map(json -> JSON.parseObject(json, Double.class));
        } catch (Exception e) {
            logger.error("load loadArpuOptimizeFactorCacheCache,redis key:{}", key);
            throw new RecommendEngineException("loadArpuOptimizeFactorCacheCache exception", e);
        } finally {
            DBTimeProfile.release();
        }

    }

}
