package cn.com.duiba.oto.oto.service.api.remoteservice.board;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.oto.dto.oto.board.*;

import java.math.BigDecimal;
import java.util.List;

/**
 * 看板数据远程服务接口
 *
 * @author xiaotian
 * @date 2025-01-27
 */
@AdvancedFeignClient
public interface RemoteOtoBoardService {

    /**
     * 获取产说会活动数据（经营池数据）
     *
     * @param param 查询参数
     * @return 产说会活动数据列表
     */
    List<BoardMeetingDataDto> getMeetingData(BoardQueryParam param);

    /**
     * 获取客户池数据
     *
     * @param param 查询参数
     * @return 客户池数据列表
     */
    List<BoardCustomerPoolDto> getCustomerPoolData(BoardQueryParam param);

    /**
     * 获取成交客户数据
     *
     * @param param 查询参数
     * @return 成交客户数据列表
     */
    List<BoardDealDataDto> getDealData(BoardQueryParam param);

    /**
     * 获取收入业绩数据
     *
     * @param param 查询参数
     * @return 收入业绩数据列表
     */
    List<BoardIncomeDataDto> getIncomeData(BoardQueryParam param);

    /**
     * 计算KPI指标（10到3指标、3出1指标、高客种草率、种草高客成交率）
     *
     * @param param 查询参数
     * @return KPI指标数据
     */
    BoardKpiDto calculateKpi(BoardQueryParam param);

    /**
     * 计算客户总数（考虑脱落）
     * 公式: 月度新接客户 + 转介绍客户 + 缘故客户 - 脱落
     *
     * @param month   月份
     * @param sellerId 销售ID
     * @param dropOffs 脱落数量（需要外部提供或通过其他方式计算）
     * @return 客户总数
     */
    Integer calculateTotalClients(String month, Long sellerId, Integer dropOffs);

    /**
     * 计算已成交客户数
     * 公式: 加保客户数 - 高客成交数 - 转介绍成交数 - 缘故成交数
     *
     * @param month   月份
     * @param sellerId 销售ID
     * @return 已成交客户数
     */
    Integer calculateDealClients(String month, Long sellerId);

    /**
     * 获取累计业绩（年度完整自然月）
     *
     * @param year     年份，例如 2025
     * @param endMonth 结束月份，例如 2025-09
     * @param sellerId 销售ID（可选，传null则查询所有）
     * @return 累计业绩
     */
    BigDecimal getCumulativePerformance(Integer year, String endMonth, Long sellerId);

    /**
     * 获取累计收入（年度完整自然月）
     *
     * @param year     年份，例如 2025
     * @param endMonth 结束月份，例如 2025-09
     * @param sellerId 销售ID（可选，传null则查询所有）
     * @return 累计收入
     */
    BigDecimal getCumulativeIncome(Integer year, String endMonth, Long sellerId);

    /**
     * 批量保存或更新产说会活动数据
     *
     * @param dataList 数据列表
     * @return 是否成功
     */
    Boolean batchSaveOrUpdateMeetingData(List<BoardMeetingDataDto> dataList);

    /**
     * 批量保存或更新客户池数据
     *
     * @param dataList 数据列表
     * @return 是否成功
     */
    Boolean batchSaveOrUpdateCustomerPoolData(List<BoardCustomerPoolDto> dataList);

    /**
     * 批量保存或更新成交客户数据
     *
     * @param dataList 数据列表
     * @return 是否成功
     */
    Boolean batchSaveOrUpdateDealData(List<BoardDealDataDto> dataList);

    /**
     * 批量保存或更新收入业绩数据
     *
     * @param dataList 数据列表
     * @return 是否成功
     */
    Boolean batchSaveOrUpdateIncomeData(List<BoardIncomeDataDto> dataList);
}

