package cn.com.duiba.oto.util;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.TemporalUnit;
import java.util.Date;

/**
 * 功能描述：时间工具类
 *
 * @Auther: xiaotian
 * @Date: 2025/2/6 15:17
 * @Description:
 */
public class OtoDateUtil {
    // 常用时间格式常量
    public static final String PATTERN_DEFAULT = "yyyy-MM-dd HH:mm:ss";
    public static final String PATTERN_DATE = "yyyy-MM-dd";
    public static final String PATTERN_COMPACT = "yyyyMMddHHmmss";
    public static final String DAY_NUMBER = "yyyyMMdd";
    public static final String YEAR_DAY_NUMBER = "yyMMdd";
    public static final String YEAR = "yyyy";
    public static final String DAY_ONLY = "MM-dd";
    public static final String SECOND_ONLY = "HH:mm:ss";
    public static final String MINUTE = "yyyy-MM-dd HH:mm";
    public static final String MINUTE_ONLY = "HH:mm";
    public static final String MONTH_DAY_YEAR = "MM/dd/yyyy";
    public static final String MILLI_SECOND = "yyyyMMddHHmmssSSS";

    /**
     * 格式化时间为字符串（默认格式）
     */
    public static String format(LocalDateTime dateTime) {
        return format(dateTime, PATTERN_DEFAULT);
    }

    /**
     * 自定义格式格式化时间
     */
    public static String format(LocalDateTime dateTime, String pattern) {
        return dateTime.format(DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 解析字符串为LocalDateTime
     */
    public static LocalDateTime parse(String dateStr, String pattern) {
        return LocalDateTime.parse(dateStr, DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 解析字符串为LocalDate
     */
    public static LocalDate parseLocalDate(String dateStr, String pattern) {
        return LocalDate.parse(dateStr, DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 获取当前时间戳（毫秒）
     */
    public static long getCurrentTimeMillis() {
        return System.currentTimeMillis();
    }

    /**
     * 获取当前时间字符串（默认格式）
     */
    public static String getCurrentTimeString() {
        return format(LocalDateTime.now());
    }


    /**
     * Date 转 LocalDate
     */
    public static LocalDate dateToLocalDate(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }

    /**
     * LocalDate 转 Date
     */
    public static Date localDateToDate(LocalDate localDate) {
        return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    /**
     * Date转LocalDateTime
     */
    public static LocalDateTime dateToLocalDateTime(Date date) {
        return LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
    }

    /**
     * LocalDateTime转Date
     */
    public static Date localDateTimeToDate(LocalDateTime localDateTime) {
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 增加时间
     *
     * @param unit 时间单位（ChronoUnit.DAYS、ChronoUnit.HOURS等）
     */
    public static LocalDateTime plus(LocalDateTime dateTime, long amount, TemporalUnit unit) {
        return dateTime.plus(amount, unit);
    }

    /**
     * 减少时间
     */
    public static LocalDateTime minus(LocalDateTime dateTime, long amount, TemporalUnit unit) {
        return dateTime.minus(amount, unit);
    }

    /**
     * 计算两个时间间隔（按单位）
     */
    public static long between(LocalDateTime start, LocalDateTime end, TemporalUnit unit) {
        return unit.between(start, end);
    }

    /**
     * 是否在时间之前
     */
    public static boolean isBefore(LocalDateTime dateTime, LocalDateTime compareTo) {
        return dateTime.isBefore(compareTo);
    }

    /**
     * 是否在时间之后
     */
    public static boolean isAfter(LocalDateTime dateTime, LocalDateTime compareTo) {
        return dateTime.isAfter(compareTo);
    }

    /**
     * 获取当天的开始时间（00:00:00）
     */
    public static LocalDateTime getDayStart(LocalDateTime dateTime) {
        return dateTime.toLocalDate().atStartOfDay();
    }

    /**
     * 获取当天的结束时间（23:59:59）
     */
    public static LocalDateTime getDayEnd(LocalDateTime dateTime) {
        return dateTime.toLocalDate().atTime(LocalTime.MAX);
    }

    /**
     * 获取本月第一天
     */
    public static LocalDateTime getFirstDayOfMonth(LocalDateTime dateTime) {
        return dateTime.with(TemporalAdjusters.firstDayOfMonth());
    }

    /**
     * 获取本月最后一天
     */
    public static LocalDateTime getLastDayOfMonth(LocalDateTime dateTime) {
        return dateTime.with(TemporalAdjusters.lastDayOfMonth());
    }

    /**
     * 是否同一天
     */
    public static boolean isSameDay(LocalDateTime date1, LocalDateTime date2) {
        return date1.toLocalDate().equals(date2.toLocalDate());
    }

    /**
     * 转换时区
     */
    public static LocalDateTime convertZone(LocalDateTime dateTime, ZoneId fromZone, ZoneId toZone) {
        return dateTime.atZone(fromZone).withZoneSameInstant(toZone).toLocalDateTime();
    }

    /**
     * 计算年龄（根据生日）
     */
    public static int calculateAge(LocalDate birthDate) {
        return Period.between(birthDate, LocalDate.now()).getYears();
    }

    /**
     * 获取本周开始时间（周一）
     */
    public static LocalDateTime getStartOfWeek(LocalDateTime dateTime) {
        return dateTime.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY))
                .toLocalDate().atStartOfDay();
    }

    /**
     * 获取本周结束时间（周日）
     */
    public static LocalDateTime getEndOfWeek(LocalDateTime dateTime) {
        return dateTime.with(TemporalAdjusters.nextOrSame(DayOfWeek.SUNDAY))
                .toLocalDate().atTime(LocalTime.MAX);
    }
}
