package cn.com.duiba.pinellie.agent;

import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.com.duiba.pinellie.agent.support.MilliSecondTimer;
import cn.com.duiba.pinellie.agent.support.NetUtils;

/**
 * Created by xuezhaoming on 16/6/21.
 */
public final class PTransactions {

    private static final Logger LOG          = LoggerFactory.getLogger(PTransactions.class);

    private AtomicLong          atomicLong   = new AtomicLong();

    private static final int    DEFAULT_STEP = 60;

    private String              appName;

    private int                 step         = DEFAULT_STEP;

    private String              metricForCount;

    private String              metriForSumKey;

    private double              metriForSumValue;

    private boolean             isCount      = false;

    private long                startTime;
    private long                endTime;

    static {

        PContext.getInstance().init();
    }

    public static PTransactions newTransactions() {
        return new PTransactions();
    }

    /**
     * 应用名称
     * 
     * @param appName
     * @return
     */
    public PTransactions appName(String appName) {
        this.appName = appName;
        return this;
    }

    /**
     * 数据频率，默认是60
     * 
     * @param step
     * @return
     */
    public PTransactions step(int step) {
        this.step = step;
        return this;
    }

    public PTransactions build() {
        if (!PContext.getInstance().isInit() || !PContext.getInstance().getpConfig().isOpen()) {
            return this;
        }

        startTime = MilliSecondTimer.currentTimeMillis();
        return this;
    }

    /**
     * 技术类
     * 
     * @param data
     */
    public void logMetricForCount(String data) {
        this.metricForCount = data;
        isCount = true;
    }

    /**
     * 数值类
     * 
     * @param key 数值的key
     * @param value 具体的数值
     */
    public void logMetriForSum(String key, double value) {
        this.metriForSumKey = key;
        this.metriForSumValue = value;
    }

    /**
     * 完成
     */
    public void complete() {

        try {
            // 初始化失败 或是未开启服务
            if (!PContext.getInstance().isInit() || !PContext.getInstance().getpConfig().isOpen()) {
                return;
            }

            endTime = MilliSecondTimer.currentTimeMillis();

            appName = StringUtils.isBlank(appName) ? PContext.getInstance().getpConfig().getAppName() : "default";

            // 统计业务计数
            if (StringUtils.isNotBlank(metricForCount)) {
                sendCount();
            }

            // 统计指标类
            if (StringUtils.isNotBlank(metriForSumKey)) {
                sendMetriForSum();
            }

            // 统计消耗时间
            sendCost();

        } catch (Throwable e) {

            long andIncrement = atomicLong.getAndIncrement();
            if (andIncrement % 5000 == 0) {
                atomicLong.set(0);
                LOG.error("send pinellie error ", e);
            }
        }
    }

    private void sendMetriForSum() {

        PEvent pEvent = new PEvent();
        pEvent.setEndpoint(NetUtils.getHostNameLinux());
        pEvent.setMetric(metriForSumKey);
        pEvent.setValue(metriForSumValue);
        pEvent.setStep(step);
        pEvent.setTags(appName);
        pEvent.setTimestamp((int) (MilliSecondTimer.currentTimeMillis() / 1000));
        pEvent.setCounterType(PEvent.TYPE_GAUGE);

        PContext.getInstance().getpHandler().addQueue(pEvent);
    }

    private void sendCost() {

        PEvent pEvent = new PEvent();
        pEvent.setEndpoint(NetUtils.getHostNameLinux());
        String key = metriForSumKey != null ? metriForSumKey : metricForCount;
        pEvent.setMetric(key + "-cost");
        pEvent.setValue(endTime - startTime);
        pEvent.setStep(step);
        pEvent.setTags(appName);
        pEvent.setTimestamp((int) (MilliSecondTimer.currentTimeMillis() / 1000));
        pEvent.setCounterType(PEvent.TYPE_GAUGE);

        PContext.getInstance().getpHandler().addQueue(pEvent);

    }

    private void sendCount() {
        PEvent pEvent = new PEvent();
        pEvent.setEndpoint(NetUtils.getHostNameLinux());
        String metric = metricForCount;
        pEvent.setMetric(metric);
        double value = 1.0d;
        pEvent.setValue(value);
        pEvent.setStep(step);
        pEvent.setTags(appName);
        pEvent.setTimestamp((int) (MilliSecondTimer.currentTimeMillis() / 1000));
        pEvent.setCounterType(isCount ? PEvent.TYPE_COUNTER : PEvent.TYPE_GAUGE);

        PContext.getInstance().getpHandler().addQueue(pEvent);
    }
}
