package cn.com.duiba.pinellie.agent.http;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

public abstract class AbstractHttpAsyncClientPool {

	private CloseableHttpAsyncClient httpClient;

	public abstract CloseableHttpAsyncClient getHttpAsyncClient();

	public abstract int getAppMaxQueue();

	public abstract int getMaxQueue();

	private Map<String, AtomicInteger> runningStat = new ConcurrentHashMap<String, AtomicInteger>();

	public AbstractHttpAsyncClientPool() {
		httpClient = getHttpAsyncClient();
		httpClient.start();
	}

	public synchronized void submit(String queueKey, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
		AtomicInteger running = runningStat.get(queueKey);
		if (running == null) {
			runningStat.put(queueKey, new AtomicInteger(1));
		} else {
			running.incrementAndGet();
		}
		httpClient.execute(request, new CallbackProcesser(queueKey, request, callback));
	}

	/**
	 * 判断APP能否提交任务到执行队列中
	 */
	public boolean canSubmitToAppPool(String queueKey) {
		int queueCount = 0;
		AtomicInteger running = runningStat.get(queueKey);
		if (running != null) {
			queueCount = running.intValue();
		}
		if (queueCount < getAppMaxQueue()) {
			return true;
		}
		return false;
	}

	/**
	 * 队列详细信息
	 */
	public Map<String, Object> dumpDetail() {
		Map<String, Object> map = new HashMap<String, Object>();
		map.put("runningStat", runningStat);
		return map;
	}

	/**
	 * HTTP处理中统计信息
	 */
	public int dumpSize() {
		int running = 0;
		for (AtomicInteger a : runningStat.values()) {
			running += a.get();
		}
		return running;
	}

	/**
	 * 自定义回调
	 */
	private class CallbackProcesser implements FutureCallback<HttpResponse> {

		private FutureCallback<HttpResponse> callback;
		private String queueKey;

		public CallbackProcesser(String queueKey, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
			this.callback = callback;
			this.queueKey = queueKey;
		}

		public void completed(HttpResponse result) {
			try {
				callback.completed(result);
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

		public void failed(Exception ex) {
			try {
				callback.failed(ex);
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

		public void cancelled() {
			try {
				callback.cancelled();
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

	}

}
