package cn.com.duiba.projectx.sdk;

import cn.com.duiba.projectx.sdk.annotation.querytable.Ext1;
import cn.com.duiba.projectx.sdk.annotation.querytable.Ext2;
import cn.com.duiba.projectx.sdk.annotation.querytable.Ext3;
import cn.com.duiba.projectx.sdk.annotation.querytable.Ext4;
import cn.com.duiba.projectx.sdk.annotation.querytable.Id;
import cn.com.duiba.projectx.sdk.annotation.querytable.Score;
import cn.com.duiba.projectx.sdk.querytable.QueryWrapper;
import cn.com.duiba.projectx.sdk.querytable.SFunction;
import cn.com.duiba.projectx.sdk.utils.PageList;
import cn.com.duiba.projectx.sdk.utils.QueryTable;
import cn.com.duiba.projectx.sdk.utils.QueryTableSp;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;

import java.lang.invoke.SerializedLambda;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.Date;
import java.util.List;
import java.util.Map;

public class BaseDao<T> {
    public final Class<T> clazz;
    public final SFunction<T, ?> idFunction;
    public final SFunction<T, ?> ext1Function;
    public final SFunction<T, ?> ext2Function;
    public final SFunction<T, ?> ext3Function;
    public final SFunction<T, ?> ext4Function;
    public final SFunction<T, ?> scoreFunction;

    public BaseDao() { // NOSONAR
        try {
            Map<Class, SFunction<T, ?>> getterMethodMap = Maps.newHashMap();
            ParameterizedType superclass = (ParameterizedType) getClass().getGenericSuperclass();
            clazz = (Class<T>) superclass.getActualTypeArguments()[0];
            for (Field field : clazz.getDeclaredFields()) {
                String getterMethodName;
                Method getterMethod;
                try {
                    getterMethodName = "get" + Character.toUpperCase(field.getName().charAt(0)) + field.getName().substring(1);
                    getterMethod = clazz.getMethod(getterMethodName);
                } catch (Exception e) {
                    getterMethodName = "is" + Character.toUpperCase(field.getName().charAt(0)) + field.getName().substring(1);
                    getterMethod = clazz.getMethod(getterMethodName);
                }
                if(getterMethod == null) {
                    continue;
                }
                final Method finalGetterMethod = getterMethod;
                SFunction<T, ?> getFunction = new SFunction<T, Object>() {
                    @Override
                    public Object apply(T t) {
                        try {
                            return finalGetterMethod.invoke(t);
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    }

                    public SerializedLambda writeReplace() {
                        return new SerializedLambda(
                                clazz,
                                SFunction.class.getName(),
                                "apply",
                                null,
                                1,
                                clazz.getName(),
                                finalGetterMethod.getName(),
                                null,
                                null,
                                new Object[]{}
                        );
                    }
                };
                if (field.isAnnotationPresent(Id.class)) {
                    getterMethodMap.put(Id.class, getFunction);
                } else if(field.isAnnotationPresent(Ext1.class)) {
                    getterMethodMap.put(Ext1.class, getFunction);
                } else if(field.isAnnotationPresent(Ext2.class)) {
                    getterMethodMap.put(Ext2.class, getFunction);
                } else if(field.isAnnotationPresent(Ext3.class)) {
                    getterMethodMap.put(Ext3.class, getFunction);
                } else if(field.isAnnotationPresent(Ext4.class)) {
                    getterMethodMap.put(Ext4.class, getFunction);
                } else if(field.isAnnotationPresent(Score.class)) {
                    getterMethodMap.put(Score.class, getFunction);
                }
            }
            idFunction = getterMethodMap.get(Id.class);
            ext1Function = getterMethodMap.get(Ext1.class);
            ext2Function = getterMethodMap.get(Ext2.class);
            ext3Function = getterMethodMap.get(Ext3.class);
            ext4Function = getterMethodMap.get(Ext4.class);
            scoreFunction = getterMethodMap.get(Score.class);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public QueryTableSp<T> getQueryTableSp(CommonApi api) {
        return api.getQueryTableSp(clazz);
    }

    /**
     * 增加记录
     * @param api
     * @param record
     */
    public void addRecord(CommonApi api, T record) {
        getQueryTableSp(api).add(record);
    }

    /**
     * 根据id更新记录
     * @param api
     * @param record
     */
    public void updateById(CommonApi api, T record) {
        getQueryTableSp(api).updateById(record);
    }

    /**
     * 根据id查询单条记录
     * @param api
     * @param id
     * @return
     */
    public T queryOneById(CommonApi api, Object id) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(idFunction, id);
        return queryTableSp.queryOne(queryWrapper);
    }

    /**
     * 根据id列表查询记录列表
     * @param api
     * @param idList
     * @return
     */
    public List<T> queryListByIdList(CommonApi api, List<?> idList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(idFunction, idList);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 查询所有记录（一般用于定时任务中，用户请求中慎用，返回列表按照id升序排列）
     * @param function 没有时可以传null，不支持SubData字段
     * @param value 筛选条件的值
     * @return
     */
    public List<T> queryAll(CommonApi api, SFunction<T, ?> function, Object value) {
        return getQueryTableSp(api).queryAll(function, value);
    }

    /********************* ext1单字段相关 *********************/

    /**
     * 根据ext1值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @return
     */
    public T queryOneByExt1(CommonApi api, Object ext1Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @return
     */
    public T queryRecentOneByExt1(CommonApi api, Object ext1Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @return
     */
    public List<T> queryListByExt1(CommonApi api, Object ext1Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值查询记录总数
     * @param api
     * @param ext1Value
     * @return
     */
    public long queryCountByExt1(CommonApi api, Object ext1Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.pageQuery(queryWrapper, 1L, 1L).getTotalCount();
    }

    /**
     * 根据ext1值分页获取记录列表（按照id倒序）
     * @param api
     * @param ext1Value
     * @param pageSize 每页数量
     * @param pageNo 从1开始
     * @param lastId 可传null（建议非第1页传）
     * @return 《本页记录列表，是否还有更多》
     */
    public Pair<List<T>, Boolean> pageQueryByExt1(CommonApi api, Object ext1Value, int pageSize, int pageNo, Long lastId) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        PageList<T> pageList;
        if(lastId == null) {
            QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                    .eq(ext1Function, ext1Value)
                    .orderBy(idFunction, QueryTable.SortType.DESC);
            pageList = queryTableSp.pageQuery(queryWrapper, pageNo, pageSize);
        } else {
            pageList = queryTableSp.pageQueryFilterByLastId(ext1Function, ext1Value, lastId, QueryTable.SortType.DESC, pageSize);
        }
        return Pair.of(pageList.getList(), pageList.getTotalPage() > pageNo);
    }

    /**
     * 根据ext1值查询记录列表（按照id升序，取前n条）
     * @param api
     * @param ext1Value
     * @param limit
     * @return
     */
    public List<T> queryListByExt1WithLimit(CommonApi api, Object ext1Value, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1ValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt1ListWithLimit(CommonApi api, List<?> ext1ValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext1Function, ext1ValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1WithLimit(CommonApi api, Object ext1Value, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1ValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1ListWithLimit(CommonApi api, List<?> ext1ValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext1Function, ext1ValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1单字段相关 *********************/
    /********************* ext2单字段相关 *********************/

    /**
     * 根据ext2值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext2Value
     * @return
     */
    public T queryOneByExt2(CommonApi api, Object ext2Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext2Value
     * @return
     */
    public T queryRecentOneByExt2(CommonApi api, Object ext2Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext2Value
     * @return
     */
    public List<T> queryListByExt2(CommonApi api, Object ext2Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext2值查询记录总数
     * @param api
     * @param ext2Value
     * @return
     */
    public long queryCountByExt2(CommonApi api, Object ext2Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.pageQuery(queryWrapper, 1L, 1L).getTotalCount();
    }

    /**
     * 根据ext2值分页获取记录列表（按照id倒序）
     * @param api
     * @param ext2Value
     * @param pageSize 每页数量
     * @param pageNo 从1开始
     * @param lastId 可传null（建议非第1页传）
     * @return 《本页记录列表，是否还有更多》
     */
    public Pair<List<T>, Boolean> pageQueryByExt2(CommonApi api, Object ext2Value, int pageSize, int pageNo, Long lastId) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        PageList<T> pageList;
        if(lastId == null) {
            QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                    .eq(ext2Function, ext2Value)
                    .orderBy(idFunction, QueryTable.SortType.DESC);
            pageList = queryTableSp.pageQuery(queryWrapper, pageNo, pageSize);
        } else {
            pageList = queryTableSp.pageQueryFilterByLastId(ext2Function, ext2Value, lastId, QueryTable.SortType.DESC, pageSize);
        }
        return Pair.of(pageList.getList(), pageList.getTotalPage() > pageNo);
    }

    /**
     * 根据ext2值查询记录列表（按照id升序，取前n条）
     * @param api
     * @param ext2Value
     * @param limit
     * @return
     */
    public List<T> queryListByExt2WithLimit(CommonApi api, Object ext2Value, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext2ValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt2ListWithLimit(CommonApi api, List<?> ext2ValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext2Function, ext2ValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2Value
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2WithLimit(CommonApi api, Object ext2Value, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2ValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2ListWithLimit(CommonApi api, List<?> ext2ValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext2Function, ext2ValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }
    
    /********************* ext2单字段相关 *********************/
    /********************* ext3单字段相关 *********************/

    /**
     * 根据ext3值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext3Value
     * @return
     */
    public T queryOneByExt3(CommonApi api, Object ext3Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext3值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext3Value
     * @return
     */
    public T queryRecentOneByExt3(CommonApi api, Object ext3Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext3值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext3Value
     * @return
     */
    public List<T> queryListByExt3(CommonApi api, Object ext3Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext3值查询记录总数
     * @param api
     * @param ext3Value
     * @return
     */
    public long queryCountByExt3(CommonApi api, Object ext3Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.pageQuery(queryWrapper, 1L, 1L).getTotalCount();
    }

    /**
     * 根据ext3值分页获取记录列表（按照id倒序）
     * @param api
     * @param ext3Value
     * @param pageSize 每页数量
     * @param pageNo 从1开始
     * @param lastId 可传null（建议非第1页传）
     * @return 《本页记录列表，是否还有更多》
     */
    public Pair<List<T>, Boolean> pageQueryByExt3(CommonApi api, Object ext3Value, int pageSize, int pageNo, Long lastId) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        PageList<T> pageList;
        if(lastId == null) {
            QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                    .eq(ext3Function, ext3Value)
                    .orderBy(idFunction, QueryTable.SortType.DESC);
            pageList = queryTableSp.pageQuery(queryWrapper, pageNo, pageSize);
        } else {
            pageList = queryTableSp.pageQueryFilterByLastId(ext3Function, ext3Value, lastId, QueryTable.SortType.DESC, pageSize);
        }
        return Pair.of(pageList.getList(), pageList.getTotalPage() > pageNo);
    }

    /**
     * 根据ext3值查询记录列表（按照id升序，取前n条）
     * @param api
     * @param ext3Value
     * @param limit
     * @return
     */
    public List<T> queryListByExt3WithLimit(CommonApi api, Object ext3Value, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext3ValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt3ListWithLimit(CommonApi api, List<?> ext3ValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext3Function, ext3ValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext3Value
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt3WithLimit(CommonApi api, Object ext3Value, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext3ValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt3ListWithLimit(CommonApi api, List<?> ext3ValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext3Function, ext3ValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext3单字段相关 *********************/
    /********************* ext4单字段相关 *********************/

    /**
     * 根据ext4值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext4Value
     * @return
     */
    public T queryOneByExt4(CommonApi api, Object ext4Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext4值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext4Value
     * @return
     */
    public T queryRecentOneByExt4(CommonApi api, Object ext4Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext4值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext4Value
     * @return
     */
    public List<T> queryListByExt4(CommonApi api, Object ext4Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext4值查询记录总数
     * @param api
     * @param ext4Value
     * @return
     */
    public long queryCountByExt4(CommonApi api, Object ext4Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.pageQuery(queryWrapper, 1L, 1L).getTotalCount();
    }

    /**
     * 根据ext4值分页获取记录列表（按照id倒序）
     * @param api
     * @param ext4Value
     * @param pageSize 每页数量
     * @param pageNo 从1开始
     * @param lastId 可传null（建议非第1页传）
     * @return 《本页记录列表，是否还有更多》
     */
    public Pair<List<T>, Boolean> pageQueryByExt4(CommonApi api, Object ext4Value, int pageSize, int pageNo, Long lastId) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        PageList<T> pageList;
        if(lastId == null) {
            QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                    .eq(ext4Function, ext4Value)
                    .orderBy(idFunction, QueryTable.SortType.DESC);
            pageList = queryTableSp.pageQuery(queryWrapper, pageNo, pageSize);
        } else {
            pageList = queryTableSp.pageQueryFilterByLastId(ext4Function, ext4Value, lastId, QueryTable.SortType.DESC, pageSize);
        }
        return Pair.of(pageList.getList(), pageList.getTotalPage() > pageNo);
    }

    /**
     * 根据ext4值查询记录列表（按照id升序，取前n条）
     * @param api
     * @param ext4Value
     * @param limit
     * @return
     */
    public List<T> queryListByExt4WithLimit(CommonApi api, Object ext4Value, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext4值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext4ValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt4ListWithLimit(CommonApi api, List<?> ext4ValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext4Function, ext4ValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext4值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext4Value
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt4WithLimit(CommonApi api, Object ext4Value, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext4值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext4ValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt4ListWithLimit(CommonApi api, List<?> ext4ValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(ext4Function, ext4ValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext4单字段相关 *********************/
    /********************* score单字段相关 *********************/

    /**
     * 根据score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param scoreValue
     * @return
     */
    public T queryOneByScore(CommonApi api, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByScore(CommonApi api, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param scoreValue
     * @return
     */
    public List<T> queryListByScore(CommonApi api, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据score值查询记录总数
     * @param api
     * @param scoreValue
     * @return
     */
    public long queryCountByScore(CommonApi api, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.pageQuery(queryWrapper, 1L, 1L).getTotalCount();
    }

    /**
     * 根据score值分页获取记录列表（按照id倒序）
     * @param api
     * @param scoreValue
     * @param pageSize 每页数量
     * @param pageNo 从1开始
     * @param lastId 可传null（建议非第1页传）
     * @return 《本页记录列表，是否还有更多》
     */
    public Pair<List<T>, Boolean> pageQueryByScore(CommonApi api, Object scoreValue, int pageSize, int pageNo, Long lastId) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        PageList<T> pageList;
        if(lastId == null) {
            QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                    .eq(scoreFunction, scoreValue)
                    .orderBy(idFunction, QueryTable.SortType.DESC);
            pageList = queryTableSp.pageQuery(queryWrapper, pageNo, pageSize);
        } else {
            pageList = queryTableSp.pageQueryFilterByLastId(scoreFunction, scoreValue, lastId, QueryTable.SortType.DESC, pageSize);
        }
        return Pair.of(pageList.getList(), pageList.getTotalPage() > pageNo);
    }

    /**
     * 根据score值查询记录列表（按照id升序，取前n条）
     * @param api
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByScoreWithLimit(CommonApi api, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByScoreListWithLimit(CommonApi api, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByScoreWithLimit(CommonApi api, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByScoreListWithLimit(CommonApi api, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* score单字段相关 *********************/
    /********************* ext1 + ext2 相关 *********************/

    /**
     * 根据ext1值和ext2值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @return
     */
    public T queryOneByExt1AndExt2(CommonApi api, Object ext1Value, Object ext2Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext2值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @return
     */
    public T queryRecentOneByExt1AndExt2(CommonApi api, Object ext1Value, Object ext2Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext2值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @return
     */
    public List<T> queryListByExt1AndExt2(CommonApi api, Object ext1Value, Object ext2Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext2值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param limit
     * @return
     */
    public List<T> queryListByExt1AndExt2WithLimit(CommonApi api, Object ext1Value, Object ext2Value, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1AndExt2WithLimit(CommonApi api, Object ext1Value, Object ext2Value, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1 + ext2 相关 *********************/
    /********************* ext1 + ext3 相关 *********************/

    /**
     * 根据ext1值和ext3值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @return
     */
    public T queryOneByExt1AndExt3(CommonApi api, Object ext1Value, Object ext3Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext3值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @return
     */
    public T queryRecentOneByExt1AndExt3(CommonApi api, Object ext1Value, Object ext3Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext3值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @return
     */
    public List<T> queryListByExt1AndExt3(CommonApi api, Object ext1Value, Object ext3Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext3值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param limit
     * @return
     */
    public List<T> queryListByExt1AndExt3WithLimit(CommonApi api, Object ext1Value, Object ext3Value, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1AndExt3WithLimit(CommonApi api, Object ext1Value, Object ext3Value, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1 + ext3 相关 *********************/
    /********************* ext1 + ext4 相关 *********************/

    /**
     * 根据ext1值和ext4值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @return
     */
    public T queryOneByExt1AndExt4(CommonApi api, Object ext1Value, Object ext4Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext4值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @return
     */
    public T queryRecentOneByExt1AndExt4(CommonApi api, Object ext1Value, Object ext4Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext4值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @return
     */
    public List<T> queryListByExt1AndExt4(CommonApi api, Object ext1Value, Object ext4Value) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext4值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param limit
     * @return
     */
    public List<T> queryListByExt1AndExt4WithLimit(CommonApi api, Object ext1Value, Object ext4Value, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1AndExt4WithLimit(CommonApi api, Object ext1Value, Object ext4Value, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1 + ext4 相关 *********************/
    /********************* ext1 + score 相关 *********************/

    /**
     * 根据ext1值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt1AndScore(CommonApi api, Object ext1Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt1AndScore(CommonApi api, Object ext1Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt1AndScore(CommonApi api, Object ext1Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt1AndScoreWithLimit(CommonApi api, Object ext1Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1AndScoreWithLimit(CommonApi api, Object ext1Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt1AndScoreList(CommonApi api, Object ext1Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt1AndScoreListWithLimit(CommonApi api, Object ext1Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1AndScoreListWithLimit(CommonApi api, Object ext1Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1 + score 相关 *********************/
    /********************* ext2 + score 相关 *********************/

    /**
     * 根据ext2值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext2Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt2AndScore(CommonApi api, Object ext2Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext2Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt2AndScore(CommonApi api, Object ext2Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext2Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt2AndScore(CommonApi api, Object ext2Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext2值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext2Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt2AndScoreWithLimit(CommonApi api, Object ext2Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2AndScoreWithLimit(CommonApi api, Object ext2Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext2Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt2AndScoreList(CommonApi api, Object ext2Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext2值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext2Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt2AndScoreListWithLimit(CommonApi api, Object ext2Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2AndScoreListWithLimit(CommonApi api, Object ext2Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext2 + score 相关 *********************/
    /********************* ext3 + score 相关 *********************/

    /**
     * 根据ext3值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt3AndScore(CommonApi api, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext3值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt3AndScore(CommonApi api, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext3值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt3AndScore(CommonApi api, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext3值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext3Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt3AndScoreWithLimit(CommonApi api, Object ext3Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext3Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt3AndScoreWithLimit(CommonApi api, Object ext3Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext3Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt3AndScoreList(CommonApi api, Object ext3Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext3值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext3Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt3AndScoreListWithLimit(CommonApi api, Object ext3Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext3Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt3AndScoreListWithLimit(CommonApi api, Object ext3Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext3 + score 相关 *********************/
    /********************* ext4 + score 相关 *********************/

    /**
     * 根据ext4值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt4AndScore(CommonApi api, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext4值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt4AndScore(CommonApi api, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext4值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt4AndScore(CommonApi api, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext4值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext4Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt4AndScoreWithLimit(CommonApi api, Object ext4Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext4值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext4Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt4AndScoreWithLimit(CommonApi api, Object ext4Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext4值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext4Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt4AndScoreList(CommonApi api, Object ext4Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext4值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext4Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt4AndScoreListWithLimit(CommonApi api, Object ext4Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext4值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext4Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt4AndScoreListWithLimit(CommonApi api, Object ext4Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext4 + score 相关 *********************/
    /********************* ext1 + ext2 + score 相关 *********************/

    /**
     * 根据ext1值和ext2值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt1Ext2AndScore(CommonApi api, Object ext1Value, Object ext2Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext2值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt1Ext2AndScore(CommonApi api, Object ext1Value, Object ext2Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext2值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt1Ext2AndScore(CommonApi api, Object ext1Value, Object ext2Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext2值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt1Ext2AndScoreWithLimit(CommonApi api, Object ext1Value, Object ext2Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext2值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1Ext2AndScoreWithLimit(CommonApi api, Object ext1Value, Object ext2Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext2值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt1Ext2AndScoreList(CommonApi api, Object ext1Value, Object ext2Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext2值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt1Ext2AndScoreListWithLimit(CommonApi api, Object ext1Value, Object ext2Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext2值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext2Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1Ext2AndScoreListWithLimit(CommonApi api, Object ext1Value, Object ext2Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext2Function, ext2Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1 + ext2 + score 相关 *********************/
    /********************* ext1 + ext3 + score 相关 *********************/

    /**
     * 根据ext1值和ext3值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt1Ext3AndScore(CommonApi api, Object ext1Value, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext3值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt1Ext3AndScore(CommonApi api, Object ext1Value, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext3值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt1Ext3AndScore(CommonApi api, Object ext1Value, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext3值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt1Ext3AndScoreWithLimit(CommonApi api, Object ext1Value, Object ext3Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext3值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1Ext3AndScoreWithLimit(CommonApi api, Object ext1Value, Object ext3Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext3值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt1Ext3AndScoreList(CommonApi api, Object ext1Value, Object ext3Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext3值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt1Ext3AndScoreListWithLimit(CommonApi api, Object ext1Value, Object ext3Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext3值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext3Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1Ext3AndScoreListWithLimit(CommonApi api, Object ext1Value, Object ext3Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1 + ext3 + score 相关 *********************/
    /********************* ext1 + ext4 + score 相关 *********************/

    /**
     * 根据ext1值和ext4值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt1Ext4AndScore(CommonApi api, Object ext1Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext4值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt1Ext4AndScore(CommonApi api, Object ext1Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext1值和ext4值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt1Ext4AndScore(CommonApi api, Object ext1Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext4值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt1Ext4AndScoreWithLimit(CommonApi api, Object ext1Value, Object ext4Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext4值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1Ext4AndScoreWithLimit(CommonApi api, Object ext1Value, Object ext4Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext4值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt1Ext4AndScoreList(CommonApi api, Object ext1Value, Object ext4Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext1值和ext4值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt1Ext4AndScoreListWithLimit(CommonApi api, Object ext1Value, Object ext4Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext1值和ext4值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext1Value
     * @param ext4Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt1Ext4AndScoreListWithLimit(CommonApi api, Object ext1Value, Object ext4Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext1Function, ext1Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext1 + ext4 + score 相关 *********************/
    /********************* ext2 + ext3 + score 相关 *********************/

    /**
     * 根据ext2值和ext3值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt2Ext3AndScore(CommonApi api, Object ext2Value, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值和ext3值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt2Ext3AndScore(CommonApi api, Object ext2Value, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值和ext3值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt2Ext3AndScore(CommonApi api, Object ext2Value, Object ext3Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext2值和ext3值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt2Ext3AndScoreWithLimit(CommonApi api, Object ext2Value, Object ext3Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和ext3值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2Ext3AndScoreWithLimit(CommonApi api, Object ext2Value, Object ext3Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和ext3值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt2Ext3AndScoreList(CommonApi api, Object ext2Value, Object ext3Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext2值和ext3值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt2Ext3AndScoreListWithLimit(CommonApi api, Object ext2Value, Object ext3Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和ext3值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2Value
     * @param ext3Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2Ext3AndScoreListWithLimit(CommonApi api, Object ext2Value, Object ext3Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext3Function, ext3Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext2 + ext3 + score 相关 *********************/
    /********************* ext2 + ext4 + score 相关 *********************/

    /**
     * 根据ext2值和ext4值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt2Ext4AndScore(CommonApi api, Object ext2Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值和ext4值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt2Ext4AndScore(CommonApi api, Object ext2Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext2值和ext4值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt2Ext4AndScore(CommonApi api, Object ext2Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext2值和ext4值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt2Ext4AndScoreWithLimit(CommonApi api, Object ext2Value, Object ext4Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和ext4值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2Ext4AndScoreWithLimit(CommonApi api, Object ext2Value, Object ext4Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和ext4值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt2Ext4AndScoreList(CommonApi api, Object ext2Value, Object ext4Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext2值和ext4值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt2Ext4AndScoreListWithLimit(CommonApi api, Object ext2Value, Object ext4Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext2值和ext4值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext2Value
     * @param ext4Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt2Ext4AndScoreListWithLimit(CommonApi api, Object ext2Value, Object ext4Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext2Function, ext2Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext2 + ext4 + score 相关 *********************/
    /********************* ext3 + ext4 + score 相关 *********************/

    /**
     * 根据ext3值和ext4值和score值查询单条记录（按照id升序，取最早的）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryOneByExt3Ext4AndScore(CommonApi api, Object ext3Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext3值和ext4值和score值查询单条记录（按照id降序，取最新的）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public T queryRecentOneByExt3Ext4AndScore(CommonApi api, Object ext3Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        List<T> recordList = queryTableSp.queryListLimit(queryWrapper, 1);
        return CollectionUtils.isNotEmpty(recordList) ? recordList.get(0) : null;
    }

    /**
     * 根据ext3值和ext4值和score值查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValue
     * @return
     */
    public List<T> queryListByExt3Ext4AndScore(CommonApi api, Object ext3Value, Object ext4Value, Object scoreValue) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext3值和ext4值和score值查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValue
     * @param limit
     * @return
     */
    public List<T> queryListByExt3Ext4AndScoreWithLimit(CommonApi api, Object ext3Value, Object ext4Value, Object scoreValue, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值和ext4值和score值查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValue
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt3Ext4AndScoreWithLimit(CommonApi api, Object ext3Value, Object ext4Value, Object scoreValue, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .eq(scoreFunction, scoreValue)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值和ext4值和score值列表查询记录列表（按照id升序，最多2000条）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValueList
     * @return
     */
    public List<T> queryListByExt3Ext4AndScoreList(CommonApi api, Object ext3Value, Object ext4Value, List<?> scoreValueList) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit2000(queryWrapper);
    }

    /**
     * 根据ext3值和ext4值和score值列表查询记录列表（按照id升序，最多n条）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValueList
     * @param limit
     * @return
     */
    public List<T> queryListByExt3Ext4AndScoreListWithLimit(CommonApi api, Object ext3Value, Object ext4Value, List<?> scoreValueList, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .orderBy(idFunction, QueryTable.SortType.ASC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /**
     * 根据ext3值和ext4值和score值列表查询最近n条记录（按照id降序，取最近n条）
     * @param api
     * @param ext3Value
     * @param ext4Value
     * @param scoreValueList
     * @param gmtCreateStartTime 不限制则传null
     * @param limit
     * @return
     */
    public List<T> queryRecentListByExt3Ext4AndScoreListWithLimit(CommonApi api, Object ext3Value, Object ext4Value, List<?> scoreValueList, Date gmtCreateStartTime, int limit) {
        QueryTableSp<T> queryTableSp = getQueryTableSp(api);
        QueryWrapper<T> queryWrapper = queryTableSp.getQueryWrapper()
                .eq(ext3Function, ext3Value)
                .eq(ext4Function, ext4Value)
                .in(scoreFunction, scoreValueList)
                .gmtCreateTimeStartBy(gmtCreateStartTime)
                .orderBy(idFunction, QueryTable.SortType.DESC);
        return queryTableSp.queryListLimit(queryWrapper, limit);
    }

    /********************* ext3 + ext4 + score 相关 *********************/

}
