package cn.com.duiba.quanyi.center.api.log.operate;

import cn.com.duiba.quanyi.center.api.utils.QuanYiStringUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.core.ParameterNameDiscoverer;
import org.springframework.validation.BindingResult;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.Objects;

/**
 * @author lizhi
 * @date 2023/5/15 7:04 PM
 */
@Slf4j
public class LogAopUtil {
    
    private LogAopUtil() {}

    /**
     * 获取请求路径
     * @param request http请求
     * @return 路径
     */
    public static String getPath(HttpServletRequest request) {
        if (request == null) {
            return "";
        }
        return clearRequestUri(request.getRequestURI());
    }

    /**
     * 将requestURI中的//替换为/
     * @param requestUri 请求uri
     * @return 请求path
     */
    public static String clearRequestUri(String requestUri){
        if(StringUtils.isBlank(requestUri)){
            return requestUri;
        }
        return StringUtils.replace(requestUri, "//", "/");
    }

    public static String getRequestBody(ProceedingJoinPoint joinPoint, int maxLength, int valueMaxLength) {
        JSONObject paramObj = getFieldsName(joinPoint, valueMaxLength);
        return QuanYiStringUtils.subStringIfAbsent(paramObj.toJSONString(), maxLength);
    }
    
    public static Method getMethod(ProceedingJoinPoint joinPoint) {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        return signature.getMethod();
    }

    public static JSONObject getFieldsName(ProceedingJoinPoint joinPoint, int valueMaxLength) {
        // 参数值
        Object[] args = joinPoint.getArgs();
        ParameterNameDiscoverer pnd = new DefaultParameterNameDiscoverer();
        Method method = getMethod(joinPoint);
        String[] parameterNames = pnd.getParameterNames(method);
        JSONObject paramObj = new JSONObject();
        if (parameterNames == null || parameterNames.length < 1) {
            return paramObj;
        }
        for (int i = 0; i < Objects.requireNonNull(parameterNames).length; i++) {
            Object ob = args[i];
            if (ob instanceof HttpServletRequest || ob instanceof HttpServletResponse || ob instanceof BindingResult || ob instanceof MultipartFile){
                continue;
            }
            paramObj.put(parameterNames[i], toJsonStringAndSubIfAbsent(ob, valueMaxLength));
        }
        return paramObj;
    }
    
    public static String getResponseBody(Object obj, int maxLength, int valueMaxLength){
        if (obj == null) {
            return null;
        }
        if (obj instanceof String){
            return toJsonStringAndSubIfAbsent(obj, maxLength);
        }
        try {
            Object jsonObj = JSON.toJSON(obj);
            if (jsonObj instanceof JSONObject) {
                return subValueIfAbsent((JSONObject)jsonObj, maxLength, valueMaxLength);
            }
            return toJsonStringAndSubIfAbsent(obj, maxLength);
        } catch (Exception e) {
            log.info("access log aop warn obj={}", obj, e);
            return toJsonStringAndSubIfAbsent(obj, maxLength);
        }
    }
    
    private static String subValueIfAbsent(JSONObject jsonObj, int maxLength, int valueMaxLength) {
        jsonObj.replaceAll((k, v) -> toJsonStringAndSubIfAbsent(v, valueMaxLength));
        return toJsonStringAndSubIfAbsent(jsonObj, maxLength);
    }

    private static String toJsonStringAndSubIfAbsent(Object obj, int maxLength) {
        if (obj == null) {
            return null;
        }
        if (obj instanceof String) {
            return QuanYiStringUtils.subStringAndAddLog((String)obj, maxLength);
        }
        try {
            return QuanYiStringUtils.subStringAndAddLog(JSON.toJSONString(obj), maxLength);
        } catch (Exception e) {
            log.info("access log aop warn obj={}", obj, e);
            return QuanYiStringUtils.subStringAndAddLog(obj.toString(), maxLength);
        }
    }
}
