package cn.com.duiba.quanyi.center.api.log.operate.log;

import cn.com.duiba.quanyi.center.api.dto.common.OperateLogDetailDto;
import cn.com.duiba.quanyi.center.api.log.operate.LogAopUtil;
import cn.com.duiba.quanyi.center.api.log.operate.annotation.OperateLogMethod;
import cn.com.duiba.quanyi.center.api.remoteservice.RemoteOperateLogService;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.HttpOutputMessage;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;

/**
 * 访问日志
 * @author lizhi
 * @date 2023/5/9 11:56 AM
 */
@Aspect
@Slf4j
@Component
public class OperateLogAop {

    @Resource
    private RemoteOperateLogService operateLogService;
    @Resource(name= "logExecutorService")
    private ExecutorService logExecutorService;

    @Pointcut("@annotation(cn.com.duiba.quanyi.center.api.log.operate.annotation.OperateLogMethod)")
    public void logger(){
        //定义日志的切面
    }

    @Around("logger()")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
        HttpServletRequest request = ((ServletRequestAttributes) Objects.requireNonNull(RequestContextHolder.getRequestAttributes())).getRequest();
        Object result = null;
        try {
            OperateLogMethod annotation = getAnnotation(joinPoint);
            if (annotation != null) {
                OperateLogger.init();
            }
            result = joinPoint.proceed();
            return result;
        } catch (Exception ex){
            result = ex.getMessage();
            throw ex;
        } finally {
            try {
                operateLog(request, joinPoint, result);
            }catch (Exception e){
                log.warn("",e);
            } finally {
                OperateLogger.clearThreadLocal();
            }
        }
    }

    private OperateLogMethod getAnnotation(ProceedingJoinPoint joinPoint) {
        if (joinPoint == null) {
            return null;
        }
        Method method = LogAopUtil.getMethod(joinPoint);
        if (method == null) {
            return null;
        }
        return method.getAnnotation(OperateLogMethod.class);
    }

    private void operateLog(HttpServletRequest request, ProceedingJoinPoint joinPoint, Object result){
        OperateLogMethod annotation = getAnnotation(joinPoint);
        if (annotation == null) {
            return;
        }
        logExecutorService.submit(() -> {
            // 异步写入数据库
            OperateLogDetailDto dto = new OperateLogDetailDto();
            dto.setGroupName(annotation.group());
            dto.setOperateTypeName(annotation.type());
            fillOperatorData(dto);
            dto.setOperatePath(LogAopUtil.getPath(request));
            dto.setParamInfo(LogAopUtil.getRequestBody(joinPoint, 10000, 1000));
            dto.setResultInfo(LogAopUtil.getResponseBody(result, 10000, 1000));
            operateLogService.insert(dto);
        });
    }
    
    private void fillOperatorData(OperateLogDetailDto dto) {
        try {
            dto.setOperatorId(OperateLogger.getOperateLoggerStackContext().getOperatorId());
            dto.setOperatorName(OperateLogger.getOperateLoggerStackContext().getOperatorName());
            dto.setBizId(OperateLogger.getOperateLoggerStackContext().getBizId());
        } catch (Exception e) {
            log.warn("[操作日志], 获取业务Id时发生异常, ", e);
        }
    }
}
