package cn.com.duiba.quanyi.center.api.utils.idmaker;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.math.NumberUtils;

import java.util.Objects;

/**
 * Created by dugq on 2019-05-05.
 */
@Slf4j
public class IdMakerUtil {

    private IdMakerUtil() {
    }

    /**
     * 用base64 的方式编码加密ID
     * 详细编码请看 IdEncodeByBase64.class
     * @param id
     * @return
     */
    public static String encodingIdByBase64(Long id) {
        if (Objects.isNull(id)) {
            return null;
        }
        return IdEncodeByBase64.encodingId(id, IdConvertBase64Enum.DEF);
    }

    /**
     * 用base64 的方式编码加密ID
     * 详细编码请看 IdEncodeByBase64.class
     * @param id
     * @return
     */
    public static String encodingIdByBase64(Long id, IdConvertBase64Enum base64Enum) {
        if (Objects.isNull(id) || base64Enum == null) {
            return null;
        }
        return IdEncodeByBase64.encodingId(id, base64Enum);
    }

    /**
     * 解密base64
     * @param encode
     * @return 失败返回null
     */
    public static Long decodingIdByBase64(String encode) {
        return IdEncodeByBase64.decodingId(encode);
    }

    /**
     * 通过计算的方式把ID加密为纯数字，固定位数
     * @param id
     * @param digit  为null时 默认为10
     * @return
     */
    public static Long encodingId2Number(Long id, Integer digit){
        if(Objects.isNull(digit)){
            digit = 10;
        }
        return IdEncodeByNumber.encode(id,digit);
    }

    /**
     * 解密纯数字形式的加密ID
     * @param encode
     * @return 失败返回null
     */
    public static Long decodingIdByNumber(Long encode){
        try {
            return IdEncodeByNumber.decode(encode);
        }catch (Exception e){
            log.info("解密失败",e);
            return null;
        }

    }

    /**
     * 解密纯数字形式的加密ID
     * @param encode
     * @param digit  位数强制要求，非固定位数返回null
     * @return  解密失败返回null
     */
    public static Long decodingIdByNumber(Long encode, Integer digit){
        if(Objects.isNull(digit) || digit<9 || digit!=String.valueOf(encode).length()){
            return null;
        }
        return decodingIdByNumber(encode);
    }

    /**
     * 混合式解密，给个字符串，尝试多种方法解密
     * @param encode
     * @return  失败返回null
     */
    public static Long decodingId(String encode){
        if(NumberUtils.isNumber(encode) && encode.length()>9){
            return decodingIdByNumber(Long.valueOf(encode));
        }else{
            return decodingIdByBase64(encode);
        }
    }

    public static Long compatibleDecodingId(String encode,Long compatibleId){
        Long id = decodingId(encode);
        if(Objects.isNull(id) && NumberUtils.isNumber(encode)){
            id = Long.valueOf(encode);
            if(Objects.nonNull(compatibleId) && id > compatibleId){
                return null;
            }
        }
        return id;
    }

    /**
     * 兼容式解密。只针对于base64加密。不能针对纯数字加密，因为数字会出现无法判断它是加密的还是没有加密的
     * 部分接口，比如分享，ID必须兼容以前的版本。
     * 如果encode 是顺数字，并且小于 compatibleId ，则直接返回 encode 本身
     * @param encode
     * @param compatibleId 兼容的最大ID。 -1 无限兼容
     * @return
     */
    public static Long compatibleDecodingIdByBase64(String encode, Long compatibleId) {
        if (NumberUtils.isNumber(encode)) {
            Long id = Long.valueOf(encode);
            if (id < compatibleId || compatibleId == -1L) {
                return id;
            } else {
                return null;
            }
        } else {
            return decodingId(encode);
        }
    }

}
