package cn.com.duiba.quanyi.center.api.enums.coupon;

import cn.com.duiba.quanyi.center.api.enums.qyrobot.QyRobotWxStockTypeEnum;
import lombok.AllArgsConstructor;
import lombok.Getter;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author lizhi
 * @date 2023/9/1 11:42
 */
@Getter
@AllArgsConstructor
public enum WxStockTypeEnum {
    
    COUPON_BANK(2, "银行立减金", true),
    FIRST_BIND(3, "首绑有礼", false),
    MONTHLY_CARD_SWIPING(4, "银行月月刷", false),
    CASHIER_DIRECT_CUT(6, "收银台定向立减", false),
    COUPON_FULL(7, "代金券-全场券", true),
    COUPON_SINGLE(8, "代金券-单品券", true),
    CASHIER_CUT_FULL_CUT(9, "收银台立减-全场立减", false),
    CASHIER_CUT_FULL_DISCOUNT(10, "收银台立减-全场折扣", false),
    CASHIER_CUT_SINGLE_CUT(11, "收银台立减-单品立减", false),
    ;

    /**
     * 批次类型
     */
    private final Integer type;

    /**
     * 批次类型名称
     */
    private final String typeName;

    /**
     * 是否支持API发放
     */
    private final boolean supportApi;
    
    private static final Map<Integer, WxStockTypeEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(values()).collect(Collectors.toMap(WxStockTypeEnum::getType, Function.identity(), (v1, v2) -> v2)));
    
    /**
     * 根据枚举值获取枚举
     * @param type 枚举值
     * @return 枚举
     */
    public static WxStockTypeEnum getByType(Integer type) {
        if (type == null) {
            return null;
        }
        return ENUM_MAP.get(type);
    }

    /**
     * 根据枚举值获取枚举
     * @param type 枚举值
     * @return 枚举
     */
    public static String getNameByType(Integer type) {
        WxStockTypeEnum typeEnum = getByType(type);
        if (typeEnum == null) {
            return "未知";
        }
        return typeEnum.getTypeName();
    }

    /**
     * 是否支持api发放
     * @param type 枚举值
     * @return true-是，false-否
     */
    public static boolean isSupportApi(Integer type) {
        WxStockTypeEnum typeEnum = getByType(type);
        if (typeEnum == null) {
            return false;
        }
        return typeEnum.isSupportApi();
    }
    
    public static List<Integer> getSupportApiList() {
        return Arrays.stream(values()).filter(WxStockTypeEnum::isSupportApi).map(WxStockTypeEnum::getType).collect(Collectors.toList());
    }

    /**
     * 转换批次类型
     *
     * @param stockType 批次类型
     * @return 权益机器人批次类型 {@link QyRobotWxStockTypeEnum}
     */
    public static QyRobotWxStockTypeEnum convertStockType(Integer stockType) {
        switch (getByType(stockType)) {
            case FIRST_BIND:
            case MONTHLY_CARD_SWIPING:
            case CASHIER_DIRECT_CUT:
            case CASHIER_CUT_FULL_CUT:
            case CASHIER_CUT_FULL_DISCOUNT:
            case CASHIER_CUT_SINGLE_CUT:
                return QyRobotWxStockTypeEnum.DISCOUNT;
            case COUPON_BANK:
            case COUPON_FULL:
            case COUPON_SINGLE:
                return QyRobotWxStockTypeEnum.COUPON;
            default:
                throw new RuntimeException("stockType not found!");
        }
    }
}
