package cn.com.duiba.quanyi.center.api.utils.alipay;

import cn.com.duiba.quanyi.center.api.dto.coupon.autocreateext.alipay.AlipayActivityAutoCreateExtInfoDto;
import cn.com.duiba.quanyi.center.api.dto.coupon.autocreateext.alipay.CardPayCutAutoCreateExtInfoDto;
import cn.com.duiba.quanyi.center.api.enums.coupon.autocreate.alipay.GamePlaySingleTypeEnum;
import cn.com.duiba.quanyi.center.api.utils.PriceUtils;
import cn.com.duiba.quanyi.center.api.utils.StringUtil;
import cn.com.duiba.wolf.utils.DateUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author xuzhigang
 * @date 2024/7/30 10:49
 **/
@Slf4j
@Component
public class AlipayActivityAutoCreateUtils {
    private static final String JOINER = "-";

    private static final long FIVE_THOUSANDS_CENT = 5000;

    private static final int ALARM_APP_ID_LIMIT = 10;

    private static final int ALARM_PHONE_LIMIT = 3;

    /**
     * 根据活动预算计算实际金额
     * 公式：活动预算/活动周期 * n（创建时预留n天的预算），且金额必须是面额的整数倍
     *
     * @param budgetAmount          活动总预算
     * @param couponAmount          面额
     * @param startDate             活动开始时间
     * @param endDate               活动结束时间
     * @param maxLimitAmount        最大上限金额（每日发放上限或者用户领取上限的较大值）
     * @param autoAddMoney          true-自动加款 false-不自动加款
     * @return 实际金额（分）
     */
    public static Long calTotalAmount(Long budgetAmount, Long couponAmount, String startDate, String endDate, Long maxLimitAmount, boolean autoAddMoney) {
        // 不自动加款 直接取活动预算
        if (!autoAddMoney) {
            return budgetAmount;
        }
        // 计算初始化需要消耗的金额
        long initAmount = (budgetAmount / calDiffDays(startDate, endDate)) * 10;
        // 加款金额必须是面额的整数倍 取余向上取整
        initAmount = initAmount + (couponAmount - (initAmount % couponAmount));
        // 初始化预算与5倍面额取最大值，不能超过总预算
        if (maxLimitAmount == null) {
            return Math.min(Math.max(initAmount, 5 * couponAmount), budgetAmount);
        }
        // max(单日领取上限数量,用户领取上限) 与 初始化预算取最大值，不能超过总预算
        return Math.min(Math.max(Math.max(initAmount, 5 * couponAmount), maxLimitAmount), budgetAmount);
    }

    private static Integer calDiffDays(String startDate, String endDate) {
        return DateUtils.daysBetween(DateUtils.getDayDate(startDate), DateUtils.getDayDate(endDate)) + 1;
    }

    /**
     * 银行名称-银行卡类型（1-借记卡 2-信用卡）
     * 格式：xx银行-1,xx银行-2，多个银行英文逗号分割
     */
    public static List<AlipayActivityAutoCreateExtInfoDto.Bank> parseBankList(String bankListStr) {
        if (StringUtils.isBlank(bankListStr)) {
            return Collections.emptyList();
        }
        List<String> bankStrList = StringUtil.splitStringWithComma(bankListStr);
        return bankStrList.stream().map(AlipayActivityAutoCreateUtils::parseBank).filter(Objects::nonNull).collect(Collectors.toList());
    }

    /**
     * 银行名称-银行卡类型
     */
    public static AlipayActivityAutoCreateExtInfoDto.Bank parseBank(String bankStr) {
        List<String> bankNameAndBankCardType = StringUtil.splitStringWithSeparator(bankStr, JOINER);
        if (CollectionUtils.isEmpty(bankNameAndBankCardType)) {
            return null;
        }
        AlipayActivityAutoCreateExtInfoDto.Bank bank = new AlipayActivityAutoCreateExtInfoDto.Bank();
        bank.setName(bankNameAndBankCardType.get(0));
        bank.setCardType(Integer.valueOf(bankNameAndBankCardType.get(1)));
        return bank;
    }

    /**
     * 可用时间设置 每周指定
     *
     * @param weeks 0,1,0,1,0,1,0
     * @param times 00:00:00-01:00:00
     */
    public static AlipayActivityAutoCreateExtInfoDto.EnableTime parseEnableTime(String weeks, String times) {
        if (StringUtils.isAnyBlank(weeks, times)) {
            return null;
        }
        AlipayActivityAutoCreateExtInfoDto.EnableTime enableTime = new AlipayActivityAutoCreateExtInfoDto.EnableTime();
        enableTime.setWeeks(StringUtil.splitIntegerWithComma(weeks));
        enableTime.setTimes(StringUtil.splitStringWithComma(times));
        return enableTime;
    }

    /**
     * 预警设置
     *
     * @param alarmAmount 预警金额
     * @param alarmPhone 预警手机号 多个逗号分割
     */
    public static AlipayActivityAutoCreateExtInfoDto.Alarm parseAlarm(String alarmAmount, String alarmPhone) {
        if (StringUtils.isAllBlank(alarmAmount, alarmPhone)) {
            return null;
        }
        AlipayActivityAutoCreateExtInfoDto.Alarm alarm = new AlipayActivityAutoCreateExtInfoDto.Alarm();
        alarm.setAmount(alarmAmount);
        alarm.setPhoneList(StringUtil.splitStringWithComma(alarmPhone));
        return alarm;
    }

    /**
     * 每日预算上限
     * @param dayBudgetType 每日预算上限类型
     * @param dayBudgetAmount 每日预算上限（元/份）
     * @return 每日预算上限
     */
    public static CardPayCutAutoCreateExtInfoDto.DayBudget parseDayBudget(Integer dayBudgetType, String dayBudgetAmount) {
        if (dayBudgetType == null || StringUtils.isBlank(dayBudgetAmount)) {
            return null;
        }
        CardPayCutAutoCreateExtInfoDto.DayBudget dayBudget = new CardPayCutAutoCreateExtInfoDto.DayBudget();
        dayBudget.setAmount(dayBudgetAmount);
        dayBudget.setType(dayBudgetType);
        return dayBudget;
    }

    /**
     * 单一门槛优惠
     *
     * @param gamePlaySingleType 单一玩法类型 {@link GamePlaySingleTypeEnum}
     * @param gamePlaySingleMaxAmount 单一玩法最大优惠金额
     * @param gamePlaySingleThreshold 单一玩法门槛
     * @param gamePlaySingleAveragePrice 单一玩法优惠金额均价
     * @param gamePlaySingleAverage 单一玩法优惠区间
     * @param gamePlaySingleRule 单一玩法规则
     * @return 单一门槛优惠
     */
    public static CardPayCutAutoCreateExtInfoDto.GamePlaySingle parseGamePlaySingle(Integer gamePlaySingleType, String gamePlaySingleMaxAmount, String gamePlaySingleThreshold, String gamePlaySingleAveragePrice, String gamePlaySingleAverage, String gamePlaySingleRule) {
        if (GamePlaySingleTypeEnum.getByType(gamePlaySingleType) == null) {
            return null;
        }
        CardPayCutAutoCreateExtInfoDto.GamePlaySingle gamePlaySingle = new CardPayCutAutoCreateExtInfoDto.GamePlaySingle();
        gamePlaySingle.setType(gamePlaySingleType);
        gamePlaySingle.setThreshold(gamePlaySingleThreshold);
        if (GamePlaySingleTypeEnum.FIXED_REDUCE.getType().equals(gamePlaySingleType)) {
            gamePlaySingle.setAmount(gamePlaySingleRule);
        }
        if (GamePlaySingleTypeEnum.DISCOUNT_REDUCE.getType().equals(gamePlaySingleType)) {
            gamePlaySingle.setRule(gamePlaySingleRule);
            gamePlaySingle.setMaxAmount(gamePlaySingleMaxAmount);
        }
        if (GamePlaySingleTypeEnum.SMART_REDUCE.getType().equals(gamePlaySingleType)) {
            gamePlaySingle.setAverage(StringUtil.splitStringWithSeparator(gamePlaySingleAverage, JOINER));
            gamePlaySingle.setAveragePrice(gamePlaySingleAveragePrice);
        }
        return gamePlaySingle;
    }

    /**
     * 阶梯门槛优惠
     *
     * @param gamePlayStepRule 阶梯门槛优惠规则: x-y-1-1，满x元-y元-优惠类型1减至2定额立减-优惠规则，多组根据英文逗号分割
     * @param gamePlayStepMaxAmount 最高优惠金额（元）
     * @return 阶梯门槛优惠
     */
    public static CardPayCutAutoCreateExtInfoDto.GamePlayStep parseGamePlayStep(String gamePlayStepRule, String gamePlayStepMaxAmount) {
        if (StringUtils.isAnyBlank(gamePlayStepRule, gamePlayStepMaxAmount)) {
            return null;
        }
        CardPayCutAutoCreateExtInfoDto.GamePlayStep gamePlayStep = new CardPayCutAutoCreateExtInfoDto.GamePlayStep();
        gamePlayStep.setMaxAmount(gamePlayStepMaxAmount);
        gamePlayStep.setArray(StringUtil.splitStringWithComma(gamePlayStepRule).stream().map(rule -> {
            List<String> rules = StringUtil.splitStringWithSeparator(rule, JOINER);
            CardPayCutAutoCreateExtInfoDto.GamePlayRule gamePlayRule = new CardPayCutAutoCreateExtInfoDto.GamePlayRule();
            gamePlayRule.setThresholdArea(Lists.newArrayList(rules.get(0), rules.get(1)));
            gamePlayRule.setType(Integer.parseInt(rules.get(2)));
            gamePlayRule.setRule(rules.get(3));
            return gamePlayRule;
        }).collect(Collectors.toList()));
        return gamePlayStep;
    }

    /**
     * 频率限制
     *
     * @param timeLimitStr 频率限制 x-y，x表示周期 1-自然日 2-自然周 3-自然月 4-自然年，y表示次数，如果多种限制根据英文逗号分割
     * @return 频率限制
     */
    public static List<CardPayCutAutoCreateExtInfoDto.TimeLimit> parseTimeLimit(String timeLimitStr) {
        if (StringUtils.isBlank(timeLimitStr)) {
            return null;
        }
        List<String> timeLimits = StringUtil.splitStringWithComma(timeLimitStr);
        if (CollectionUtils.isEmpty(timeLimits)) {
            return null;
        }
        return timeLimits.stream().map(timeLimitSingleStr -> {
            CardPayCutAutoCreateExtInfoDto.TimeLimit timeLimit = new CardPayCutAutoCreateExtInfoDto.TimeLimit();
            List<Integer> timeLimitSingle = StringUtil.splitIntegerWithSeparator(timeLimitSingleStr, JOINER);
            timeLimit.setType(timeLimitSingle.get(0));
            timeLimit.setNum(timeLimitSingle.get(1));
            return timeLimit;
        }).collect(Collectors.toList());
    }

    public static boolean isValidAlarm(AlipayActivityAutoCreateExtInfoDto.Alarm alarm, long totalAmount) {
        if (alarm == null) {
            return true;
        }
        if (alarm.getAmount() == null) {
            return false;
        }
        if (alarm.getPhoneList() == null && alarm.getAppidList() == null) {
            return false;
        }
        long alarmAmount = PriceUtils.convertY2F(alarm.getAmount());
        if (alarmAmount < FIVE_THOUSANDS_CENT || alarmAmount < totalAmount) {
            return false;
        }
        if (CollectionUtils.isNotEmpty(alarm.getAppidList()) && alarm.getAppidList().size() > ALARM_APP_ID_LIMIT) {
            return false;
        }
        return CollectionUtils.isNotEmpty(alarm.getPhoneList()) && alarm.getPhoneList().size() > ALARM_PHONE_LIMIT;
    }
}
