package cn.com.duiba.quanyi.center.api.utils.activity;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.quanyi.center.api.dto.activity.ActivityDto;
import cn.com.duiba.quanyi.center.api.dto.activity.ActivityPrizeDto;
import cn.com.duiba.quanyi.center.api.dto.activity.ActivityPrizeStockDto;
import cn.com.duiba.quanyi.center.api.dto.activity.PrizePeriodStockDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityBlackWhiteListCacheDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityCommonCacheTypeEnum;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityCommonCustomSignUpVerifyDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityCommonReachStandardDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityCommonUserApiVerifyDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityCommonUserCustomVerifyDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityCommonUserInfoCacheDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityPrizeReceiveRecordCacheDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivitySignUpCacheDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.PayOrderTakeVoucherCacheDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.cache.UserTakeStatCacheDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.detail.ActivityCommonDetailDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.detail.ActivityCommonPrizeDetailDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonBlackConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonInterfaceConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonNoticeConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonPayConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonPrizeStockLimitConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonReachStandardApiConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonReachStandardConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonReachStandardWhiteListConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonSignUpConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonTakeLimitConfDetailDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonTakeLimitConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonTimeLimitAbsoluteConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonTimeLimitConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonUserLimitConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.ActivityCommonWhiteConfDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.result.ActivityCommonUserPrizeResultDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.result.ActivityCommonUserResultCanTakeDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.result.ActivityCommonUserResultDto;
import cn.com.duiba.quanyi.center.api.dto.activity.common.result.ActivityCommonUserResultSubPrizeDto;
import cn.com.duiba.quanyi.center.api.dto.common.ActivityPeriodicDto;
import cn.com.duiba.quanyi.center.api.dto.pay.PayOrderDto;
import cn.com.duiba.quanyi.center.api.dto.user.UserDto;
import cn.com.duiba.quanyi.center.api.enums.BooleanEnum;
import cn.com.duiba.quanyi.center.api.enums.LogicDeleteEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.ActivityExtEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.ActivityPrizeDrawTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.ActivityPrizeExtEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.ActivityPrizeTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.BizActTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityBlackWhiteListTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityCommonBlackWhiteUserMarkTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityCommonDimensionEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityCommonReachStandardTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityCommonSignUpVerifyRuleTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityCommonUserIdTypeEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityCommonUserVerifyStatusEnum;
import cn.com.duiba.quanyi.center.api.enums.activity.common.ActivityPeriodicUnitEnum;
import cn.com.duiba.quanyi.center.api.enums.user.UserExtCommonKeyEnum;
import cn.com.duiba.quanyi.center.api.utils.BizConsumer;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import com.alibaba.fastjson.JSON;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author lizhi
 * @date 2025/4/22 10:52
 */
public class ActivityCommonUtils {

    private ActivityCommonUtils() {
    }

    public static void throwBiz(ActivityCommonErrorCodeEnum errorCodeEnum) throws BizException {
        throw new BizException(errorCodeEnum.getDesc()).withCode(errorCodeEnum.name());
    }

    /**
     * 将活动配置，转化为具体对象
     *
     * @param activityDto    活动基础信息
     * @param activityExtMap 活动扩展信息
     * @param prizeList      奖品列表
     * @param prizeExtMap    奖品扩展信息
     * @return 活动完整配置对象
     * @throws BizException 配置异常
     */
    public static ActivityCommonDetailDto convertDetail(ActivityDto activityDto, Map<Integer, String> activityExtMap,
            List<ActivityPrizeDto> prizeList, Map<Long, Map<Integer, String>> prizeExtMap) throws BizException {
        if (activityDto == null || !LogicDeleteEnum.NON_DELETED.getCode().equals(activityDto.getLogicDelete())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000001);
            return null;
        }
        if (!BizActTypeEnum.COMMON.getType().equals(activityDto.getActivityType())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000017);
            return null;
        }
        if (MapUtils.isEmpty(activityExtMap)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000004);
            return null;
        }
        if (CollectionUtils.isEmpty(prizeList)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000005);
            return null;
        }
        if (MapUtils.isEmpty(prizeExtMap)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000006);
            return null;
        }
        ActivityCommonDetailDto detailDto = BeanUtils.copy(activityDto, ActivityCommonDetailDto.class);
        detailDto.setDrawType(getIntegerByExt(activityExtMap, ActivityExtEnum.PRIZE_DRAW_TYPE.getType()));
        detailDto.setSkinId(activityExtMap.get(ActivityExtEnum.SKIN_ID.getType()));
        detailDto.setSubType(getIntegerByExt(activityExtMap, ActivityExtEnum.COMMON_SUB_TYPE.getType()));
        detailDto.setUserLimit(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_USER_LIMIT.getType(),
                ActivityCommonUserLimitConfDto.class));
        detailDto.setTimeLimit(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_TAKE_OPEN_TIME_LIMIT.getType(),
                ActivityCommonTimeLimitConfDto.class));
        detailDto.setTakeNumLimit(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_TAKE_TIMES_LIMIT.getType(),
                ActivityCommonTakeLimitConfDto.class));
        detailDto.setWhiteConf(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_WHITE_LIST.getType(),
                ActivityCommonWhiteConfDto.class));
        detailDto.setBlackConf(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_BLACK_LIST.getType(),
                ActivityCommonBlackConfDto.class));
        detailDto.setReachStandardConf(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_REACH_STANDARD.getType(),
                ActivityCommonReachStandardConfDto.class));
        detailDto.setPayConf(
                getObjByExt(activityExtMap, ActivityExtEnum.COMMON_PAY_CONF.getType(), ActivityCommonPayConfDto.class));
        detailDto.setSignUpConf(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_SIGN_UP.getType(),
                ActivityCommonSignUpConfDto.class));
        detailDto.setEnableStore(getIntegerByExt(activityExtMap, ActivityExtEnum.COMMON_STORE_CONF.getType()));
        detailDto.setNoticeConf(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_NOTICE_CONF.getType(),
                ActivityCommonNoticeConfDto.class));
        detailDto.setInterfaceConf(getObjByExt(activityExtMap, ActivityExtEnum.COMMON_INTERFACE_CONF.getType(),
                ActivityCommonInterfaceConfDto.class));
        detailDto.setPrizeList(buildPrizeDetail(prizeList, prizeExtMap));
        return detailDto;
    }

    private static List<ActivityCommonPrizeDetailDto> buildPrizeDetail(List<ActivityPrizeDto> prizeList,
            Map<Long, Map<Integer, String>> prizeExtMap) {
        List<ActivityCommonPrizeDetailDto> list = new ArrayList<>();
        for (ActivityPrizeDto prizeDto : prizeList) {
            ActivityCommonPrizeDetailDto detailDto = buildPrizeDetail(prizeDto, prizeExtMap.get(prizeDto.getId()));
            list.add(detailDto);
        }
        return list;
    }

    private static ActivityCommonPrizeDetailDto buildPrizeDetail(ActivityPrizeDto prizeDto, Map<Integer, String> extMap) {
        ActivityCommonPrizeDetailDto detailDto = BeanUtils.copy(prizeDto, ActivityCommonPrizeDetailDto.class);
        detailDto.setProbabilityValue(extMap.get(ActivityPrizeExtEnum.PROBABILITY_VALUE.getType()));
        detailDto.setTakeNumLimit(getObjByExt(extMap, ActivityPrizeExtEnum.TAKE_NUM_LIMIT.getType(),
                ActivityCommonTakeLimitConfDto.class));
        detailDto.setStockLimit(getObjByExt(extMap, ActivityPrizeExtEnum.STOCK_LIMIT.getType(),
                ActivityCommonPrizeStockLimitConfDto.class));
        return detailDto;
    }

    private static Integer getIntegerByExt(Map<Integer, String> extMap, Integer confName) throws BizException {
        if (MapUtils.isEmpty(extMap)) {
            return null;
        }
        String val = extMap.get(confName);
        if (StringUtils.isBlank(val)) {
            return null;
        }
        if (!NumberUtils.isNumeric(val)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000008);
        }
        return Integer.parseInt(val);
    }

    private static <T> T getObjByExt(Map<Integer, String> extMap, Integer confName, Class<T> clazz) {
        if (MapUtils.isEmpty(extMap)) {
            return null;
        }
        String val = extMap.get(confName);
        if (StringUtils.isBlank(val)) {
            return null;
        }
        return JSON.parseObject(val, clazz);
    }

    /**
     * 判断是否在可领取
     *
     * @param timeLimitConf 领取周期配置
     * @param now 当前时间
     * @param needTime 是否需要 下个周期开始时间/本周期结束时间
     * @return left：是否在可领取时间内，middle：下个周期开始时间，right：本周期结束时间
     */
    public static Triple<Boolean, Date, Date> calculateAccessTime(ActivityCommonTimeLimitConfDto timeLimitConf, @Nonnull Date now, boolean needTime)
            throws BizException {
        ActivityPeriodicUnitEnum unitEnum = ActivityPeriodicUnitEnum.getByCode(timeLimitConf.getPeriodicUnit());
        if (unitEnum == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000015);
        }
        switch (unitEnum) {
            case FOREVER:
                return Triple.of(true, null, null);
            case ABSOLUTE:
                return calculateAccessTimeByAbsolute(timeLimitConf.getAbsoluteTimeList(), now, needTime);
            case DAY:
                return calculateAccessTimeByDay(timeLimitConf, now, needTime, 1);
            case WEEK:
                return calculateAccessTimeByWeek(timeLimitConf, now, needTime);
            case MONTH:
                return calculateAccessTimeByMonth(timeLimitConf, now, needTime);
            default:
                throwBiz(ActivityCommonErrorCodeEnum.AC000015);
                return Triple.of(false, null, null);
        }
    }

    private static Triple<Boolean, Date, Date> calculateAccessTimeByAbsolute(List<ActivityCommonTimeLimitAbsoluteConfDto> absoluteTimeList, Date now, boolean needTime) {
        if (CollectionUtils.isEmpty(absoluteTimeList)) {
            return Triple.of(false, null, null);
        }
        // 过滤出在时间范围内的
        List<ActivityCommonTimeLimitAbsoluteConfDto> inTimeList = absoluteTimeList.stream()
                .filter(time -> time.getAbsoluteStartTime().before(now) && time.getAbsoluteEndTime().after(now))
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(inTimeList)) {
            // 本周期结束时间 = 在时间范围内的，最晚的结束时间
            Date date = needTime ? inTimeList.stream().map(ActivityCommonTimeLimitAbsoluteConfDto::getAbsoluteEndTime)
                    .max(Comparator.naturalOrder()).orElse(null) : null;
            return Triple.of(true, null, date);
        }
        // 下个周期开始时间 = 开始时间大于当前时间，最早的开始时间
        Date date = needTime ? absoluteTimeList.stream().map(ActivityCommonTimeLimitAbsoluteConfDto::getAbsoluteStartTime)
                .filter(absoluteStartTime -> absoluteStartTime.after(now)).min(Comparator.naturalOrder()).orElse(null) : null;
        return Triple.of(false, date, null);
    }

    private static Triple<Boolean, Date, Date> calculateAccessTimeByDay(ActivityCommonTimeLimitConfDto timeLimitConf, @Nonnull Date now, boolean needTime, int nextDayOffset) {
        Date dayStartTime = getTime(now, null, null, timeLimitConf.getDayStartHour(), timeLimitConf.getDayStartMinute(),
                0);
        Date dayEndTime = getTime(now, null, null, timeLimitConf.getDayEndHour(), timeLimitConf.getDayEndMinute(), 59);
        boolean inTime = dayStartTime.before(now) && dayEndTime.after(now);
        if (inTime) {
            return Triple.of(true, null, needTime ? dayEndTime : null);
        }
        if (now.before(dayStartTime)) {
            return Triple.of(false, needTime ? dayStartTime : null, null);
        }
        return Triple.of(false, needTime ? DateUtils.daysAddOrSub(dayStartTime, nextDayOffset) : null, null);
    }

    private static Triple<Boolean, Date, Date> calculateAccessTimeByWeek(ActivityCommonTimeLimitConfDto timeLimitConf, @Nonnull Date now, boolean needTime) {
        if (CollectionUtils.isEmpty(timeLimitConf.getWeekLimitList())) {
            return Triple.of(false, null, null);
        }
        // 获取当前是周几
        int currentWeekDay = getCurrentWeekDay(now);
        if (!timeLimitConf.getWeekLimitList().contains(currentWeekDay)) {
            // 今天不在可访问时间范围内
            // 日纬度最早可访问时间
            Date dayStartTime = needTime ? getTime(now, null, null, timeLimitConf.getDayStartHour(), timeLimitConf.getDayStartMinute(), 0) : null;
            // 最近可访问时间 = 日纬度 + 下一个可访问日与今天的天数差值
            return Triple.of(false, needTime ? DateUtils.daysAddOrSub(dayStartTime, getNextWeekDayOffset(currentWeekDay, timeLimitConf.getWeekLimitList())) : null, null);
        }
        // 今天可访问
        if (dayNotAllTime(timeLimitConf)) {
            // 非全天，直接判断当天时间是否可访问即可
            return calculateAccessTimeByDay(timeLimitConf, now, needTime, needTime ? getNextWeekDayOffset(currentWeekDay, timeLimitConf.getWeekLimitList()) : 0);
        }
        // 全天
        if (timeLimitConf.getWeekLimitList().size() >= 7) {
            // 周一到周日都选了
            return Triple.of(true, null, null);
        }
        Date dayEndTime = needTime ? getTime(now, null, null, timeLimitConf.getDayEndHour(), timeLimitConf.getDayEndMinute(), 59) : null;
        return Triple.of(true, null, needTime ? DateUtils.daysAddOrSub(dayEndTime, getLastWeekDayOffset(currentWeekDay, timeLimitConf.getWeekLimitList())) : null);
    }

    private static int getLastWeekDayOffset(int currentWeekDay, List<Integer> weekLimitList) {
        Collections.sort(weekLimitList);
        int lastWeekDay = currentWeekDay;
        int index = weekLimitList.indexOf(currentWeekDay);
        for (int i = index; i < weekLimitList.size(); i++) {
            Integer weekDay = weekLimitList.get(i);
            if (weekDay - lastWeekDay > 1) {
                return lastWeekDay - currentWeekDay;
            }
            if (weekDay - lastWeekDay == 1) {
                lastWeekDay = weekDay;
            }
        }
        for (int i = 0; i < index; i++) {
            int weekDay = weekLimitList.get(i) + 7;
            if (weekDay - lastWeekDay > 1) {
                return lastWeekDay - currentWeekDay;
            }
            if (weekDay - lastWeekDay == 1) {
                lastWeekDay = weekDay;
            }
        }
        return lastWeekDay - currentWeekDay;
    }

    /**
     * 计算下一个可访问日与今天差距多少天
     * @param currentWeekDay 当前所在周几
     * @param weekLimitList 可访问的 周几 集合， 比如 1，2，6
     * @return 下一个可访问日，距离今天还有几天，比如当前周六，那么就是2天，当前周二，就是4天
     */
    private static int getNextWeekDayOffset(int currentWeekDay, List<Integer> weekLimitList) {
        // 获取下一个可访问的日期
        int nextWeek = getNextWeekDay(currentWeekDay, weekLimitList);
        if (nextWeek >= currentWeekDay) {
            // 周的数字大于当前周，说明当天不是本周可访问的最后一天，取两个周几的差值就是差距几天
            return nextWeek - currentWeekDay;
        }
        // 比如当前是周六，下一个可访问时间是周一，那么还有 1 + 7 - 6 = 2天
        return nextWeek + 7 - currentWeekDay;
    }

    /**
     * 计算下一个可访问日属于周几
     * @param currentWeekDay 当前属于周几
     * @param weekLimitList 可访问的周日期集合
     * @return 下一个可访问日属于周几
     */
    private static int getNextWeekDay(int currentWeekDay, List<Integer> weekLimitList) {
        // 按照正序排序
        Collections.sort(weekLimitList);
        for (Integer weekDay : weekLimitList) {
            if (weekDay > currentWeekDay) {
                return weekDay;
            }
        }
        return weekLimitList.get(0);
    }

    private static int getCurrentWeekDay(Date now) {
        int week = getDateField(now, Calendar.DAY_OF_WEEK);
        if (week == Calendar.SUNDAY) {
            return 7;
        }
        return week - 1;
    }

    private static Triple<Boolean, Date, Date> calculateAccessTimeByMonth(ActivityCommonTimeLimitConfDto timeLimitConf, @Nonnull Date now, boolean needTime) {
        int currentMonthDay = getDateField(now, Calendar.DAY_OF_MONTH);
        int currentMonth = getDateField(now, Calendar.MONTH);
        if (currentMonthDay < timeLimitConf.getMonthStartTimeByDay() || currentMonthDay > timeLimitConf.getMonthEndTimeByDay()) {
            // 今天不在可访问时间范围内
            int month = currentMonthDay < timeLimitConf.getMonthStartTimeByDay() ? currentMonth : currentMonth + 1;
            return Triple.of(false, needTime ? getTime(now, month, timeLimitConf.getMonthStartTimeByDay(), timeLimitConf.getDayStartHour(), timeLimitConf.getDayStartMinute(), 0) : null, null);
        }
        if (dayNotAllTime(timeLimitConf)) {
            // 非全天，直接判断当天时间是否可访问即可
            return calculateAccessTimeByDay(timeLimitConf, now, needTime, needTime ? getNextMonthDayOffset(currentMonthDay, currentMonth, timeLimitConf, now) : 0);
        }
        // 全天，周期结束时间 = 最后一天
        return Triple.of(true, null, needTime ? getTime(now, currentMonth, timeLimitConf.getMonthEndTimeByDay(), timeLimitConf.getDayEndHour(), timeLimitConf.getDayEndMinute(), 59) : null);
    }

    private static int getNextMonthDayOffset(int currentMonthDay, int currentMonth, ActivityCommonTimeLimitConfDto timeLimitConf, Date now) {
        if (currentMonthDay < timeLimitConf.getMonthEndTimeByDay()) {
            return 1;
        }
        int currentDayOfYear = getDateField(now, Calendar.DAY_OF_YEAR);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.set(Calendar.MONTH, currentMonth + 1);
        calendar.set(Calendar.DAY_OF_MONTH, timeLimitConf.getMonthStartTimeByDay());
        return calendar.get(Calendar.DAY_OF_YEAR) - currentDayOfYear;
    }

    private static int getDateField(Date now, int field) {
        Calendar calWeek = Calendar.getInstance();
        calWeek.setTime(now);
        return calWeek.get(field);
    }

    private static boolean dayNotAllTime(ActivityCommonTimeLimitConfDto timeLimitConf) {
        return timeLimitConf.getDayStartHour() != 0 || timeLimitConf.getDayStartMinute() != 0
                || timeLimitConf.getDayEndHour() != 23 || timeLimitConf.getDayEndMinute() != 59;
    }

    public static Date getTime(Date date, Integer month, Integer dayOfMonth, Integer hour, Integer minute, Integer second) {
        if (null == date) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        if (month != null) {
            calendar.set(Calendar.MONTH, month);
        }
        if (dayOfMonth != null) {
            calendar.set(Calendar.DAY_OF_MONTH, dayOfMonth);
        }
        if (hour != null) {
            calendar.set(Calendar.HOUR_OF_DAY, hour);
        }
        if (minute != null) {
            calendar.set(Calendar.MINUTE, minute);
        }
        if (second != null) {
            calendar.set(Calendar.SECOND, second);
        }
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取活动需要查询的用户缓存枚举
     * @param detailDto 活动配置
     * @return 需要查询的用户缓存枚举
     * @see cn.com.duiba.quanyi.center.api.dto.activity.common.cache.ActivityCommonCacheTypeEnum
     */
    public static List<Integer> getUserInfoEnumList(ActivityCommonDetailDto detailDto) {
        List<Integer> typeList = new ArrayList<>();
        typeList.add(ActivityCommonCacheTypeEnum.RECEIVE_RECORD.getType());
        typeList.add(ActivityCommonCacheTypeEnum.USER_TAKE_STAT.getType());
        typeList.add(ActivityCommonCacheTypeEnum.USER.getType());
        typeList.add(ActivityCommonCacheTypeEnum.USER_EXT.getType());
        if (isOpenUserApiVerify(detailDto)) {
            typeList.add(ActivityCommonCacheTypeEnum.USER_API_VERIFY.getType());
        }
        if (isOpenUserCustomVerify(detailDto)) {
            typeList.add(ActivityCommonCacheTypeEnum.USER_CUSTOM_VERIFY.getType());
        }
        if (isOpenApiReachStandard(detailDto)) {
            typeList.add(ActivityCommonCacheTypeEnum.API_REACH_STANDARD.getType());
        }
        if (isOpenSignUp(detailDto)) {
            typeList.add(ActivityCommonCacheTypeEnum.SIGN_UP_LIST.getType());
        }
        if (isOpenSignUpCustomVerify(detailDto)) {
            typeList.add(ActivityCommonCacheTypeEnum.SIGN_UP_CUSTOM_VERIFY.getType());
        }
        if (isOpenProcessingBizOrder(detailDto)) {
            typeList.add(ActivityCommonCacheTypeEnum.LAST_PAY_ORDER.getType());
        }
        if (isOpenPay(detailDto)) {
            typeList.add(ActivityCommonCacheTypeEnum.PAY_ORDER_VOUCHER.getType());
        }
        return typeList;
    }

    /**
     * 获取用户黑白名单用户唯一标记
     * @param detailDto 活动配置
     * @param userDto 用户信息
     * @param userExtMap 用户扩展信息
     * @return 用户唯一标记
     * @throws BizException 配置异常时抛出异常
     */
    public static String getUserMark(ActivityCommonDetailDto detailDto, UserDto userDto, Map<Integer, String> userExtMap) throws BizException {
        Integer userMarkType = getUserMarkType(detailDto);
        if (userMarkType == null) {
            return null;
        }
        if (ActivityCommonBlackWhiteUserMarkTypeEnum.USER_PHONE.getType().equals(userMarkType)) {
            return MapUtils.isEmpty(userExtMap) ? null : userExtMap.get(UserExtCommonKeyEnum.USER_PHONE.getCode());
        }
        return null;
    }

    // 获取黑白名单验证标记
    private static Integer getUserMarkType(ActivityCommonDetailDto detailDto) throws BizException {
        Integer whiteUserMarkType = isOpenWhiteList(detailDto) ? detailDto.getWhiteConf().getUserMarkType() : null;
        Integer blackUserMarkType = isOpenBlackList(detailDto) ? detailDto.getBlackConf().getUserMarkType() : null;
        if (whiteUserMarkType == null && blackUserMarkType == null) {
            return null;
        }
        if (whiteUserMarkType == null) {
            return blackUserMarkType;
        }
        if (blackUserMarkType == null) {
            return whiteUserMarkType;
        }
        if (!Objects.equals(whiteUserMarkType, blackUserMarkType)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000016);
            return null;
        }
        return whiteUserMarkType;
    }

    /**
     * 用户信息是否已全部命中缓存
     * @param detailDto 活动配置
     * @param userInfoDto 用户信息
     * @param blackWhiteList 黑白名单
     * @return true-已全部命中，false-未全部命中
     */
    public static boolean isAllHitCache(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList) {
        boolean allHitCache = isAllHitCache(detailDto, userInfoDto);
        if (!allHitCache) {
            return false;
        }
        // 需要加载黑白名单
        return !needLoadBlackWhiteList(detailDto, blackWhiteList);
    }

    /**
     * 用户信息是否已全部命中缓存
     * @param detailDto 活动配置
     * @param userInfoDto 用户信息
     * @return true-已全部命中，false-未全部命中
     */
    public static boolean isAllHitCache(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        // 领取记录、领取记录统计数据、用户、用户扩展信息缓存，必须查询
        if (userInfoDto.getReceiveRecordList() == null || userInfoDto.getUserTakeStatList() == null
                || userInfoDto.getUserDto() == null || userInfoDto.getUserExtMap() == null) {
            return false;
        }
        if (needLoadUserApiVerify(detailDto, userInfoDto)) {
            // 需要加载用户API验证结果
            return false;
        }
        if (needLoadUserCustomVerify(detailDto, userInfoDto)) {
            // 需要加载用户定制验证结果
            return false;
        }
        if (needLoadApiReachStandard(detailDto, userInfoDto)) {
            // 需要加载api达标结果
            return false;
        }
        if (needLoadSignUp(detailDto, userInfoDto)) {
            // 需要加载报名记录
            return false;
        }
        if (needLoadSignUpCustomVerify(detailDto, userInfoDto)) {
            // 需要加载用户定制验证结果
            return false;
        }
        if (needLoadProcessingBizOrder(detailDto, userInfoDto)) {
            // 需要加载处理中订单
            return false;
        }
        // 需要加载支付订单凭证
        return !needLoadPayOrderVoucher(detailDto, userInfoDto);
    }

    // 是否需要加载用户API验证
    public static boolean needLoadUserApiVerify(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        if (!isOpenUserApiVerify(detailDto)) {
            // 未开启
            return false;
        }
        return userInfoDto.getUserApiVerify() == null
                || ActivityCommonUserVerifyStatusEnum.EXCEPTION.getStatus().equals(userInfoDto.getUserApiVerify().getStatus());
    }

    // 是否开启了用户API验证
    public static boolean isOpenUserApiVerify(ActivityCommonDetailDto detailDto) {
        if (detailDto.getUserLimit() == null) {
            // 未开启
            return false;
        }
        return detailDto.getUserLimit().isNeedApiVerify();
    }

    // 是否需要加载用户定制验证
    public static boolean needLoadUserCustomVerify(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        if (!isOpenUserCustomVerify(detailDto)) {
            // 未开启
            return false;
        }
        return userInfoDto.getUserCustomVerify() == null
                || ActivityCommonUserVerifyStatusEnum.EXCEPTION.getStatus().equals(userInfoDto.getUserCustomVerify().getStatus());
    }

    // 是否开启了用户定制验证
    public static boolean isOpenUserCustomVerify(ActivityCommonDetailDto detailDto) {
        if (detailDto.getUserLimit() == null) {
            // 未开启
            return false;
        }
        return detailDto.getUserLimit().isNeedCustomVerify();
    }

    // 是否需要加载API达标
    public static boolean needLoadApiReachStandard(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        if (!isOpenApiReachStandard(detailDto)) {
            // 未开启
            return false;
        }
        return userInfoDto.getApiReachStandard() == null || BooleanUtils.isNotTrue(userInfoDto.getApiReachStandard().getSuccess());
    }

    // 是否开启了API达标验证
    public static boolean isOpenApiReachStandard(ActivityCommonDetailDto detailDto) {
        if (detailDto.getReachStandardConf() == null) {
            // 未开启
            return false;
        }
        if (!BooleanEnum.TRUE.getCode().equals(detailDto.getReachStandardConf().getEnable())) {
            // 未开启
            return false;
        }
        return ActivityCommonReachStandardTypeEnum.API.getType().equals(detailDto.getReachStandardConf().getType());
    }

    // 是否需要加载报名记录
    public static boolean needLoadSignUp(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        if (!isOpenSignUp(detailDto)) {
            // 未开启
            return false;
        }
        return userInfoDto.getSignUpList() == null;
    }

    // 是否开启了报名
    public static boolean isOpenSignUp(ActivityCommonDetailDto detailDto) {
        if (detailDto.getSignUpConf() == null) {
            // 未开启
            return false;
        }
        return BooleanEnum.TRUE.getCode().equals(detailDto.getSignUpConf().getEnable());
    }

    // 是否需要加载报名定制校验结果
    public static boolean needLoadSignUpCustomVerify(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        if (!isOpenSignUpCustomVerify(detailDto)) {
            // 未开启
            return false;
        }
        return userInfoDto.getCustomSignUpVerify() == null;
    }

    // 是否开启了报名结果定制验证
    public static boolean isOpenSignUpCustomVerify(ActivityCommonDetailDto detailDto) {
        if (detailDto.getSignUpConf() == null) {
            // 未开启
            return false;
        }
        if (!BooleanEnum.TRUE.getCode().equals(detailDto.getSignUpConf().getEnable())) {
            // 未开启
            return false;
        }
        if (!BooleanEnum.TRUE.getCode().equals(detailDto.getSignUpConf().getEnableVerify())) {
            // 未开启报名校验
            return false;
        }
        return ActivityCommonSignUpVerifyRuleTypeEnum.CUSTOM.getType().equals(detailDto.getSignUpConf().getVerifyRule());
    }

    // 是否需要加载处理中订单
    public static boolean needLoadProcessingBizOrder(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        if (!isOpenProcessingBizOrder(detailDto)) {
            // 未开启处理中订单拦截
            return false;
        }
        return Objects.isNull(userInfoDto.getLastPayOrderOptional());
    }

    // 是否开启了处理中订单拦截
    public static boolean isOpenProcessingBizOrder(ActivityCommonDetailDto detailDto) {
        if (!isOpenPay(detailDto)) {
            // 未开启支付
            return false;
        }
        return BooleanEnum.TRUE.getCode().equals(detailDto.getPayConf().getInterceptProcess());
    }

    // 是否需要加载支付凭证
    public static boolean needLoadPayOrderVoucher(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        if (!isOpenPay(detailDto)) {
            // 未开启
            return false;
        }
        return userInfoDto.getPayOrderVoucherList() == null;
    }

    // 是否开启了支付
    public static boolean isOpenPay(ActivityCommonDetailDto detailDto) {
        if (detailDto.getPayConf() == null) {
            // 未开启
            return false;
        }
        return BooleanEnum.TRUE.getCode().equals(detailDto.getPayConf().getEnable());
    }

    // 是否需要加载黑白名单
    public static boolean needLoadBlackWhiteList(ActivityCommonDetailDto detailDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList) {
        if (!isOpenBlackList(detailDto) && !isOpenWhiteList(detailDto)) {
            // 黑名单、白名单均未开启
            return false;
        }
        return blackWhiteList == null;
    }

    // 是否开启了黑名单校验
    public static boolean isOpenBlackList(ActivityCommonDetailDto detailDto) {
        if (detailDto.getBlackConf() == null) {
            // 未开启
            return false;
        }
        // 未开启
        return BooleanEnum.TRUE.getCode().equals(detailDto.getBlackConf().getEnable());
    }

    // 是否开启了白名单校验
    public static boolean isOpenWhiteList(ActivityCommonDetailDto detailDto) {
        if (detailDto.getWhiteConf() == null) {
            // 未开启
            return false;
        }
        // 未开启
        return BooleanEnum.TRUE.getCode().equals(detailDto.getWhiteConf().getEnable());
    }

    /**
     * 根据活动配置、用户相关数据，计算出用户是否可参与/可领取/可支付
     * @param detailDto 活动配置
     * @param prizeStockMap key：奖品ID，value：奖品库存
     * @param prizePeriodStockMap key:奖品ID，value：奖品周期库存
     * @param userInfoDto 用户相关数据
     * @param blackWhiteList 用户黑白名单
     * @return 计算结果
     * @throws BizException 业务异常
     */
    public static ActivityCommonUserResultDto calculateResult(ActivityCommonDetailDto detailDto, Map<Long, ActivityPrizeStockDto> prizeStockMap, Map<Long, List<PrizePeriodStockDto>> prizePeriodStockMap, ActivityCommonUserInfoCacheDto userInfoDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList) throws BizException {
        if (detailDto == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000001);
            return null;
        }
        if (userInfoDto == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC001001);
            return null;
        }
        if (CollectionUtils.isEmpty(detailDto.getPrizeList())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC000005);
            return null;
        }
        if (userInfoDto.getReceiveRecordList() == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002035);
            return null;
        }
        if (userInfoDto.getUserTakeStatList() == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002036);
            return null;
        }
        if (userInfoDto.getUserExtMap() == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002037);
            return null;
        }
        Date now = new Date();
        ActivityCommonUserResultDto resultDto = new ActivityCommonUserResultDto();
        // 填充是否可访问
        fillCanAccess(resultDto, detailDto, now);
        // 填充展示用户ID
        fillShowUserId(resultDto, detailDto, userInfoDto);
        resultDto.setActivityCanTake(new ActivityCommonUserResultCanTakeDto());
        // 填充报名
        fillSignUp(resultDto, detailDto, userInfoDto.getSignUpList(), now);
        // 填充活动是否可访问
        boolean canParticipate = fillActivityParticipate(resultDto, detailDto, userInfoDto, blackWhiteList, now);
        if (!canParticipate) {
            return resultDto;
        }
        resultDto.setPrizeList(buildBasePrizeList(detailDto));
        // 回填奖品是否可参与
        fillPrizeParticipate(resultDto.getPrizeList(), detailDto, userInfoDto, blackWhiteList, now);
        // 回填奖品剩余次数
        fillPrizeTakeNum(resultDto.getPrizeList(), detailDto, userInfoDto, blackWhiteList, now);
        // 回填活动剩余次数
        fillActivityTakeNum(resultDto, detailDto, userInfoDto, blackWhiteList, now);
        // 回填支付凭证
        fillPayOrderVoucher(resultDto, detailDto, userInfoDto);
        // 回填处理中支付单号
        fillProcessingBizOrderNo(resultDto, detailDto, userInfoDto);
        // 计算是否可支付/领取
        fillCanTakeCanPay(resultDto, detailDto, userInfoDto, now);
        // 计算库存是否售罄
        fillSellOut(resultDto, detailDto, prizeStockMap, prizePeriodStockMap);
        return resultDto;
    }

    private static void fillCanAccess(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, Date now) {
        try {
            checkActivityCanAccess(detailDto, now);
            resultDto.setCanAccess(true);
            resultDto.setAccessErrCode("");
        } catch (BizException e) {
            resultDto.setCanAccess(false);
            resultDto.setAccessErrCode(e.getCode());
        }
    }

    private static void checkActivityCanAccess(ActivityCommonDetailDto detailDto, Date now) throws BizException {
        if (!BooleanEnum.TRUE.getCode().equals(detailDto.getActivityStatus())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002031);
            return;
        }
        if (detailDto.getStartTime() != null && now.before(detailDto.getStartTime())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002032);
            return;
        }
        if (detailDto.getEndTime() != null && now.after(detailDto.getEndTime())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002033);
            return;
        }
        if (detailDto.getTimeLimit() == null || !BooleanEnum.TRUE.getCode().equals(detailDto.getTimeLimit().getEnable())) {
            return;
        }
        Triple<Boolean, Date, Date> triple = calculateAccessTime(detailDto.getTimeLimit(), now, false);
        if (BooleanUtils.isNotTrue(triple.getLeft())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002034);
        }
    }

    // 计算展示用户ID
    private static void fillShowUserId(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) {
        try {
            String showUserId = getShowUserId(detailDto, userInfoDto);
            if (showUserId == null) {
                throwBiz(ActivityCommonErrorCodeEnum.AC002030);
                return;
            }
            resultDto.setShowUserId(showUserId);
            resultDto.setShowUserIdErrCode("");
        } catch (BizException e) {
            resultDto.setShowUserId(null);
            resultDto.setShowUserIdErrCode(e.getCode());
        }
    }

    private static String getShowUserId(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) throws BizException {
        ActivityCommonInterfaceConfDto interfaceConf = detailDto.getInterfaceConf();
        if (interfaceConf == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002028);
            return null;
        }
        if (!BooleanEnum.TRUE.getCode().equals(interfaceConf.getShowUserId())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002028);
            return null;
        }
        if (ActivityCommonUserIdTypeEnum.USER_ID.getType().equals(interfaceConf.getUserIdType())) {
            return userInfoDto.getUserDto() == null ? null : userInfoDto.getUserDto().getId().toString();
        }
        if (ActivityCommonUserIdTypeEnum.UNION_ID.getType().equals(interfaceConf.getUserIdType())) {
            return userInfoDto.getUserDto() == null ? null : userInfoDto.getUserDto().getUnionId();
        }
        if (ActivityCommonUserIdTypeEnum.CORE_CLIENT_NUMBER.getType().equals(interfaceConf.getUserIdType())) {
            return userInfoDto.getUserExtMap() == null ? null : userInfoDto.getUserExtMap().get(UserExtCommonKeyEnum.CORE_CLIENT_NUMBER.getCode());
        }
        if (ActivityCommonUserIdTypeEnum.BANK_CLIENT_NUMBER.getType().equals(interfaceConf.getUserIdType())) {
            return userInfoDto.getUserExtMap() == null ? null : userInfoDto.getUserExtMap().get(UserExtCommonKeyEnum.BANK_CLIENT_NUMBER.getCode());
        }
        throwBiz(ActivityCommonErrorCodeEnum.AC002029);
        return null;
    }

    private static void fillSellOut(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, Map<Long, ActivityPrizeStockDto> prizeStockMap, Map<Long, List<PrizePeriodStockDto>> prizePeriodStockMap) {
        // 填充奖品是否售罄
        fillPrizeSellOut(resultDto, detailDto, prizeStockMap, prizePeriodStockMap);
        // 填充活动纬度是否售罄
        fillActivitySellOut(resultDto);
    }

    private static void fillActivitySellOut(ActivityCommonUserResultDto resultDto) {
        // 所有有权限的奖品，都售罄了活动才售罄
        // 过滤出非谢谢参与、有权限、未售罄的奖品数量，如果数量为0，则活动售罄
        long count = resultDto.getPrizeList().stream().filter(prize -> !ActivityPrizeTypeEnum.THANKS.getType().equals(prize.getPrizeType()) && prize.getPrizeCanTake().getCanParticipate() && !prize.getPrizeCanTake().getSellOut()).count();
        resultDto.getActivityCanTake().setSellOut(count == 0);
    }

    private static void fillPrizeSellOut(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, Map<Long, ActivityPrizeStockDto> prizeStockMap, Map<Long, List<PrizePeriodStockDto>> prizePeriodStockMap) {
        // key：奖品ID，value：奖品配置
        Map<Long, ActivityCommonPrizeDetailDto> prizeDetailMap = detailDto.getPrizeList().stream().collect(Collectors.toMap(ActivityCommonPrizeDetailDto::getId, Function.identity()));

        resultDto.getPrizeList().forEach(prizeResultDto -> {
            if (ActivityPrizeTypeEnum.THANKS.getType().equals(prizeResultDto.getPrizeType())) {
                // 谢谢参与，不需要判断库存
                prizeResultDto.getPrizeCanTake().setSellOut(false);
                return;
            }
            ActivityCommonPrizeDetailDto prizeDetailDto = prizeDetailMap.get(prizeResultDto.getPrizeId());
            ActivityPrizeStockDto stock = prizeStockMap.get(prizeResultDto.getPrizeId());
            long remainNum = stock.getTotalNum() - stock.getUsedNum();
            if (prizeDetailDto.getStockLimit() == null || !BooleanEnum.TRUE.getCode().equals(prizeDetailDto.getStockLimit().getEnable())) {
                // 周期库存未启用
                prizeResultDto.getPrizeCanTake().setSellOut(remainNum <= 0);
                return;
            }
            // 周期库存已启用
            List<PrizePeriodStockDto> periodStockList = prizePeriodStockMap.get(prizeResultDto.getPrizeId());
            Long minPeriodStockNum = periodStockList.stream().map(s -> s.getTotalNum() - s.getUsedNum()).min(Long::compare).orElse(null);
            if (minPeriodStockNum == null || minPeriodStockNum >= remainNum) {
                prizeResultDto.getPrizeCanTake().setSellOut(remainNum <= 0);
                return;
            }
            prizeResultDto.getPrizeCanTake().setSellOut(minPeriodStockNum <= 0);
        });
    }


    private static void fillCanTakeCanPay(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, Date now) {
        // 计算奖品是否可领取
        fillPrizeCanTakeCanPay(resultDto.getPrizeList(), detailDto);
        // 计算活动是否可领取
        fillActivityCanTakePay(resultDto, detailDto, userInfoDto, now);
    }


    private static void fillPrizeCanTakeCanPay(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonDetailDto detailDto) {
        // 有剩余库存、若配置了支付，有支付凭证
        boolean canNotPay = detailDto.getPayConf() == null || !BooleanEnum.TRUE.getCode().equals(detailDto.getPayConf().getEnable());
        for (ActivityCommonUserPrizeResultDto resultDto : prizeList) {
            fillCanTakeCanPay(resultDto.getPrizeCanTake(), resultDto.getSurplusTakeNum(), canTakeDto -> {
                if (!ActivityPrizeTypeEnum.THANKS.getType().equals(resultDto.getPrizeType())) {
                    // 谢谢参与，都可领取
                    return;
                }
                checkCanTake(resultDto.getSurplusTakeNum(), canNotPay, resultDto.getPayOrderTakeVoucherId(), canTakeDto.getCanParticipate());
            }, canTakeDto -> {
                if (!ActivityPrizeTypeEnum.THANKS.getType().equals(resultDto.getPrizeType())) {
                    // 谢谢参与，都不可支付
                    throwBiz(ActivityCommonErrorCodeEnum.AC002040);
                    return;
                }
                checkCanPay(resultDto.getSurplusTakeNum(), canNotPay, resultDto.getPayOrderTakeVoucherId(), canTakeDto.getCanParticipate());
            });
        }
    }

    private static void fillActivityCanTakePay(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, Date now) {
        boolean canNotPay = detailDto.getPayConf() == null || !BooleanEnum.TRUE.getCode().equals(detailDto.getPayConf().getEnable());
        Long voucherId = resultDto.getPrizeList().stream().map(ActivityCommonUserPrizeResultDto::getPayOrderTakeVoucherId).filter(id -> id != null && id > 0).findFirst().orElse(null);
        fillCanTakeCanPay(resultDto.getActivityCanTake(), resultDto.getSurplusTakeNum(), canTakeDto -> {
            checkCanTake(resultDto.getSurplusTakeNum(), canNotPay, voucherId, canTakeDto.getCanParticipate());
            // 除了剩余次数、是否支付外，报名验证不通过也不能支付
            signUpVerify(resultDto, detailDto.getSignUpConf(), userInfoDto, now);
        }, canTakeDto -> {
            checkCanPay(resultDto.getSurplusTakeNum(), canNotPay, voucherId, canTakeDto.getCanParticipate());
            // 除了剩余次数、是否支付外，报名验证不通过也不能支付
            signUpVerify(resultDto, detailDto.getSignUpConf(), userInfoDto, now);
        });
    }


    private static void fillCanTakeCanPay(ActivityCommonUserResultCanTakeDto canTakeDto, Long surplusTakeNum, BizConsumer<ActivityCommonUserResultCanTakeDto> canTakeConsumer, BizConsumer<ActivityCommonUserResultCanTakeDto> canPayConsumer) {
        if (surplusTakeNum == 0) {
            canTakeDto.setAllTake(true);
        }
        // 填充是否可以领取
        fillCanTake(canTakeDto, canTakeConsumer);
        // 填充是否可以支付
        fillCanPay(canTakeDto, canPayConsumer);
    }


    private static void fillCanTake(ActivityCommonUserResultCanTakeDto canTakeDto, BizConsumer<ActivityCommonUserResultCanTakeDto> consumer) {
        try {
            consumer.accept(canTakeDto);
            canTakeDto.setCanTake(true);
            canTakeDto.setTakeErrCode("");
        } catch (BizException e) {
            canTakeDto.setCanTake(false);
            canTakeDto.setTakeErrCode(e.getCode());
        }
    }

    private static void fillCanPay(ActivityCommonUserResultCanTakeDto canTakeDto, BizConsumer<ActivityCommonUserResultCanTakeDto> consumer) {
        try {
            consumer.accept(canTakeDto);
            canTakeDto.setCanPay(true);
            canTakeDto.setPayErrCode("");
        } catch (BizException e) {
            canTakeDto.setCanPay(false);
            canTakeDto.setPayErrCode(e.getCode());
        }
    }

    private static void checkCanPay(Long surplusTakeNum, boolean canNotPay, Long payOrderTakeVoucherId, Boolean canParticipate) throws BizException {
        if (canNotPay) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002024);
            return;
        }
        if (BooleanUtils.isNotTrue(canParticipate)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002025);
            return;
        }
        if (surplusTakeNum == 0) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002026);
            return;
        }
        if (payOrderTakeVoucherId == null || payOrderTakeVoucherId <= 0) {
            return;
        }
        throwBiz(ActivityCommonErrorCodeEnum.AC002027);
    }

    private static void checkCanTake(Long surplusTakeNum, boolean canNotPay, Long payOrderTakeVoucherId, Boolean canParticipate) throws BizException {
        if (BooleanUtils.isNotTrue(canParticipate)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002021);
            return;
        }
        if (surplusTakeNum == 0) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002022);
            return;
        }
        if (canNotPay) {
            return;
        }
        if (payOrderTakeVoucherId == null || payOrderTakeVoucherId <= 0) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002023);
        }
    }

    private static void fillProcessingBizOrderNo(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) throws BizException {
        ActivityCommonPayConfDto payConf = detailDto.getPayConf();
        if (payConf == null || !BooleanEnum.TRUE.getCode().equals(payConf.getEnable())) {
            // 无需支付
            return;
        }
        Optional<PayOrderDto> lastPayOrderOptional = userInfoDto.getLastPayOrderOptional();
        if (Objects.nonNull(lastPayOrderOptional)) {
            // 缓存存在
            resultDto.setProcessingBizOrderNo(lastPayOrderOptional.map(PayOrderDto::getBizOrderNo).orElse(null));
            return;
        }
        // 缓存不存在
        if (BooleanEnum.TRUE.getCode().equals(payConf.getInterceptProcess())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002038);
        }
    }

    private static void fillPayOrderVoucher(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) throws BizException {
        ActivityCommonPayConfDto payConf = detailDto.getPayConf();
        if (payConf == null || !BooleanEnum.TRUE.getCode().equals(payConf.getEnable())) {
            // 无需支付
            return;
        }
        List<PayOrderTakeVoucherCacheDto> payOrderVoucherList = userInfoDto.getPayOrderVoucherList();
        if (payOrderVoucherList == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002039);
            return;
        }
        if (CollectionUtils.isEmpty(payOrderVoucherList)) {
            return;
        }
        // 支付凭证，根据奖品ID分组, key：奖品ID，value：可使用的支付凭证列表
        Map<Long, List<PayOrderTakeVoucherCacheDto>> payOrderVoucherMap = payOrderVoucherList.stream().collect(Collectors.groupingBy(PayOrderTakeVoucherCacheDto::getPrizeId));
        // 概率抽奖，全部奖品的支付凭证一样
        if (ActivityPrizeDrawTypeEnum.PROBABILITY.getType().equals(detailDto.getDrawType())) {
            resultDto.getPrizeList().forEach(prizeResult -> fillPayOrderVoucher(prizeResult, payOrderVoucherMap.get(0L)));
            return;
        }
        // 直接出奖，每个奖品有自己的凭证
        resultDto.getPrizeList().forEach(prizeResult -> fillPayOrderVoucher(prizeResult, payOrderVoucherMap.get(prizeResult.getPrizeId())));
    }

    private static void fillPayOrderVoucher(ActivityCommonUserPrizeResultDto prizeResult, List<PayOrderTakeVoucherCacheDto> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        if (ActivityPrizeTypeEnum.THANKS.getType().equals(prizeResult.getPrizeType())) {
            // 谢谢参与，无需支付
            return;
        }
        // 理论上应该只有一个可用的支付平镇才对，取第一个
        PayOrderTakeVoucherCacheDto payOrderTakeVoucher = list.get(0);
        prizeResult.setPayOrderTakeVoucherId(payOrderTakeVoucher.getId());
        prizeResult.setPayOrderTakeVoucherTakeCount(payOrderTakeVoucher.getTakeCount());
        prizeResult.setPayOrderTakeVoucherRecordIds(payOrderTakeVoucher.getRecordIds());
        prizeResult.setBizOrderNo(payOrderTakeVoucher.getBizOrderNo());
    }

    private static void fillActivityTakeNum(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) throws BizException {
        // 累加所有的奖品剩余次数(过滤掉剩余次数小于0的)
        long prizeTotalSurplusTakeNum = getPrizeTotalSurplusTakeNum(resultDto);
        resultDto.setSurplusTakeNum(prizeTotalSurplusTakeNum);
        // 活动纬度次数限制
        fillActivityLimitSurplusTakeNum(resultDto, detailDto, userInfoDto, now);
        // 活动纬度白名单次数
        fillActivityWhiteListSurplusTakeNum(resultDto, blackWhiteList);
    }

    private static void fillActivityLimitSurplusTakeNum(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, Date now) throws BizException {
        // key：奖品ID，value：领取统计对象
        Map<Long, UserTakeStatCacheDto> statMap = userInfoDto.getUserTakeStatList().stream().collect(Collectors.toMap(UserTakeStatCacheDto::getPrizeId, Function.identity()));
        UserTakeStatCacheDto stat = statMap.get(0L);
        if (stat != null) {
            resultDto.getActivityCanTake().setUserTakeStatId(stat.getId());
        }
        resultDto.getActivityCanTake().setTakeCount((long)userInfoDto.getReceiveRecordList().size());
        resultDto.getActivityCanTake().setLastRecordId(getMaxId(userInfoDto.getReceiveRecordList()));
        if (detailDto.getTakeNumLimit() == null || !BooleanEnum.TRUE.getCode().equals(detailDto.getTakeNumLimit().getEnable())) {
            // 无库存限制
            return;
        }
        resultDto.setSurplusTakeNum(getSurplusNum(resultDto.getSurplusTakeNum(), detailDto.getTakeNumLimit(), userInfoDto.getReceiveRecordList(), now));
    }

    private static void fillActivityWhiteListSurplusTakeNum(ActivityCommonUserResultDto resultDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList) {
        if (BooleanUtils.isNotTrue(resultDto.getActivityCanTake().getInWhiteList())) {
            // 不在白名单/未开启白名单
            return;
        }
        if (resultDto.getActivityCanTake().getWhiteListId() == null || resultDto.getActivityCanTake().getWhiteListId() <= 0) {
            // 没有拿到有库存的白名单，剩余库存为0
            resultDto.setSurplusTakeNum(0L);
            return;
        }
        // key：白名单主键，value：白名单详情
        Map<Long, ActivityBlackWhiteListCacheDto> whiteListMap = blackWhiteList.stream().collect(Collectors.toMap(ActivityBlackWhiteListCacheDto::getId, Function.identity()));
        ActivityBlackWhiteListCacheDto whiteListCacheDto = whiteListMap.get(resultDto.getActivityCanTake().getWhiteListId());
        if (resultDto.getSurplusTakeNum() == null || resultDto.getSurplusTakeNum() < 0) {
            // 之前没有回填过库存，直接填写白名单库存
            resultDto.setSurplusTakeNum(whiteListCacheDto.getTotalNum() - whiteListCacheDto.getUsedNum());
            return;
        }
        // 取之前库存与当前库存较小的一个
        resultDto.setSurplusTakeNum(Math.min(resultDto.getSurplusTakeNum(), whiteListCacheDto.getTotalNum() - whiteListCacheDto.getUsedNum()));
    }

    private static long getPrizeTotalSurplusTakeNum(ActivityCommonUserResultDto resultDto) {
        // 若奖品剩余次数有小于0的，返回-1 (过滤掉谢谢参与)
        if (resultDto.getPrizeList().stream().filter(prize -> !ActivityPrizeTypeEnum.THANKS.getType().equals(prize.getPrizeType())).anyMatch(prize -> prize.getSurplusTakeNum() < 0)) {
            return -1;
        }
        // 否则返回每个奖品剩余次数总和 (过滤掉谢谢参与)
        return resultDto.getPrizeList().stream().filter(prize -> !ActivityPrizeTypeEnum.THANKS.getType().equals(prize.getPrizeType())).mapToLong(ActivityCommonUserPrizeResultDto::getSurplusTakeNum).sum();
    }

    private static void fillPrizeTakeNum(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) throws BizException {
        // 奖品发放次数
        fillPrizeLimitSurplusTakeNum(prizeList, detailDto, userInfoDto, now);
        // 填充白名单发放次数
        fillPrizeWhiteListSurplusTakeNum(prizeList, blackWhiteList);
    }

    private static void fillPrizeWhiteListSurplusTakeNum(List<ActivityCommonUserPrizeResultDto> prizeList, List<ActivityBlackWhiteListCacheDto> blackWhiteList) {
        // key：白名单主键，value：白名单详情
        Map<Long, ActivityBlackWhiteListCacheDto> whiteListMap = blackWhiteList.stream().collect(Collectors.toMap(ActivityBlackWhiteListCacheDto::getId, Function.identity()));
        // 过滤掉不在白名单内的
        prizeList.stream().filter(prize -> BooleanUtils.isTrue(prize.getPrizeCanTake().getInWhiteList())).collect(Collectors.toList()).forEach(prize -> {
            if (prize.getPrizeCanTake().getWhiteListId() == null || prize.getPrizeCanTake().getWhiteListId() <= 0) {
                // 没有拿到有次数的白名单，该奖项不可领取
                prize.setSurplusTakeNum(0L);
                return;
            }
            ActivityBlackWhiteListCacheDto whiteListCacheDto = whiteListMap.get(prize.getPrizeCanTake().getWhiteListId());
            prize.setGrantAmount(whiteListCacheDto.getGrantAmount());
            if (prize.getSurplusTakeNum() == null || prize.getSurplusTakeNum() < 0) {
                // 之前没有回填过剩余次数，直接填写白名单剩余次数
                prize.setSurplusTakeNum(whiteListCacheDto.getTotalNum() - whiteListCacheDto.getUsedNum());
                return;
            }
            // 取之前可领取次数与当前白名单剩余次数较小的一个
            prize.setSurplusTakeNum(Math.min(prize.getSurplusTakeNum(), whiteListCacheDto.getTotalNum() - whiteListCacheDto.getUsedNum()));
        });
    }

    private static void fillPrizeLimitSurplusTakeNum(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, Date now) throws BizException {
        // key：奖品ID，value：领取记录
        Map<Long, List<ActivityPrizeReceiveRecordCacheDto>> recordMap = userInfoDto.getReceiveRecordList().stream().collect(Collectors.groupingBy(ActivityPrizeReceiveRecordCacheDto::getActivityPrizeId));
        // key：奖品ID，value：领取统计对象
        Map<Long, UserTakeStatCacheDto> statMap = userInfoDto.getUserTakeStatList().stream().collect(Collectors.toMap(UserTakeStatCacheDto::getPrizeId, Function.identity()));
        // key：奖品ID，value：奖品配置
        Map<Long, ActivityCommonPrizeDetailDto> prizeMap = detailDto.getPrizeList().stream().collect(Collectors.toMap(ActivityCommonPrizeDetailDto::getId, Function.identity()));
        for (ActivityCommonUserPrizeResultDto prize : prizeList) {
            UserTakeStatCacheDto stat = statMap.get(prize.getPrizeId());
            if (stat != null) {
                prize.getPrizeCanTake().setUserTakeStatId(stat.getId());
            }
            List<ActivityPrizeReceiveRecordCacheDto> receiveRecordList = recordMap.get(prize.getPrizeId());
            prize.getPrizeCanTake().setTakeCount(receiveRecordList == null ? 0L : receiveRecordList.size());
            prize.getPrizeCanTake().setLastRecordId(getMaxId(receiveRecordList));
            ActivityCommonPrizeDetailDto prizeDetail = prizeMap.get(prize.getPrizeId());
            if (prizeDetail.getTakeNumLimit() == null || !BooleanEnum.TRUE.getCode().equals(prizeDetail.getTakeNumLimit().getEnable())) {
                return;
            }
            prize.setSurplusTakeNum(getSurplusNum(prize.getSurplusTakeNum(), prizeDetail.getTakeNumLimit(), receiveRecordList, now));
        }
    }

    private static Long getSurplusNum(Long surplusTakeNum, ActivityCommonTakeLimitConfDto takeLimitConfDto, List<ActivityPrizeReceiveRecordCacheDto> receiveRecordList, Date now) throws BizException {
        List<ActivityCommonTakeLimitConfDetailDto> detailList = takeLimitConfDto.getDetailList();
        long surplusNum = surplusTakeNum == null ? -1 : surplusTakeNum;
        for (ActivityCommonTakeLimitConfDetailDto detail : detailList) {
            ActivityPeriodicUnitEnum periodicUnitEnum = ActivityPeriodicUnitEnum.getByCode(detail.getPeriodicUnit());
            if (periodicUnitEnum == null) {
                throwBiz(ActivityCommonErrorCodeEnum.AC000015);
                return null;
            }
            ActivityPeriodicDto periodicDto = periodicUnitEnum.getDate(now, detail.getAbsoluteStartTime(), detail.getAbsoluteEndTime());
            if (periodicDto == null) {
                throwBiz(ActivityCommonErrorCodeEnum.AC000015);
                return null;
            }
            long usedNum = getUsedNum(periodicDto, receiveRecordList);
            long periodicSurplusNum = detail.getTakeNum() - usedNum;
            if (periodicSurplusNum <= 0) {
                // 当前周期没有可领取次数了
                surplusNum = 0;
            } else if (surplusNum < 0) {
                // 之前是不限制领取次数，直接改为当前周期剩余次数
                surplusNum = periodicSurplusNum;
            } else {
                // 取最小值
                surplusNum = Math.min(surplusNum, periodicSurplusNum);
            }
        }
        return surplusNum;
    }

    private static long getUsedNum(ActivityPeriodicDto periodicDto, List<ActivityPrizeReceiveRecordCacheDto> receiveRecordList) {
        if (receiveRecordList == null) {
            return 0;
        }
        if (periodicDto.getBeginDate() == null || periodicDto.getEndDate() == null) {
            return receiveRecordList.size();
        }
        return receiveRecordList.stream().filter(r -> periodicDto.getBeginDate().before(r.getGmtCreate()) && periodicDto.getEndDate().after(r.getGmtCreate())).count();
    }

    private static Long getMaxId(List<ActivityPrizeReceiveRecordCacheDto> receiveRecordList) {
        if (receiveRecordList == null) {
            return null;
        }
        ActivityPrizeReceiveRecordCacheDto maxIdDto = receiveRecordList.stream().max(Comparator.comparing(ActivityPrizeReceiveRecordCacheDto::getId)).orElse(null);
        if (maxIdDto == null) {
            return null;
        }
        return maxIdDto.getId();
    }

    private static List<ActivityCommonUserPrizeResultDto> buildBasePrizeList(ActivityCommonDetailDto detailDto) {
        // key：父级奖品ID，value：子奖品
        Map<Long, List<ActivityCommonPrizeDetailDto>> parentPrizeMap = detailDto.getPrizeList().stream().collect(Collectors.groupingBy(ActivityCommonPrizeDetailDto::getParentPrizeId));
        return detailDto.getPrizeList().stream().map(prizeDetailDto -> {
            // 若奖品类型是子奖品，则返回null
            if (ActivityPrizeTypeEnum.EQUITY_SUB.getType().equals(prizeDetailDto.getPrizeType())) {
                return null;
            }
            ActivityCommonUserPrizeResultDto resultDto = new ActivityCommonUserPrizeResultDto();
            resultDto.setPrizeId(prizeDetailDto.getId());
            resultDto.setPrizeCanTake(new ActivityCommonUserResultCanTakeDto());
            resultDto.getPrizeCanTake().setCanParticipate(true);
            resultDto.getPrizeCanTake().setParticipateErrCode("");
            resultDto.setSurplusTakeNum(-1L);
            List<ActivityCommonPrizeDetailDto> subList = parentPrizeMap.get(prizeDetailDto.getId());
            if (CollectionUtils.isEmpty(subList)) {
                return resultDto;
            }
            List<ActivityCommonUserResultSubPrizeDto> subPrizeList = subList.stream().map(subPrize -> {
                ActivityCommonUserResultSubPrizeDto subResultDto = new ActivityCommonUserResultSubPrizeDto();
                subResultDto.setPrizeId(subPrize.getId());
                return subResultDto;
            }).collect(Collectors.toList());
            resultDto.setSubPrizeList(subPrizeList);
            return resultDto;
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private static void fillPrizeParticipate(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) {
        // 白名单
        checkPrizeWhiteList(prizeList, detailDto, blackWhiteList, now);
        // 达标
        checkPrizeReachStandard(prizeList, detailDto, userInfoDto, blackWhiteList, now);
    }

    private static void checkPrizeReachStandard(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) {
        ActivityCommonReachStandardConfDto reachStandardConf = detailDto.getReachStandardConf();
        if (reachStandardConf == null || !BooleanEnum.TRUE.getCode().equals(reachStandardConf.getEnable())) {
            return;
        }
        if (ActivityCommonReachStandardTypeEnum.WHITE_LIST.getType().equals(reachStandardConf.getType())) {
            // 白名单达标
            checkPrizeReachStandardByWhiteList(prizeList, reachStandardConf, blackWhiteList, now);
            return;
        }
        // API接口达标
        checkPrizeReachStandardByApi(prizeList, reachStandardConf, userInfoDto);
    }

    private static void checkPrizeReachStandardByApi(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonReachStandardConfDto reachStandardConf, ActivityCommonUserInfoCacheDto userInfoDto) {
        ActivityCommonReachStandardDto apiReachStandard = userInfoDto.getApiReachStandard();
        if (apiReachStandard == null || BooleanUtils.isNotTrue(userInfoDto.getApiReachStandard().getSuccess())) {
            prizeAllCanNotParticipate(prizeList, ActivityCommonErrorCodeEnum.AC002018);
            return;
        }
        if (apiReachStandard.getLevel() == null) {
            prizeAllCanNotParticipate(prizeList, ActivityCommonErrorCodeEnum.AC002019);
            return;
        }
        Map<Integer, List<Long>> levelPrizeIdsMap = reachStandardConf.getApiList() == null ? Collections.emptyMap() : reachStandardConf.getApiList().stream().collect(Collectors.toMap(ActivityCommonReachStandardApiConfDto::getLevel, ActivityCommonReachStandardApiConfDto::getPrizeIds));
        List<Long> prizeIds = levelPrizeIdsMap.get(apiReachStandard.getLevel());
        if (CollectionUtils.isEmpty(prizeIds)) {
            prizeAllCanNotParticipate(prizeList, ActivityCommonErrorCodeEnum.AC002019);
            return;
        }
        // 过滤掉已经不可参与的，再判断是否达标
        prizeList.stream().filter(resultDto -> BooleanUtils.isNotFalse(resultDto.getPrizeCanTake().getCanParticipate())).collect(Collectors.toList()).forEach(resultDto -> {
            if (!prizeIds.contains(resultDto.getPrizeId())) {
                fillPrizeParticipateErr(resultDto, ActivityCommonErrorCodeEnum.AC002020.name());
            }
        });
    }

    private static void prizeAllCanNotParticipate(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonErrorCodeEnum errorCodeEnum) {
        // 过滤掉已经不可参与的
        prizeList.stream().filter(resultDto -> BooleanUtils.isNotFalse(resultDto.getPrizeCanTake().getCanParticipate())).collect(Collectors.toList())
                .forEach(resultDto ->
                        fillPrizeParticipateErr(resultDto, errorCodeEnum.name())
                );
    }

    private static void checkPrizeReachStandardByWhiteList(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonReachStandardConfDto reachStandardConf, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) {
        // key：达标金额，value：白名单
        Map<Long, List<ActivityBlackWhiteListCacheDto>> prizeWhiteListMap = blackWhiteList.stream().filter(list -> isEffectiveWhiteList(list, now)).collect(Collectors.groupingBy(ActivityBlackWhiteListCacheDto::getReachStandardAmount));
        // key：奖品ID，value：达标金额列表
        Map<Long, List<Long>> reachStandardWhiteListMap = reachStandardConf.getWhiteListList() == null ? Collections.emptyMap() : reachStandardConf.getWhiteListList().stream().collect(Collectors.groupingBy(ActivityCommonReachStandardWhiteListConfDto::getPrizeId, Collectors.mapping(ActivityCommonReachStandardWhiteListConfDto::getAmount, Collectors.toList())));
        // 过滤掉已经不可参与的，再判断是否达标
        prizeList.stream().filter(resultDto -> BooleanUtils.isNotFalse(resultDto.getPrizeCanTake().getCanParticipate())).collect(Collectors.toList()).forEach(resultDto -> {
            List<Long> amountList = reachStandardWhiteListMap.get(resultDto.getPrizeId());
            // 达标金额，对应的白名单列表
            List<ActivityBlackWhiteListCacheDto> whiteList = amountList.stream().filter(prizeWhiteListMap::containsKey).flatMap(amount -> prizeWhiteListMap.get(amount).stream()).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(whiteList)) {
                fillPrizeParticipateErr(resultDto, ActivityCommonErrorCodeEnum.AC002017.name());
            } else {
                // 获取可领取的、最早过期的奖品白名单ID
                Long earliestExpiredWhitelist = getEarliestExpiredWhitelist(whiteList);
                resultDto.getPrizeCanTake().setWhiteListId(earliestExpiredWhitelist);
                resultDto.getPrizeCanTake().setInWhiteList(true);
            }
        });
    }

    private static void checkPrizeWhiteList(List<ActivityCommonUserPrizeResultDto> prizeList, ActivityCommonDetailDto detailDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) {
        ActivityCommonWhiteConfDto whiteConf = detailDto.getWhiteConf();
        if (whiteConf == null || !BooleanEnum.TRUE.getCode().equals(whiteConf.getEnable())) {
            return;
        }
        if (ActivityCommonDimensionEnum.ACTIVITY.getDimension().equals(whiteConf.getDimension())) {
            // 活动纬度，不影响奖品访问权限
            return;
        }
        // key：奖品ID，value：白名单
        Map<Long, List<ActivityBlackWhiteListCacheDto>> prizeWhiteListMap = blackWhiteList.stream().filter(list -> isEffectiveWhiteList(list, now)).collect(Collectors.groupingBy(ActivityBlackWhiteListCacheDto::getPrizeId));
        // 过滤掉已经不可参与的，再判断白名单是否可参与
        prizeList.stream().filter(resultDto -> BooleanUtils.isNotFalse(resultDto.getPrizeCanTake().getCanParticipate())).collect(Collectors.toList()).forEach(resultDto -> {
            List<ActivityBlackWhiteListCacheDto> whiteList = prizeWhiteListMap.get(resultDto.getPrizeId());
            if (CollectionUtils.isEmpty(whiteList)) {
                fillPrizeParticipateErr(resultDto, ActivityCommonErrorCodeEnum.AC002016.name());
            } else {
                // 获取可领取的、最早过期的奖品白名单ID
                Long earliestExpiredWhitelist = getEarliestExpiredWhitelist(whiteList);
                resultDto.getPrizeCanTake().setWhiteListId(earliestExpiredWhitelist);
                resultDto.getPrizeCanTake().setInWhiteList(true);
            }
        });
    }

    private static Long getEarliestExpiredWhitelist(List<ActivityBlackWhiteListCacheDto> whiteList) {
        ActivityBlackWhiteListCacheDto whiteListCacheDto = whiteList.stream().filter(list -> list.getTotalNum() > list.getUsedNum()).min(Comparator.comparing(ActivityBlackWhiteListCacheDto::getEndTime)).orElse(null);
        if (whiteListCacheDto == null) {
            return null;
        }
        return whiteListCacheDto.getId();
    }

    private static void fillPrizeParticipateErr(ActivityCommonUserPrizeResultDto resultDto, String code) {
        if (ActivityPrizeTypeEnum.THANKS.getType().equals(resultDto.getPrizeType())) {
            // 谢谢参与，所有用户都有该奖品权限
            return;
        }
        resultDto.getPrizeCanTake().setCanParticipate(false);
        resultDto.getPrizeCanTake().setParticipateErrCode(code);
    }

    private static void fillSignUp(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, List<ActivitySignUpCacheDto> signUpList, Date now) throws BizException {
        if (detailDto.getSignUpConf() == null) {
            resultDto.setCanSignedUp(false);
            return;
        }
        if (!BooleanEnum.TRUE.getCode().equals(detailDto.getSignUpConf().getEnable())) {
            resultDto.setCanSignedUp(false);
            return;
        }
        if (BooleanUtils.isNotTrue(resultDto.getCanAccess())) {
            // 活动不可访问
            resultDto.setCanSignedUp(false);
            return;
        }
        ActivityPeriodicDto nowPeriodic = getNowPeriodic(now, detailDto.getSignUpConf());
        if (nowPeriodic == null) {
            // 当前不在周期内
            resultDto.setCanSignedUp(false);
            return;
        }
        resultDto.setCanSignedUp(true);
        Pair<Boolean, Long> pair = getSignUpAndId(nowPeriodic, signUpList);
        resultDto.setSignedUp(pair.getLeft());
        resultDto.setSignUpId(pair.getRight());
    }

    private static Pair<Boolean, Long> getSignUpAndId(ActivityPeriodicDto periodic, List<ActivitySignUpCacheDto> signUpList) {
        if (CollectionUtils.isEmpty(signUpList)) {
            return Pair.of(false, null);
        }
        if (periodic.getBeginDate() == null || periodic.getEndDate() == null) {
            return Pair.of(true, signUpList.get(0).getId());
        }
        List<ActivitySignUpCacheDto> inTimeList = signUpList.stream().filter(cache -> periodic.getBeginDate().before(cache.getGmtCreate()) && periodic.getEndDate().after(cache.getGmtCreate())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(inTimeList)) {
            return Pair.of(false, null);
        }
        return Pair.of(true, inTimeList.get(0).getId());
    }

    private static ActivityPeriodicDto getNowPeriodic(Date now, ActivityCommonSignUpConfDto signUpConf) throws BizException {
        ActivityPeriodicUnitEnum periodicUnitEnum = ActivityPeriodicUnitEnum.getByCode(signUpConf.getPeriodicUnit());
        if (periodicUnitEnum == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002010);
            return null;
        }
        if (periodicUnitEnum == ActivityPeriodicUnitEnum.ABSOLUTE) {
            return getNowAbsolute(now, signUpConf);
        }
        return periodicUnitEnum.getDate(now, null, null);
    }

    private static ActivityPeriodicDto getLastPeriodic(Date now, ActivityCommonSignUpConfDto signUpConf) throws BizException {
        ActivityPeriodicUnitEnum periodicUnitEnum = ActivityPeriodicUnitEnum.getByCode(signUpConf.getPeriodicUnit());
        if (periodicUnitEnum == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002010);
            return null;
        }
        if (periodicUnitEnum == ActivityPeriodicUnitEnum.ABSOLUTE) {
            return getLastAbsolute(now, signUpConf);
        }
        return periodicUnitEnum.getOffsetPeriod(-1);
    }

    private static ActivityPeriodicDto getNowAbsolute(Date now, ActivityCommonSignUpConfDto signUpConf) {
        if (CollectionUtils.isEmpty(signUpConf.getAbsoluteTimeLimit())) {
            return null;
        }
        // 在当前时间段内，开始时间最晚的
        ActivityCommonTimeLimitAbsoluteConfDto timeLimit = signUpConf.getAbsoluteTimeLimit().stream().filter(time -> time.getAbsoluteStartTime().before(now) && time.getAbsoluteEndTime().after(now)).max(Comparator.comparing(ActivityCommonTimeLimitAbsoluteConfDto::getAbsoluteStartTime)).orElse(null);
        if (timeLimit == null) {
            return null;
        }
        return new ActivityPeriodicDto(null, timeLimit.getAbsoluteStartTime(), timeLimit.getAbsoluteEndTime());
    }

    private static ActivityPeriodicDto getLastAbsolute(Date now, ActivityCommonSignUpConfDto signUpConf) {
        if (CollectionUtils.isEmpty(signUpConf.getAbsoluteTimeLimit())) {
            return null;
        }
        // 上周期 = 结束时间均早于当前时间，且结束时间最晚的
        ActivityCommonTimeLimitAbsoluteConfDto timeLimit = signUpConf.getAbsoluteTimeLimit().stream().filter(time -> time.getAbsoluteEndTime().before(now)).max(Comparator.comparing(ActivityCommonTimeLimitAbsoluteConfDto::getAbsoluteEndTime)).orElse(null);
        if (timeLimit == null) {
            return null;
        }
        return new ActivityPeriodicDto(null, timeLimit.getAbsoluteStartTime(), timeLimit.getAbsoluteEndTime());
    }

    private static boolean fillActivityParticipate(ActivityCommonUserResultDto resultDto, ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) {
        try {
            // 用户校验
            checkUserLimit(detailDto, userInfoDto);
            // 是否在黑名单/白名单内校验
            checkInBlackWhiteList(resultDto.getActivityCanTake(), detailDto, blackWhiteList, now);
            resultDto.getActivityCanTake().setCanParticipate(true);
            resultDto.getActivityCanTake().setParticipateErrCode("");
        } catch (BizException e) {
            resultDto.getActivityCanTake().setCanParticipate(false);
            resultDto.getActivityCanTake().setParticipateErrCode(e.getCode());
            resultDto.setCanSignedUp(false);
        }
        return resultDto.getActivityCanTake().getCanParticipate();
    }

    private static void signUpVerify(ActivityCommonUserResultDto resultDto, ActivityCommonSignUpConfDto signUpConf, ActivityCommonUserInfoCacheDto userInfoDto, Date now) throws BizException {
        if (signUpConf == null) {
            return;
        }
        if (!BooleanEnum.TRUE.getCode().equals(signUpConf.getEnable()) || !BooleanEnum.TRUE.getCode().equals(signUpConf.getEnableVerify())) {
            return;
        }
        if (ActivityCommonSignUpVerifyRuleTypeEnum.CURRENT_PERIOD.getType().equals(signUpConf.getVerifyRule())) {
            // 本期报名，填充报名时已回填
            checkSignUpVerifyCurrentPeriod(resultDto);
            return;
        }
        if (ActivityCommonSignUpVerifyRuleTypeEnum.CUSTOM.getType().equals(signUpConf.getVerifyRule())) {
            // 定制校验
            checkCustomSignUpVerify(resultDto, userInfoDto.getCustomSignUpVerify());
            return;
        }
        // 上期报名
        checkSignUpVerifyLastPeriod(resultDto, signUpConf, userInfoDto.getSignUpList(), now);
    }

    private static void checkSignUpVerifyLastPeriod(ActivityCommonUserResultDto resultDto, ActivityCommonSignUpConfDto signUpConf, List<ActivitySignUpCacheDto> signUpList, Date now) throws BizException {
        ActivityPeriodicDto lastPeriodic = getLastPeriodic(now, signUpConf);
        if (lastPeriodic == null) {
            // 无上周期
            throwBiz(ActivityCommonErrorCodeEnum.AC002014);
            return;
        }
        if (CollectionUtils.isEmpty(signUpList)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002015);
            return;
        }
        if (lastPeriodic.getBeginDate() == null || lastPeriodic.getEndDate() == null) {
            resultDto.setSignUpId(signUpList.get(0).getId());
            return;
        }
        List<ActivitySignUpCacheDto> inTimeList = signUpList.stream().filter(cache -> lastPeriodic.getBeginDate().before(cache.getGmtCreate()) && lastPeriodic.getEndDate().after(cache.getGmtCreate())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(inTimeList)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002015);
            return;
        }
        resultDto.setSignUpId(inTimeList.get(0).getId());
    }

    private static void checkSignUpVerifyCurrentPeriod(ActivityCommonUserResultDto resultDto) throws BizException {
        // 本期报名，填充报名时已回填，无需再回填ID
        if (BooleanUtils.isNotTrue(resultDto.getSignedUp())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002013);
        }
    }

    private static void checkCustomSignUpVerify(ActivityCommonUserResultDto resultDto, ActivityCommonCustomSignUpVerifyDto customSignUpVerify) throws BizException {
        if (customSignUpVerify == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002011);
            return;
        }
        if (ActivityCommonUserVerifyStatusEnum.PASSED.getStatus().equals(customSignUpVerify.getStatus())) {
            resultDto.setSignUpId(customSignUpVerify.getSignUpId());
            return;
        }
        throwBiz(ActivityCommonErrorCodeEnum.AC002012);
    }

    private static void checkUserLimit(ActivityCommonDetailDto detailDto, ActivityCommonUserInfoCacheDto userInfoDto) throws BizException {
        ActivityCommonUserLimitConfDto userLimit = detailDto.getUserLimit();
        if (userLimit == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002001);
            return;
        }
        checkUserType(userLimit, userInfoDto.getUserDto());
        checkUserApiVerify(userLimit, userInfoDto.getUserApiVerify());
        checkUserCustomVerify(userLimit, userInfoDto.getUserCustomVerify());
    }

    private static void checkUserType(ActivityCommonUserLimitConfDto userLimit, UserDto userDto) throws BizException {
        if (userDto == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC001002);
            return;
        }
        if (CollectionUtils.isEmpty(userLimit.getUserTypeList())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002006);
            return;
        }
        if (!userLimit.getUserTypeList().contains(userDto.getUserType())) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002007);
        }
    }

    private static void checkUserCustomVerify(ActivityCommonUserLimitConfDto userLimit, ActivityCommonUserCustomVerifyDto userCustomVerify) throws BizException {
        if (!userLimit.isNeedCustomVerify()) {
            return;
        }
        if (userCustomVerify == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002004);
            return;
        }
        if (ActivityCommonUserVerifyStatusEnum.PASSED.getStatus().equals(userCustomVerify.getStatus())) {
            return;
        }
        throwBiz(ActivityCommonErrorCodeEnum.AC002005);
    }

    private static void checkUserApiVerify(ActivityCommonUserLimitConfDto userLimit, ActivityCommonUserApiVerifyDto userApiVerify) throws BizException {
        if (!userLimit.isNeedApiVerify()) {
            return;
        }
        if (userApiVerify == null) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002002);
            return;
        }
        if (ActivityCommonUserVerifyStatusEnum.PASSED.getStatus().equals(userApiVerify.getStatus())) {
            return;
        }
        throwBiz(ActivityCommonErrorCodeEnum.AC002003);
    }

    private static void checkInBlackWhiteList(ActivityCommonUserResultCanTakeDto activityCanTake, ActivityCommonDetailDto detailDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) throws BizException {
        checkInBlackList(detailDto, blackWhiteList);
        checkInWhiteList(activityCanTake, detailDto, blackWhiteList, now);
    }

    private static void checkInWhiteList(ActivityCommonUserResultCanTakeDto activityCanTake, ActivityCommonDetailDto detailDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList, Date now) throws BizException {
        ActivityCommonWhiteConfDto whiteConf = detailDto.getWhiteConf();
        if (whiteConf == null) {
            return;
        }
        if (!BooleanEnum.TRUE.getCode().equals(whiteConf.getEnable())) {
            return;
        }
        if (CollectionUtils.isEmpty(blackWhiteList)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002009);
            return;
        }
        List<ActivityBlackWhiteListCacheDto> whiteList = blackWhiteList.stream().filter(list -> isEffectiveWhiteList(list, now)).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(whiteList)) {
            throwBiz(ActivityCommonErrorCodeEnum.AC002009);
            return;
        }
        if (!ActivityCommonDimensionEnum.ACTIVITY.getDimension().equals(whiteConf.getDimension()) || BooleanEnum.TRUE.getCode().equals(whiteConf.getLimitReachStandardAmount())) {
            // 非活动纬度的，或者开启了达标金额的，无需活动纬度白名单ID
            return;
        }
        // 获取可领取的、最早过期的奖品白名单ID
        Long earliestExpiredWhitelist = getEarliestExpiredWhitelist(whiteList);
        activityCanTake.setWhiteListId(earliestExpiredWhitelist);
        activityCanTake.setInWhiteList(true);
    }

    private static boolean isEffectiveWhiteList(ActivityBlackWhiteListCacheDto list, Date now) {
        return ActivityBlackWhiteListTypeEnum.WHITE.getType().equals(list.getListType()) && list.getStartTime().before(now) && list.getEndTime().after(now);
    }

    private static void checkInBlackList(ActivityCommonDetailDto detailDto, List<ActivityBlackWhiteListCacheDto> blackWhiteList) throws BizException {
        ActivityCommonBlackConfDto blackConf = detailDto.getBlackConf();
        if (blackConf == null) {
            return;
        }
        if (!BooleanEnum.TRUE.getCode().equals(blackConf.getEnable())) {
            return;
        }
        if (CollectionUtils.isEmpty(blackWhiteList)) {
            return;
        }
        long count = blackWhiteList.stream().filter(list -> ActivityBlackWhiteListTypeEnum.BLACK.getType().equals(list.getListType())).count();
        if (count < 1) {
            return;
        }
        throwBiz(ActivityCommonErrorCodeEnum.AC002008);
    }

}
