package cn.com.duiba.quanyi.center.api.remoteservice.workflow;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.quanyi.center.api.dto.workflow.ActivityWorkflowDto;
import cn.com.duiba.quanyi.center.api.dto.workflow.WorkflowApprovalDetailDto;
import cn.com.duiba.quanyi.center.api.param.workflow.ActivityWorkflowBatchQueryParam;
import cn.com.duiba.quanyi.center.api.param.workflow.ActivityWorkflowStartApprovalProcessParam;

import java.util.List;

/**
 * 活动审批流
 *
 * @author xuzhigang
 * @date 2025-06-19
 */
@AdvancedFeignClient
public interface RemoteActivityWorkflowService {

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 活动审批流
     */
    ActivityWorkflowDto selectById(Long id);

    /**
     * 根据活动ID和审批流类型查询最新记录
     *
     * @param activityId   活动ID
     * @param activityType 活动类型
     * @param workflowType 审批流类型
     * @return 活动审批流
     * @see cn.com.duiba.quanyi.center.api.enums.workflow.WorkflowTypeEnum
     */
    ActivityWorkflowDto selectLastByActivityId(Long activityId, Integer activityType, Integer workflowType);

    /**
     * 根据活动ID查询审批流列表
     *
     * @param activityId   活动ID
     * @param activityType 活动类型
     * @return 活动审批流列表
     */
    List<ActivityWorkflowDto> selectByActivityId(Long activityId, Integer activityType);

    /**
     * 根据审批状态查询列表
     *
     * @param approvalStatus 审批状态
     * @return 活动审批流列表
     * @see cn.com.duiba.quanyi.center.api.enums.workflow.ApprovalStatusEnum
     */
    List<ActivityWorkflowDto> selectByApprovalStatus(Integer approvalStatus);

    /**
     * 批量查询活动审批流
     *
     * @param param 批量查询参数
     * @return 活动审批流列表
     */
    List<ActivityWorkflowDto> selectByBatchQuery(ActivityWorkflowBatchQueryParam param);

    /**
     * 初始化活动审批流
     *
     * @param activityId                  活动ID
     * @param activityType                活动类型
     * @param workflowType                审批流类型
     * @param initiatorSsoId              发起人sso用户ID
     * @param activityDeveloperSsoId      活动开发sso用户ID
     * @param activityProductManagerSsoId 活动产品sso用户ID
     * @return 活动审批流
     */
    ActivityWorkflowDto initActivityWorkflow(Long activityId, Integer activityType, Integer workflowType,
                                             Long initiatorSsoId, Long activityDeveloperSsoId, Long activityProductManagerSsoId);

    /**
     * 发起活动审批流程
     *
     * @param param 审批参数
     * @return 审批流ID
     * @throws BizException 业务异常
     */
    Long startActivityApprovalProcess(ActivityWorkflowStartApprovalProcessParam param) throws BizException;


    /**
     * 活动审批详情
     * @param activityId 活动ID
     * @param activityType 活动类型
     * @param workflowType 审批流类型
     * @param refresh 是否刷新
     * @return 审批详情
     */
    WorkflowApprovalDetailDto getActivityApprovalDetail(Long activityId, Integer activityType, Integer workflowType, boolean refresh);


    /**
     * 更新审批状态
     * @param id 主键
     * @param beforeStatus 原状态
     * @param afterStatus 更改后状态
     * @param errorMsg 失败原因
     * @return 影响行数
     */
    int updateApprovalStatus(Long id, Integer beforeStatus, Integer afterStatus, String errorMsg);


    /**
     * 重置审批状态，从审批成功恢复成待提交，可以重新审批
     * @param id 主键
     * @return 影响行数
     */
    boolean resetApprovalStatus(Long id);
}
