package cn.com.duiba.quanyi.center.api.utils.activity;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import cn.com.duiba.quanyi.center.api.dto.activity.common.ext.collection.ActivityCommonCollectionPageParticipateLimitConfDto;
import cn.com.duiba.quanyi.center.api.enums.activity.MultiSelectNDimensionEnum;

/**
 * 维度值计算工具类
 * 根据维度类型计算对应的维度值
 * 例如：维度为日，维度值为yyyymmdd；维度为月，维度值为yyymm
 *
 * @author yangyukang
 * @date 2025-12-08 20:55
 */
public class DimensionValueUtils {

    /**
     * 根据维度类型和日期计算维度值
     *
     * @param dimension 维度类型（不能为null）
     * @param date 日期（如果为null则使用当前时间）
     * @param dimensionConf 维度配置（自定义维度时需要）
     * @return 维度值（不能为null，永久返回'0'），格式：
     *         永久（不限制）：0
     *         日维度：20250120 (yyyyMMdd)
     *         自然周：2025W03 (yyyyWww)
     *         自然月：202512 (yyyyMM)
     *         自然季度：2025Q4 (yyyyQq)
     *         自定义：custom_yyyymmdd-yyyymmdd（在有效期内）或0（不在有效期内）
     */
    public static String calculateDimensionValue(Integer dimension, Date date, ActivityCommonCollectionPageParticipateLimitConfDto dimensionConf) {
        if (dimension == null) {
            // 如果dimension为null，返回永久的period值（'0'，不限制和永久合并）
            return "0";
        }

        Date targetDate = date != null ? date : new Date();
        MultiSelectNDimensionEnum dimensionEnum = MultiSelectNDimensionEnum.getByCode(dimension);
        if (dimensionEnum == null) {
            // 未知的维度类型，返回永久的period值（'0'，不限制和永久合并）
            return "0";
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(targetDate);
        int year = calendar.get(Calendar.YEAR);

        String period;
        switch (dimensionEnum) {
            case PERMANENT:
                // 不限制和永久合并，统一使用0表示永久
                period = "0";
                break;
            case DAY:
                // 日维度：20250120 (yyyyMMdd)
                SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
                period = dayFormat.format(targetDate);
                break;
            case WEEK:
                // 自然周：2025W03（ISO 8601标准，每年第一周是包含1月4日的那一周）
                int weekOfYear = calendar.get(Calendar.WEEK_OF_YEAR);
                period = String.format("%dW%02d", year, weekOfYear);
                break;
            case MONTH:
                // 自然月：202512 (yyyyMM)
                int month = calendar.get(Calendar.MONTH) + 1;
                period = String.format("%d%02d", year, month);
                break;
            case QUARTER:
                // 自然季度：2025Q4 (yyyyQq)
                int monthForQuarter = calendar.get(Calendar.MONTH) + 1;
                int quarter = (monthForQuarter - 1) / 3 + 1;
                period = String.format("%dQ%d", year, quarter);
                break;
            case CUSTOM:
                // 自定义维度：如果当前时间在自定义时间范围内，返回固定标识；否则返回'0'
                if (dimensionConf != null && dimensionConf.getCustomStartTime() != null && dimensionConf.getCustomEndTime() != null) {
                    Date startTime = dimensionConf.getCustomStartTime();
                    Date endTime = dimensionConf.getCustomEndTime();
                    if (targetDate.compareTo(startTime) >= 0 && targetDate.compareTo(endTime) <= 0) {
                        // 在自定义时间范围内，使用开始时间和结束时间作为周期标识，格式：custom_yyyymmdd-yyyymmdd
                        SimpleDateFormat customFormat = new SimpleDateFormat("yyyyMMdd");
                        period = "custom_" + customFormat.format(startTime) + "-" + customFormat.format(endTime);
                    } else {
                        // 不在自定义时间范围内，返回'0'
                        period = "0";
                    }
                } else {
                    // 没有自定义时间配置，返回'0'
                    period = "0";
                }
                break;
            default:
                period = "0";
                break;
        }
        // 确保返回值不为null
        return period != null ? period : "0";
    }

    /**
     * 计算当前时间的维度值
     *
     * @param dimension 维度类型（不能为null）
     * @param dimensionConf 维度配置（自定义维度时需要）
     * @return 维度值（不能为null，永久返回'0'）
     */
    public static String calculateCurrentDimensionValue(Integer dimension, ActivityCommonCollectionPageParticipateLimitConfDto dimensionConf) {
        // 确保返回值不为null
        return calculateDimensionValue(dimension, new Date(), dimensionConf);
    }

    /**
     * 判断自定义维度是否在有效期内
     * 
     * @param dimensionConf 维度配置
     * @return 是否在有效期内
     */
    public static boolean isCustomDimensionValid(ActivityCommonCollectionPageParticipateLimitConfDto dimensionConf) {
        if (dimensionConf == null || dimensionConf.getCustomStartTime() == null || dimensionConf.getCustomEndTime() == null) {
            return false;
        }
        Date now = new Date();
        return now.compareTo(dimensionConf.getCustomStartTime()) >= 0 && now.compareTo(dimensionConf.getCustomEndTime()) <= 0;
    }
}

