package cn.com.duiba.cloud.single.sign.on.client.handler;

import cn.com.duiba.cloud.single.sign.on.contract.common.exception.SsoRunTimeException;
import cn.com.duiba.cloud.single.sign.on.contract.constants.SsoProperties;
import cn.com.duiba.cloud.single.sign.on.contract.interceptor.handler.SsoFilterHandler;
import cn.com.duiba.cloud.single.sign.on.contract.tool.CookieUtil;
import cn.com.duiba.cloud.single.sign.on.contract.tool.SsoRequestTool;

import cn.com.duiba.cloud.single.sign.on.client.cache.LoginStateCache;
import cn.com.duiba.cloud.single.sign.on.client.domain.dto.LoginStateDto;
import cn.com.duiba.cloud.single.sign.on.client.remoteservice.RemoteSsoService;

import cn.com.duiba.cloud.single.sign.on.client.tool.SsoContext;
import cn.com.duiba.wolf.utils.UrlUtils2;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Objects;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;


/**
 * @author xuanhongjian 登录拦截逻辑处理
 */
@Component
@Slf4j
public class LoginFilterHandler implements SsoFilterHandler {

    @Resource
    private SsoProperties ssoProperties;

    @Resource
    private RemoteSsoService remoteSsoService;

    @Resource
    private LoginStateCache loginStateCache;

    @Override
    public Boolean before(Object handler) {
        String ticket = SsoContext.findTicket();
        if (StringUtils.isBlank(ticket)) {
            returnNotLogin();
            return false;
        }
        LoginStateDto loginState = verifyTicket(ticket);
        if (loginState == null) {
            returnNotLogin();
            return false;
        }
        SsoContext.setLoginState(loginState);
        return true;
    }

    @Override
    public int getOrder() {
        return 0;
    }

    protected void returnNotLogin() {
        try {
            if (SsoRequestTool.isAsynchronousRequests()) {
                SsoRequestTool.getResponse()
                    .setHeader("Content-Type", "application/json;charset=UTF-8");
                JSONObject model = new JSONObject();
                model.put("code", "SSO:01001");
                model.put("success", false);
                model.put("notLogin", true);
                model.put("message", "登录失效，请刷新页面");
                SsoRequestTool.getResponse().getWriter().write(model.toJSONString());
            } else {
                //附上当前请求地址,重定向到SSO登录页
                SsoRequestTool.getResponse().sendRedirect(getLoginRedirectUrl());
            }
        } catch (Exception e) {
            log.error("获取地址错误",e);
            throw new SsoRunTimeException(e);
        }
    }


    /**
     * 获取登录重定向地址
     */
    private String getLoginRedirectUrl() {
        Map<String, String> params = Maps.newHashMap();
        String redirect = getCurrentRedirect();
        params.put("redirect", redirect);
        String ssoHomeUrl = remoteSsoService.findSsoHomeUrl();
        String loginUrl = ssoHomeUrl + "/account/login";
        return UrlUtils2.appendParams(loginUrl, params);
    }

    /**
     * 把当前请求包装成重定向参数
     */
    protected String getCurrentRedirect() {
        HttpServletRequest request = SsoRequestTool.getRequest();
        String defaultUri = "/";
        String redirect;
        if (StringUtils.equals(defaultUri, request.getRequestURI())) {
            redirect = SsoRequestTool.getHomeUrl() + getDefaultRedirectUrl();
        } else {
            redirect = SsoRequestTool.getUrl();
        }
        try {
            return URLEncoder.encode(redirect, "utf-8");
        } catch (Exception e) {
            throw new SsoRunTimeException(e);
        }
    }


    private String getDefaultRedirectUrl() {
            return ssoProperties.getDefaultRedirectUrl();
    }

    /**
     * 验证通行证
     */
    protected LoginStateDto verifyTicket(String ticket){
        if(StringUtils.isBlank(ticket)){
            return null;
        }
        LoginStateDto loginState = loginStateCache.get(ticket);

        if(Objects.isNull(loginState)){
            //向sso 获取登录信息
            loginState = remoteSsoService.verifyTicket(ticket);
            if(loginState == null){
                return null;
            }
            loginStateCache.set(ticket,loginState,(long) (10*60));
        }
        //有可能是根据链接传入的通行证，把它持久化到客户端的cookie中
        if(StringUtils.isBlank(SsoRequestTool.getCookie(CookieUtil.LOGIN_COOKIE_NAME))){
            CookieUtil.setLoginCookie(ticket);
        }
        return loginState;
    }
}
