package cn.com.duiba.cloud.single.sign.on.client.wbe;


import cn.com.duiba.cloud.single.sign.on.client.cache.LoginStateCache;
import cn.com.duiba.cloud.single.sign.on.client.domain.dto.LoginStateDto;
import cn.com.duiba.cloud.single.sign.on.client.domain.request.OutLoginParam;
import cn.com.duiba.cloud.single.sign.on.client.remoteservice.RemoteSsoService;
import cn.com.duiba.cloud.single.sign.on.client.tool.SsoContext;
import cn.com.duiba.cloud.single.sign.on.contract.common.exception.SsoException;
import cn.com.duiba.cloud.single.sign.on.contract.constants.SsoProperties;
import cn.com.duiba.cloud.single.sign.on.contract.interceptor.annotation.CanAccess;
import cn.com.duiba.cloud.single.sign.on.contract.tool.CookieUtil;
import cn.com.duiba.cloud.single.sign.on.contract.tool.JsonRender;
import cn.com.duiba.cloud.single.sign.on.contract.tool.SsoRequestTool;
import cn.com.duibaboot.ext.autoconfigure.web.login.LoginSuccessEvent;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Resource;
import javax.validation.Valid;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

/**
 * @author xuanhongjian
 */
@Slf4j
@Controller
@RequestMapping("/sso")
public class SsoController {

    @Resource
    private RemoteSsoService remoteSsoService;

    @Resource
    private SsoProperties properties;

    @Resource
    private ApplicationContext applicationContext;

    @Resource
    private LoginStateCache loginStateCache;

    @Resource
    private Environment environment;


    @CanAccess
    @RequestMapping("/ssoIndex")
    public String ssoIndex(@RequestParam Long ssoStamp,
        @RequestParam(required = false) String redirect) throws SsoException {

        String ticket = remoteSsoService.findTicketByStamp(ssoStamp);
        if (StringUtils.isBlank(ticket)) {
            throw new SsoException("获取SSO通行证失败");
        }
        LoginStateDto loginState = remoteSsoService.verifyTicket(ticket);
        if (loginState == null) {
            throw new SsoException("ticket失效");
        }
        SsoContext.setLoginState(loginState);
        CookieUtil.setLoginCookie(ticket);
        LoginSuccessEvent event = new LoginSuccessEvent();
        event.setSession(loginState);
        event.setRequest(SsoRequestTool.getRequest());
        event.setResponse(SsoRequestTool.getResponse());
        event.setExpirationTime(15*36*24*3600);
        applicationContext.publishEvent(event);
        return "redirect:" + Optional.ofNullable(redirect)
                                 .orElse(properties.getDefaultRedirectUrl());
    }


    @RequestMapping(value = "/outLogin")
    public JsonRender<Void> outLogin(@RequestParam(defaultValue = "/") String redirect) {
        //清除本地缓存
        String ticket = SsoContext.findTicket();
        loginStateCache.invalidateAll(Lists.newArrayList(ticket));
        remoteSsoService.outLogin(ticket);
        return JsonRender.success();
    }

    @CanAccess
    @ResponseBody
    @RequestMapping(value = "/logout")
    public JsonRender<Map<String,String>> logout(@RequestBody @Valid OutLoginParam param) {
        loginStateCache.invalidateAll(param.getTickets());
        //获取当前系统名称和ip
        HashMap<String, String> appInfo = Maps.newHashMap();
        appInfo.put("name", environment.getProperty("spring.application.name", ""));
        appInfo.put("port", environment.getProperty("server.port", ""));
        try {
            appInfo.put("ip", InetAddress.getLocalHost().getHostAddress());
        } catch (UnknownHostException e) {
            log.error("获取本地ip失败", e);
        }
        return JsonRender.success(appInfo);
    }
}
