package cn.com.duiba.sso.api.web.factory;

import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.web.annotation.SsoComponent;
import cn.com.duiba.wolf.utils.ClassUtils;
import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.*;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.AnnotationUtils;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by liuyao on 2017/6/19.
 * 单例模式
 */
public class SsoBeanFactory {

    private static SsoBeanFactory ssoBeanFactory;

    private SsoContext context;

    private ApplicationContext applicationContext;

    private Map<Class<?>,Object> classMap = Maps.newConcurrentMap();

    private SsoBeanFactory(){}

    public static SsoBeanFactory newInstance(SsoContext context){
        try{
            if(SsoBeanFactory.ssoBeanFactory!=null){
                return ssoBeanFactory;
            }
            SsoBeanFactory.ssoBeanFactory = new SsoBeanFactory();
            SsoBeanFactory.ssoBeanFactory.setSsoContext(context);
            return SsoBeanFactory.ssoBeanFactory;
        }catch (Exception e){
            throw new SsoRunTimeException("SSO客户端初始化异常",e);
        }

    }

    public void setSsoContext(SsoContext context){
        this.context = context;
        this.applicationContext = context.getApplicationContext();
    }

    public <T> T getBean(Class<T> clazz){
        try{
            if(classMap.containsKey(clazz)){
                return (T)classMap.get(clazz);
            }
            return applicationContext.getBean(clazz);
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

    protected void scan(){
        try{
            //找出api下面所有带有@SsoComponent的类
            List<Class<?>> classes = ClassUtils.getClassList("cn.com.duiba.sso.api",true,null);
            classes = filterClassByAnnotation(classes,SsoComponent.class);
            //初始化类
            for(Class<?> clazz:classes){
                if(!classMap.containsKey(clazz)){
                    classMap.put(clazz,clazz.newInstance());
                }
            }
            //注入依赖
            for(Class<?> clazz:classes){
                Object bean = classMap.get(clazz);
                if(bean instanceof SsoBeanAware){
                    SsoBeanAware ssoBeanAware = (SsoBeanAware)bean;
                    ssoBeanAware.setContext(context);
                }
            }
        }catch (Exception e){
            throw new SsoRunTimeException("Sso初始化容器失败",e);
        }

    }

    private List<Class<?>> filterClassByAnnotation(List<Class<?>> classes,Class<? extends Annotation> annotationType){
        List<Class<?>> newList = Lists.newArrayList();
        for(Class<?> clazz:classes){
            Annotation annotation = AnnotationUtils.findAnnotation(clazz, annotationType);
            if(!Objects.equal(annotation,null)){
                newList.add(clazz);
            }
        }
        return newList;
    }

    public Set<Object> getBeansWithAnnotation(Class<? extends Annotation> annotationType) {
        Map<String, Object> bean = applicationContext.getBeansWithAnnotation(annotationType);
        Set<Object> beans = Sets.newHashSet();
        beans.addAll(bean.values());
        Set<Class<?>> classes = classMap.keySet();
        List<Object> beanList = Lists.transform(Lists.newArrayList(classes), new Function<Class<?>, Object>() {
            @Override
            public Object apply(Class<?> input) {
                return classMap.get(input);
            }
        });
        beans.addAll(beanList);
        return beans;
    }
}
