package cn.com.duiba.crecord.message;

import java.util.Arrays;
import java.util.Collections;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.message.KafkaClient;
import cn.com.duiba.message.QueueConfig;

/**
 * 兑换记录消费者<br/>
 */
@Service
public class CrecordConsumer implements InitializingBean {

	private static Logger log = LoggerFactory.getLogger(CrecordConsumer.class);

	@Autowired
	private QueueConfig queueConfig;
	@Autowired
	private KafkaClient kafkaClient;
	@Autowired
	private CrecordAsyncHttp crecordAsyncHttp;

	private static int pool = 5;

	private ExecutorService executorService;

	@Override
	public void afterPropertiesSet() {
		start();
	}

	public synchronized void start() {
		if (executorService != null) {
			return;
		}
		executorService = Executors.newFixedThreadPool(pool, new ThreadFactory() {
			private int i = 0;

			@Override
			public Thread newThread(Runnable runnable) {
				return new Thread(runnable, "Thread-Crecord-Consumer-" + (i++));
			}
		});
		for (int i = 0; i < pool; i++) {
			executorService.submit(new Runnable() {
				@Override
				public void run() {
					whilePoll();
				}
			});
		}
		log.info("CrecordConsumer started");
	}

	private void whilePoll() {
		KafkaConsumer<String, String> consumer = null;
		try {
			Properties props = new Properties();
			props.put("bootstrap.servers", kafkaClient.getBootstrapServers());
			props.put("group.id", "notify-center");
			props.put("enable.auto.commit", "false");
			props.put("session.timeout.ms", "30000");
			props.put("auto.offset.reset", "earliest");
			props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
			props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
			consumer = new KafkaConsumer<>(props);
			consumer.subscribe(Arrays.asList(queueConfig.getCrecordRequest()));
			while (true) {
				ConsumerRecords<String, String> records = consumer.poll(100);
				for (TopicPartition pt : records.partitions()) {
					for (ConsumerRecord<String, String> record : records.records(pt)) {
						try {
							crecordAsyncHttp.asyncSubmit(record.value());
						} catch (Exception e) {
							log.error("CrecordConsumer error:", e);
						} finally {
							long lastOffset = record.offset();
							consumer.commitSync(Collections.singletonMap(pt, new OffsetAndMetadata(lastOffset + 1)));
						}
					}
				}
			}
		} catch (Exception e) {
			log.error("error:", e);
		} finally {
			if(consumer != null){
				consumer.close();
			}
		}
	}

	public synchronized void stop() {
		if (executorService != null) {
			executorService.shutdownNow();
			executorService = null;
		}
	}

}
