package cn.com.duiba.notifycenter.service;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSONObject;

@Component
public class NotifyHttpClientPool {

	private static final Logger log = LoggerFactory.getLogger(NotifyHttpClientPool.class);

	/**
	 * 连接超时
	 */
	public static final int CONNECT_TIMEOUT = 30 * 1000;

	/**
	 * 长链接空闲时间
	 */
	public static final int KEEPALIVE_TIMEOUT = 30 * 1000;

	/**
	 * 处理超时
	 */
	public static final int SOCKET_TIMEOUT = 30 * 1000;

	/**
	 * 最大总连接值
	 */
	public static final int MAX_CONNECT = 1024;

	/**
	 * 每个路由最大连接{并发}值
	 */
	public static final int MAX_ROUTE_CONNECT = 128;

	private CloseableHttpAsyncClient httpClient;

	private Map<Long, AtomicInteger> runningStat = new ConcurrentHashMap<>();

	protected NotifyHttpClientPool() {
		httpClient = getAsyncHttpClient();
		httpClient.start();
	}

	public CloseableHttpAsyncClient getAsyncHttpClient() {
		return HttpAsyncClients.custom().setDefaultRequestConfig(getDefaultConfig()).setMaxConnTotal(MAX_CONNECT).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy()).build();
	}

	protected RequestConfig getDefaultConfig() {
		return RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setConnectionRequestTimeout(CONNECT_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
	}

	public DefaultConnectionKeepAliveStrategy getKeepAliveStrategy() {
		return new DefaultConnectionKeepAliveStrategy() {
			@Override
			public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
				long duration = super.getKeepAliveDuration(response, context);
				if (duration == -1) {
					return KEEPALIVE_TIMEOUT;
				}
				return duration;
			}
		};
	}

	public synchronized void submit(Long appId, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
		if (!httpClient.isRunning()) {
			log.warn(NotifyHttpClientPool.this.getClass().getName() + " httpClient not running,restart it");
			CloseableHttpAsyncClient client = getAsyncHttpClient();
			client.start();
			httpClient = client;
		}
		AtomicInteger count = runningStat.get(appId);
		if (count == null) {
			count = new AtomicInteger(1);
			runningStat.put(appId, count);
		} else {
			count.incrementAndGet();
		}
		CallbackProcesser process = new CallbackProcesser(appId, request, callback);
		httpClient.execute(process.getRequest(), process);
	}

	public JSONObject dumpDetail() {
		JSONObject o = new JSONObject();
		o.put("runningStat", runningStat);
		return o;
	}

	public int dumpSize() {
		int total = 0;
		for (AtomicInteger a : runningStat.values()) {
			total += a.get();
		}
		return total;
	}

	class CallbackProcesser implements FutureCallback<HttpResponse> {
		private HttpUriRequest request;
		private FutureCallback<HttpResponse> callback;
		private Long appId;

		public CallbackProcesser(Long appId, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
			this.request = request;
			this.callback = callback;
			this.appId = appId;
		}

		@Override
		public void completed(HttpResponse result) {
			try {
				callback.completed(result);
			} finally {
				runningStat.get(appId).decrementAndGet();
			}
		}

		@Override
		public void failed(Exception ex) {
			try {
				callback.failed(ex);
			} finally {
				runningStat.get(appId).decrementAndGet();
			}
		}

		@Override
		public void cancelled() {
			try {
				callback.cancelled();
			} finally {
				runningStat.get(appId).decrementAndGet();
			}
		}

		public HttpUriRequest getRequest() {
			return request;
		}

	}

}
