package cn.com.duiba.notifycenter.service.impl;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import cn.com.duiba.notifycenter.dao.NotifyQueueDAO;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.notifycenter.service.BussinessTypesService;
import cn.com.duiba.notifycenter.service.NotifyHttpClientPool;
import cn.com.duiba.notifycenter.service.NotifyService;
import cn.com.duiba.tool.HttpRequestLog;

import com.alibaba.fastjson.JSON;
import com.dangdang.ddframe.job.api.JobExecutionMultipleShardingContext;
import com.dangdang.ddframe.job.plugin.job.type.simple.AbstractSimpleElasticJob;

@Service("notifyService")
public class NotifyServiceImpl extends AbstractSimpleElasticJob implements NotifyService {

	private static Logger log = LoggerFactory.getLogger(NotifyServiceImpl.class);

	@Autowired
	private NotifyHttpClientPool notifyHttpClientPool;
	@Autowired
	private NotifyQueueDAO notifyQueueDAO;
	@Autowired
	private BussinessTypesService bussinessTypesService;

	@Value("${notifycenter.timer.enable}")
	private boolean timerEnable;

	/**
	 * 定时扫描每分钟执行次方法<br/>
	 * 0 0/1 * * * ?
	 */
	@Override
	public void process(JobExecutionMultipleShardingContext shardingContext) {
		scan();
	}

	@Override
	public void scan() {
		if (!timerEnable) {
			return;
		}
		List<NotifyQueueDO> list = notifyQueueDAO.findNeedNotifyList();
		log.info("notify scan " + list.size());
		for (NotifyQueueDO queue : list) {
			notify(queue);
		}
	}

	@Override
	public void notify(NotifyQueueDO queue) {
		try {
			NotifyFutureCallback callback = new NotifyFutureCallback(queue);
			HttpUriRequest request = bussinessTypesService.getRequest(queue);
			if (request != null) {
				HttpRequestLog.logUrl("[action notify] [tag request] [bizId " + queue.getDuibaOrderNum() + "] [url " + request.getURI().toString() + "]");
				notifyHttpClientPool.submit(queue.getAppId(), request, callback);
			} else {
				log.warn("notifyQueueId=" + queue.getId() + ", can't gen notify url,delete. " + JSON.toJSONString(queue));
				notifyQueueDAO.finish(queue.getId());
			}
		} catch (Exception e) {
			notifyQueueDAO.finish(queue.getId());
			log.error("notifyQueueId=" + queue.getId() + ",submit error:" + JSON.toJSONString(queue), e);
		}
	}

	@Override
	public Date getNextTime(NotifyQueueDO notifyQueue) {
		Map<Integer, Long> timeMap = new HashMap<>();
		timeMap.put(0, 2 * 60 * 1000L);
		timeMap.put(1, 10 * 60 * 1000L);
		timeMap.put(2, 10 * 60 * 1000L);
		timeMap.put(3, 1 * 60 * 60 * 1000L);
		timeMap.put(4, 2 * 60 * 60 * 1000L);
		timeMap.put(5, 6 * 60 * 60 * 1000L);
		timeMap.put(6, 15 * 60 * 60 * 1000L);
		return new Date(System.currentTimeMillis() + timeMap.get(notifyQueue.getTimes()));
	}

	class NotifyFutureCallback implements FutureCallback<HttpResponse> {

		private NotifyQueueDO notifyQueue;

		public NotifyFutureCallback(NotifyQueueDO notifyQueue) {
			this.notifyQueue = notifyQueue;
		}

		@Override
		public void completed(HttpResponse response) {
			String body = null;
			try {
				body = EntityUtils.toString(response.getEntity(), "utf-8");
				if (body != null) {
					body = body.trim();
				}
				if ("ok".equalsIgnoreCase(body)) {
					success();
				} else {
					fail();
				}
			} catch (Exception e) {
				log.error("completed", e);
				fail();
			} finally {
				HttpRequestLog.logUrl("[action notify] [tag response] [bizId " + notifyQueue.getDuibaOrderNum() + "] [type completed] [body " + body + "]");
			}
		}

		@Override
		public void failed(Exception ex) {
			HttpRequestLog.logUrl("[action notify] [tag response] [bizId " + notifyQueue.getDuibaOrderNum() + "] [type failed] [ex " + ex.getMessage() + "]");
			fail();
		}

		@Override
		public void cancelled() {
			HttpRequestLog.logUrl("[action notify] [tag response] [type cancelled] [bizId " + notifyQueue.getDuibaOrderNum() + "]");
			fail();
		}

		/**
		 * 通知成功，删除待通知记录
		 */
		private void success() {
			notifyQueueDAO.finish(notifyQueue.getId());
		}

		/**
		 * 通知失败，更新下次通知时间
		 */
		private void fail() {
			Calendar cal = Calendar.getInstance();
			cal.setTime(new Date());
			cal.add(Calendar.DATE, -1);
			if (notifyQueue.getTimes() > 6) {
				notifyQueueDAO.finish(notifyQueue.getId());
				log.warn("relationType=" + notifyQueue.getRelationType() + ",relationId=" + notifyQueue.getRelationId() + " notify times fail,drop");
			} else if (notifyQueue.getTimes() > 1 && notifyQueue.getGmtCreate().before(cal.getTime())) {
				notifyQueueDAO.finish(notifyQueue.getId());
				log.warn("relationType=" + notifyQueue.getRelationType() + ",relationId=" + notifyQueue.getRelationId() + " notify > 24h,drop");
			} else {
				Date nexttime = getNextTime(notifyQueue);
				notifyQueueDAO.updateNextTime(notifyQueue.getId(), notifyQueue.getTimes() + 1, nexttime);
			}
		}

	}

}
