package cn.com.duiba.service.impl;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import cn.com.duiba.crecord.service.HttpRetryRulesService;
import cn.com.duiba.dao.HttpMessageDAO;
import cn.com.duiba.http.HttpClientService;
import cn.com.duiba.message.KafkaClient;
import cn.com.duiba.service.CreditsService;
import cn.com.duiba.thirdparty.dto.HttpRequestMessageDto;
import cn.com.duiba.thirdparty.dto.HttpResponseMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.tool.JsonTool;

/**
 * 扣/加积分服务类
 */
@Service
public class CreditsServiceImpl implements CreditsService {

	private static final Logger LOG = LoggerFactory.getLogger(CreditsServiceImpl.class);

	@Value("${addCredits.retry}")
	private Boolean addCreditsRetry;

	@Autowired
	private KafkaClient kafkaClient;

	@Autowired
	private HttpMessageDAO httpMessageDAO;

	@Autowired
	private HttpRetryRulesService httpRetryRulesService;

	/**
	 * 提交扣积分请求
	 */
	@Override
	public void subCredits(HttpRequestMessageDto requet, String callbackTopic) {
		submit(requet, callbackTopic, false);
	}

	/**
	 * 提交加积分请求
	 */
	@Override
	public void addCredits(HttpRequestMessageDto requet, String callbackTopic) {
		submit(requet, callbackTopic, true);
	}

	/**
	 * 提交加扣积分HTTP请求
	 * 
	 * @param request
	 * @param topic
	 * @param isAddCredits
	 */
	private void submit(final HttpRequestMessageDto request, final String topic, final boolean isAddCredits) {
		HttpRequestBase http;
		if (HttpRequestMessageDto.HTTP_POST.equals(request.getHttpType())) {
			http = AssembleTool.assembleRequest(request.getHttpUrl(), request.getHttpParams());
		} else {
			http = new HttpGet(request.getHttpUrl());
		}
		HttpRequestLog.logUrl("[action credits] [tag request] [bizId " + request.getBizId() + "] [type " + request.getBizType() + "] [add " + isAddCredits + "] [url " + request.getHttpUrl() + "]");
		HttpClientService.get().submit(request.getAppId().toString(), http, new FutureCallback<HttpResponse>() {

			@Override
			public void completed(HttpResponse response) {
				HttpResponseMessageDto resp = new HttpResponseMessageDto();
				try {
					resp.setCallbackType(HttpResponseMessageDto.CALLBACK_TYPE_COMPLETED);
					String body = EntityUtils.toString(response.getEntity());
					resp.setBody(subBody(body));
				} catch (Exception e) {
					LOG.error("toDeveloper completed", e);
				} finally {
					finallyBlock(request, resp, topic, isAddCredits);
				}
			}

			@Override
			public void failed(Exception ex) {
				LOG.error("toDeveloper failed bizId:" + request.getBizId() + " bizType:" + request.getBizType(), ex);
				HttpResponseMessageDto resp = new HttpResponseMessageDto();
				try {
					resp.setCallbackType(HttpResponseMessageDto.CALLBACK_TYPE_FAILED);
					resp.setErrorMessage(ex.getClass().getName() + ":" + ex.getMessage());
				} catch (Exception e) {
					LOG.error("toDeveloper failed", e);
				} finally {
					finallyBlock(request, resp, topic, isAddCredits);
				}
			}

			@Override
			public void cancelled() {
				HttpResponseMessageDto resp = new HttpResponseMessageDto();
				try {
					resp.setCallbackType(HttpResponseMessageDto.CALLBACK_TYPE_FAILED);
					resp.setErrorMessage("http cancelled");
				} catch (Exception e) {
					LOG.error("toDeveloper cancelled", e);
				} finally {
					finallyBlock(request, resp, topic, isAddCredits);
				}
			}

		});
	}

	/**
	 * 截取返回内容
	 * 
	 * @param parm
	 * @return
	 */
	private String subBody(String parm) {
		String body = parm;
		if (StringUtils.isNotBlank(body) && body.length() > 2000) {
			body = body.substring(0, 2000);
		}
		return body;
	}

	/**
	 * 返回响应结果到业务放
	 * 
	 * @param req
	 * @param resp
	 * @param topic
	 * @param isAddCredits
	 */
	private void finallyBlock(HttpRequestMessageDto req, HttpResponseMessageDto resp, String topic, boolean isAddCredits) {
		try {
			resp.setBizId(req.getBizId());
			resp.setBizType(req.getBizType());
			resp.setExtendParams(req.getExtendParams());
			kafkaClient.sendMsg(topic, JsonTool.objectToJson(resp));
		} catch (Exception e) {
			LOG.error("credits callback: bizId:" + req.getBizId() + " bizType:" + req.getBizType(), e);
		} finally {
			HttpRequestLog.logUrl("[action credits] [tag response] [bizId " + req.getBizId() + "] [type " + req.getBizType() + "] [add " + isAddCredits + "] [callback " + resp.getCallbackType() + "] [body " + resp.getBody() + "]");
		}
	}

}