package cn.com.duiba.service;

import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * HTTP异步请求池
 */
@Service
public class HttpAsyncClientPool implements InitializingBean, DisposableBean {

	private static final Logger log = LoggerFactory.getLogger(HttpAsyncClientPool.class);

	/**
	 * 连接超时
	 */
	public static final int CONNECT_TIMEOUT = 30 * 1000;

	/**
	 * 长链接空闲时间
	 */
	public static final int KEEPALIVE_TIMEOUT = 30 * 1000;

	/**
	 * 处理超时
	 */
	public static final int SOCKET_TIMEOUT = 30 * 1000;

	/**
	 * 最大总连接值
	 */
	public static final int MAX_CONNECT = 5000;

	/**
	 * 每个路由最大连接{并发}值
	 */
	public static final int MAX_ROUTE_CONNECT = 200;

	/**
	 * APP最大处理队列
	 */
	public static final int MAX_APP_QUEUE = 200;

	private CloseableHttpAsyncClient httpClient;

	private Map<String, AtomicInteger> runningStat = new ConcurrentHashMap<>();

	@Autowired
	private ThreadPoolService threadPoolService;

	/**
	 * 执行HTTP异步请求
	 * 
	 * @param queueKey
	 * @param request
	 * @param callback
	 */
	public void execute(String queueKey, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
		try {
			long s = System.currentTimeMillis();
			AtomicInteger running = runningStat.get(queueKey);
			if (running == null) {
				runningStat.put(queueKey, new AtomicInteger(1));
			} else {
				running.incrementAndGet();
			}
			if (!httpClient.isRunning()) {
				start();
			}
			request.setHeader("User-Agent","Mozilla/5.0 (iPhone; CPU iPhone OS 9_1 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Version/9.0 Mobile/13B143 Safari/601.1");
			httpClient.execute(request, new CallbackProcesser(queueKey, callback));
			long e = System.currentTimeMillis();
			long t = e - s;
			if (t > 1000) {
				log.warn("http get pool time > {} ms for {}", t, request.getURI());
			}
		} catch (Exception e) {
			callback.failed(e);
			log.error("execute:", e);
		}
	}

	/**
	 * 提交HTTP异步请求
	 * 
	 * @param queueKey
	 * @param request
	 * @param callback
	 */
	public void submit(final String queueKey, final HttpUriRequest request, final FutureCallback<HttpResponse> callback) {
		if (threadPoolService.canSubmit()) {
			threadPoolService.submit(new Runnable() {
				@Override
				public void run() {
					execute(queueKey, request, callback);
				}
			});
		} else {
			execute(queueKey, request, callback);
		}
	}

	/**
	 * 判断APP能否提交任务到执行队列中
	 * 
	 * @param queueKey
	 * @return
	 */
	public boolean canSubmitToAppPool(String queueKey) {
		int queueCount = 0;
		AtomicInteger running = runningStat.get(queueKey);
		if (running != null) {
			queueCount = running.intValue();
		}
		if (queueCount < MAX_APP_QUEUE) {
			return true;
		}
		return false;
	}

	/**
	 * 队列详细信息
	 * 
	 * @return
	 */
	public Map<String, Object> dumpDetail() {
		Map<String, Object> map = new HashMap<>();
		for (Map.Entry<String, AtomicInteger> entry : runningStat.entrySet()) {
			if (entry.getValue().intValue() > 0) {
				map.put(entry.getKey(), entry.getValue());
			}
		}
		Map<String, Object> ret = new HashMap<>();
		ret.put("runningStat", map);
		return ret;
	}

	/**
	 * HTTP处理中的个数
	 * 
	 * @return
	 */
	public int dumpSize() {
		int running = 0;
		for (AtomicInteger a : runningStat.values()) {
			running += a.get();
		}
		return running;
	}

	/**
	 * 自定义回调
	 */
	private class CallbackProcesser implements FutureCallback<HttpResponse> {

		private FutureCallback<HttpResponse> callback;
		private String queueKey;

		public CallbackProcesser(String queueKey, FutureCallback<HttpResponse> callback) {
			this.callback = callback;
			this.queueKey = queueKey;
		}

		@Override
		public void completed(HttpResponse result) {
			try {
				callback.completed(result);
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

		@Override
		public void failed(Exception ex) {
			try {
				callback.failed(ex);
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

		@Override
		public void cancelled() {
			try {
				callback.cancelled();
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

	}

	/**
	 * 销毁关闭
	 */
	@Override
	public void destroy() throws Exception {
		if (httpClient != null && httpClient.isRunning()) {
			httpClient.close();
			log.info("HttpAsyncClientPool closed");
		}
	}

	/**
	 * 初始化启动
	 */
	@Override
	public void afterPropertiesSet() throws Exception {
		start();
	}

	/**
	 * 启动HTTP服务
	 */
	private synchronized void start() {
		if (httpClient != null && httpClient.isRunning()) {
			return;
		}
		RequestConfig config = RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setConnectionRequestTimeout(CONNECT_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
		httpClient = HttpAsyncClients.custom().setDefaultRequestConfig(config).setMaxConnTotal(MAX_CONNECT).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy()).build();
		httpClient.start();
		log.info("HttpAsyncClientPool started");
	}

	/**
	 * 长链接空闲时间
	 */
	private DefaultConnectionKeepAliveStrategy getKeepAliveStrategy() {
		return new DefaultConnectionKeepAliveStrategy() {
			@Override
			public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
				long duration = super.getKeepAliveDuration(response, context);
				if (duration == -1) {
					return KEEPALIVE_TIMEOUT;
				}
				return duration;
			}
		};
	}

}
