package cn.com.duiba.biz.credits;

import cn.com.duiba.dao.HttpMessageDAO;
import cn.com.duiba.domain.HttpMessageDO;
import cn.com.duiba.order.center.api.dto.CreditsCallbackMessage;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.service.CustomService;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.service.HttpRetryRulesService;
import cn.com.duiba.service.MessageService;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.tool.JsonTool;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 加积分接口请求服务
 */
@Service
public class AddCreditsToDeveloper {

	private static final Logger LOG = LoggerFactory.getLogger(AddCreditsToDeveloper.class);

	@Autowired
	private HttpAsyncClientPool httpAsyncClientPool;

	@Autowired
	private MessageService messageService;

	@Autowired
	private HttpRetryRulesService httpRetryRulesService;

	@Autowired
	private HttpMessageDAO httpMessageDAO;

	@Autowired
	private CustomService customService;

	/**
	 * 提交加积分HTTP请求
	 * @param message
	 */
	public void submit(final HttpMessageDO message) {
		CreditsMessageDto request = JsonTool.jsonToObject(message.getBizParams(), CreditsMessageDto.class);
		final CreditsMessageDto creditsMessageDto = customService.getRequestAddCredits(request);
		HttpRequestBase http;
		if (CreditsMessage.HTTP_POST.equals(creditsMessageDto.getHttpType())) {
			http = AssembleTool.assembleRequest(creditsMessageDto.getHttpUrl(), creditsMessageDto.getAuthParams());
		} else {
			http = new HttpGet(creditsMessageDto.getHttpUrl());
		}

		HttpRequestLog.logUrl("[action addCredits] [tag request] [bizId " + creditsMessageDto.getRelationId() + "] [type " + creditsMessageDto.getRelationType() + "] [url " + creditsMessageDto.getHttpUrl() + "]");
		httpAsyncClientPool.submit(creditsMessageDto.getAppId(), http, new FutureCallback<HttpResponse>() {

			@Override
			public void completed(HttpResponse response) {
				CreditsCallbackMessage resp = new CreditsCallbackMessage();
				try {
					resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_COMPLETED);
					String body = EntityUtils.toString(response.getEntity());
					resp.setMessage(HttpRequestLog.subBody(body));
				} catch (Exception e) {
					LOG.error("toDeveloper completed", e);
				} finally {
					finallyBlock(message, creditsMessageDto, resp);
				}
			}

			@Override
			public void failed(Exception ex) {
				LOG.error("toDeveloper failed bizId:" + creditsMessageDto.getRelationId() + " bizType:" + creditsMessageDto.getRelationType(), ex);
				CreditsCallbackMessage resp = new CreditsCallbackMessage();
				try {
					resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_FAILED);
					resp.setMessage(ex.getClass().getName() + ":" + ex.getMessage());
				} catch (Exception e) {
					LOG.error("toDeveloper failed", e);
				} finally {
					finallyBlock(message, creditsMessageDto, resp);
				}
			}

			@Override
			public void cancelled() {
				LOG.error("toDeveloper cancelled bizId:" + creditsMessageDto.getRelationId() + " bizType:" + creditsMessageDto.getRelationType());
				CreditsCallbackMessage resp = new CreditsCallbackMessage();
				resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_FAILED);
				resp.setMessage("http cancelled");
				finallyBlock(message, creditsMessageDto, resp);
			}

		});
	}

	/**
	 * 返回响应结果到业务方
	 * @param message
	 * @param req
	 * @param resp
	 */
	public void finallyBlock(HttpMessageDO message, CreditsMessageDto req, CreditsCallbackMessage resp) {
		HttpRequestLog.logUrl("[action addCredits] [tag response] [bizId " + req.getRelationId() + "] [type " + req.getRelationType() + "] [callback " + resp.getCallbackType() + "] [body " + resp.getMessage() + "]");
		if (nextRetry(message, resp)) {
			return;
		}
		try {
			resp.setRelationId(req.getRelationId());
			resp.setRelationType(req.getRelationType());
			resp.setParams(req.getParams());
			resp.setAppId(req.getAppId());
			resp.setConsumerId(req.getConsumerId());
			resp.setHttpUrl(req.getHttpUrl());
			String body = JsonTool.objectToJson(resp);
			String callbackTopic = req.getCallbackTopic();
			String callbackTag = req.getCallbackTag();
			String callbackKey = req.getCallbackKey();
			if (StringUtils.isEmpty(callbackTag) && StringUtils.isEmpty(callbackKey)) {
				messageService.sendMsg(callbackTopic, body);
			} else {
				messageService.sendMsg(callbackTopic, callbackTag, callbackKey, body);
			}
		} catch (Exception e) {
			LOG.error("credits callback: bizId:" + req.getRelationId() + " bizType:" + req.getRelationType(), e);
		}
	}

	/**
	 * 判断是否需要重试<br/>
	 * 1.记录重试数据<br/>
	 * 2.定时扫描<br/>
	 * 
	 * @param message
	 * @param resp
	 * @return true 下次重试， false 无需重试
	 */
	public boolean nextRetry(HttpMessageDO message, CreditsCallbackMessage resp) {
		try {
			// 如果响应包含OK认为是成功
			if (isSuccess(resp)) {
				httpMessageDAO.delete(message.getId());
				return false;
			}
			// 是否超过重试次数
			if (message.getNumber() >= httpRetryRulesService.getRetryNumber(message)) {
				httpMessageDAO.delete(message.getId());
				return false;
			}
			// 更新下次重试时间
			httpRetryRulesService.updateNextTime(message);
			return true;
		} catch (Exception e) {
			LOG.error("nextRetry", e);
		}
		return false;
	}

	/**
	 * 解析响应的内容是否为成功
	 * 
	 * @param resp
	 * @return
	 */
	public boolean isSuccess(CreditsCallbackMessage resp) {
		try {
			if (CreditsCallbackMessage.CALLBACK_TYPE_COMPLETED.equals(resp.getCallbackType())) {
				String body = resp.getMessage().toLowerCase();
				if (body.contains("ok") || body.contains("success")) {
					return true;
				}
			}
			return false;
		} catch (Exception e) {
			LOG.error("isSuccess", e);
			return true;
		}
	}

}
