package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.constant.RenrenTvConfig;
import cn.com.duiba.credits.sdk.CreditNotifyParams;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.JsonTool;
import cn.com.duiba.vo.RenrenTvResultVO;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Maps;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Map;
import java.util.TreeMap;

/**
 * Created by HePeng on 2019/05/23 13:48.
 */
@Service
public class RenrenTvApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(RenrenTvApi.class);
    @Autowired
    private RenrenTvConfig renrenTvConfig;

    /**
     * 成功时的code
     */
    private static final String SUCCESS_CODE = "0000";

    /**
     * 订单已经确认的code
     */
    private static final String HAS_CONFIRMED_CODE = "8644";

    /**
     * 虚拟商品订单增加前缀
     */
    private static final String VIRTUAL_ORDER_NUM_PREFIX = "virtual_";

    /**
     * 是否是人人视频app
     * @param appId
     * @return
     */
    public boolean isRenrenTv(Long appId) {
        return appId != null && appId.equals(renrenTvConfig.getAppId());
    }

    /**
     * 处理扣积分message
     * @param request
     * @return
     */
    public CreditsMessage getSubCreditsMessage(CreditsMessage request) {
        // post请求
        request.setHttpType(CreditsMessage.HTTP_POST);
        // 截取请求url
        String httpUrl = request.getHttpUrl();
        int index = httpUrl.indexOf('?');
        if(index <= 0) {
            throw new IllegalArgumentException("httpUrl:" + httpUrl);
        }
        String postUrl = httpUrl.substring(0, index);
        request.setHttpUrl(postUrl);
        // 截取url请求参数，构造新的请求参数
        String paramStr = httpUrl.substring(index + 1, httpUrl.length());
        Map<String, String> urlParams = AssembleTool.getUrlParams(paramStr);
        Map<String, String> authParams = buildAddOrSubCreditsAuthParams(urlParams, false);
        request.setAuthParams(authParams);
        return request;
    }

    /**
     * 处理扣积分message
     * @param request
     * @return
     */
    public SubCreditsMsgWrapper getRequestCredits(SubCreditsMsgWrapper request) {
        // post请求
        request.getSubCreditsMsg().setHttpType(SubCreditsMsgDto.HTTP_POST);
        // 截取请求url
        String httpUrl = request.getHttpUrl();
        int index = httpUrl.indexOf('?');
        if(index <= 0) {
            throw new IllegalArgumentException("httpUrl:" + httpUrl);
        }
        String postUrl = httpUrl.substring(0, index);
        request.setHttpUrl(postUrl);
        // 构造新的请求参数
        Map<String, String> authParams = buildAddOrSubCreditsAuthParams(request.getSubCreditsMsg().getAuthParams(), false);
        request.getSubCreditsMsg().setAuthParams(authParams);
        return request;
    }

    /**
     * 处理加积分message
     * @param request
     * @return
     */
    public CreditsMessageDto getAddCreditsMessage(CreditsMessageDto request) {
        // post请求
        request.setHttpType(CreditsMessageDto.HTTP_POST);
        // 截取请求url
        String httpUrl = request.getHttpUrl();
        int index = httpUrl.indexOf('?');
        if(index <= 0) {
            throw new IllegalArgumentException("httpUrl:" + httpUrl);
        }
        String postUrl = httpUrl.substring(0, index);
        request.setHttpUrl(postUrl);
        // 截取url请求参数，构造新的请求参数
        String paramStr = httpUrl.substring(index + 1, httpUrl.length());
        Map<String, String> urlParams = AssembleTool.getUrlParams(paramStr);
        Map<String, String> authParams = buildAddOrSubCreditsAuthParams(urlParams, true);
        request.setAuthParams(authParams);
        return request;
    }

    /**
     * 构造加积分/扣积分 请求参数
     * @param urlParams
     * @param addFlag
     * @return
     */
    private Map<String,String> buildAddOrSubCreditsAuthParams(Map<String, String> urlParams, boolean addFlag) {
        Map<String, String> authParams = Maps.newHashMap();
        authParams.put("token", urlParams.get("transfer"));
        authParams.put("orderCode", urlParams.get("orderNum"));
        authParams.put("time", String.valueOf(System.currentTimeMillis()));
        authParams.put("amounts", (addFlag ? "" : "-") + urlParams.get("credits"));
        authParams.put("description", urlParams.get("description"));
        // 积分日志额外加的字段
        authParams.put("orderNum", urlParams.get("orderNum"));
        authParams.put("credits", urlParams.get("credits"));
        authParams.put("type", urlParams.get("type"));
        return authParams;
    }

    /**
     * 解析积分请求结果
     * @param body
     * @return
     */
    public String parseCreditsRsp(String body) {
        try {
            RenrenTvResultVO<Long> resultVO = JSONObject.parseObject(body, new TypeReference<RenrenTvResultVO<Long>>() {});
            Map<String, Object> duibaDoc = Maps.newHashMap();
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", null);
            duibaDoc.put("bizId", null);
            duibaDoc.put("credits", null);
            if(resultVO != null) {
                if(SUCCESS_CODE.equals(resultVO.getCode())) {
                    duibaDoc.put("status", "ok");
                }
                duibaDoc.put("errorMessage", resultVO.getMsg());
                duibaDoc.put("bizId", resultVO.getRequestId());
                duibaDoc.put("credits", resultVO.getData());
            }
            return JsonTool.objectToJson(duibaDoc);
        } catch (Exception e) {
            LOGGER.warn("", e);
            return body;
        }
    }

    /**
     * 构造通知请求
     * @param notifyUrl
     * @param params
     * @return
     */
    public HttpRequestBase getRequestNotify(String notifyUrl, CreditNotifyParams params) {
        // 构造请求参数
        Map<String, String> authParams = Maps.newHashMap();
        authParams.put("orderCode", params.getOrderNum());
        authParams.put("status", String.valueOf(params.isSuccess()));
        authParams.put("description", params.getErrorMessage());
        // 构造post请求
        HttpPost httpPost = AssembleTool.assembleRequest(notifyUrl, authParams);
        // 设置请求头
        setHttpHeader(authParams, httpPost);
        return httpPost;
    }

    /**
     * 通知请求结果解析
     * @param body
     * @return
     */
    public String getResponseNotify(String body) {
        try {
            RenrenTvResultVO<String> resultVO = JSONObject.parseObject(body, new TypeReference<RenrenTvResultVO<String>>() {});
            if(resultVO != null && (SUCCESS_CODE.equals(resultVO.getCode()) || HAS_CONFIRMED_CODE.equals(resultVO.getCode()))) {
                return "ok";
            }
        } catch (Exception e) {
            LOGGER.warn("", e);
        }
        return body;
    }

    /**
     * 处理虚拟商品充值message
     * @param request
     * @return
     */
    public SupplierRequest getVirtualRequest(SupplierRequest request) {
        // 截取请求url
        String httpUrl = request.getHttpUrl();
        int index = httpUrl.indexOf('?');
        if(index <= 0) {
            throw new IllegalArgumentException("httpUrl:" + httpUrl);
        }
        String postUrl = httpUrl.substring(0, index);
        request.setHttpUrl(postUrl);
        // 截取url请求参数，构造新的请求参数
        String paramStr = httpUrl.substring(index + 1, httpUrl.length());
        Map<String, String> urlParams = AssembleTool.getUrlParams(paramStr);
        Map<String, String> authParams = buildVirtualCreditsAuthParams(urlParams);
        request.setAuthParams(authParams);
        return request;
    }

    /**
     * 构造 虚拟商品充值 请求参数
     * @param urlParams
     * @return
     */
    private Map<String,String> buildVirtualCreditsAuthParams(Map<String, String> urlParams) {
        Map<String, String> authParams = Maps.newHashMap();
        authParams.put("token", urlParams.get("transfer"));
        authParams.put("orderCode", VIRTUAL_ORDER_NUM_PREFIX + urlParams.get("orderNum"));
        authParams.put("time", String.valueOf(System.currentTimeMillis()));
        String credits = parseCredits(urlParams.get("params"));
        if(credits == null) {
            throw new IllegalArgumentException("params: " + urlParams.get("params"));
        }
        authParams.put("amounts", credits);
        authParams.put("description", urlParams.get("description"));
        return authParams;
    }

    /**
     * 从虚拟商品标识符中解析出积分数
     * @param params
     * @return
     */
    private String parseCredits(String params) {
        if(StringUtils.isBlank(params) || !params.startsWith(renrenTvConfig.getVirtualCreditsPrefix())) {
            return null;
        }
        try {
            return String.valueOf(
                    Long.parseUnsignedLong(
                            params.trim().substring(renrenTvConfig.getVirtualCreditsPrefix().length())
                    )
            );
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 虚拟商品充值响应结果解析
     * @param body
     * @return
     */
    public String getVirtualRespone(String body) {
        try {
            RenrenTvResultVO<Long> resultVO = JSONObject.parseObject(body, new TypeReference<RenrenTvResultVO<Long>>() {});
            Map<String, Object> duibaDoc = Maps.newHashMap();
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", null);
            duibaDoc.put("supplierBizId", null);
            duibaDoc.put("credits", null);
            if(resultVO != null) {
                if(SUCCESS_CODE.equals(resultVO.getCode())) {
                    duibaDoc.put("status", "success");
                }
                duibaDoc.put("errorMessage", resultVO.getMsg());
                duibaDoc.put("supplierBizId", resultVO.getRequestId());
                duibaDoc.put("credits", resultVO.getData());
            }
            return JsonTool.objectToJson(duibaDoc);
        } catch (Exception e) {
            LOGGER.warn("", e);
            return body;
        }
    }

    /**
     * 构建虚拟商品请求
     * @param message
     * @return
     */
    public HttpRequestBase setVirtualHttpHeader(SupplierRequest message) {
        // 构建HttpPost
        HttpPost httpPost = AssembleTool.assembleRequest(message.getHttpUrl(), message.getAuthParams());
        // 设置请求头
        setHttpHeader(message.getAuthParams(), httpPost);
        return httpPost;
    }

    /**
     * 设置请求头
     * @param authParams
     * @param http
     */
    public void setHttpHeader(Map<String, String> authParams, HttpRequestBase http) {
        http.setHeader("ak", renrenTvConfig.getAk());
        http.setHeader("sign", generateSign(new TreeMap<>(authParams)));
    }

    /**
     * 生成签名sign
     * @param params
     * @return
     */
    private String generateSign(TreeMap<String, String> params) {
        StringBuilder sb = new StringBuilder();
        if(params != null && params.size() > 0) {
            for(Map.Entry<String, String> entry : params.entrySet()) {
                sb.append(entry.getKey()).append("=").append(entry.getValue()).append("&");
            }
        }
        String signTemp = sb.append("sk=").append(renrenTvConfig.getSk()).toString();
        return DigestUtils.md5Hex(signTemp).toUpperCase();
    }

}
