package cn.com.duiba.biz.credits;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.SpdBankConfig;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.kunshanRCB.Util;
import com.alibaba.fastjson.JSONObject;
import com.spdbccc.SM2Signature;
import com.spdbccc.SM3Digest;
import com.spdbccc.SM4Crypto;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.xml.bind.DatatypeConverter;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

/**
 * @ClassName SpdBankApi
 * @Description 浦发银行api
 * @Author zhoujunquan@duiba.com.cn
 * @Date 2021/1/9 2:46 下午
 * @Version 1.0
 **/
@Service
public class SpdBankApi {
    private static final Logger log = LoggerFactory.getLogger(SpdBankApi.class);
    private static final String LOG_PREFIX = "spdBank:";

    @Autowired
    private SpdBankConfig spdBankConfig;

    public boolean isPdCardApp(Long appId) {
        return null != appId && appId.equals(spdBankConfig.getPdCardAppId());
    }

    /**
     * 浦发银行浦发卡中心三方发券接口封装
     *
     * @param request SupplierRequest
     * @return http
     */
    public HttpRequestBase getPdCardVirtualRequest(SupplierRequest request) {
        String url = request.getHttpUrl();
        log.info(LOG_PREFIX + "url:{}", url);
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramsStr);
        String orderNum = params.get("orderNum");
        String uid = params.get("uid");
        String prizeNum = params.get("params");
        JSONObject requestParam = new JSONObject();
        requestParam.put("transId", params.get("orderNum"));
        requestParam.put("channel", spdBankConfig.getChannel());
        requestParam.put("userId", params.get("uid"));
        requestParam.put("sendCouponChannel", "YB");
        requestParam.put("ruleCheckFlag", "0");
        requestParam.put("prizeNum", prizeNum);
        HttpPost post = new HttpPost(spdBankConfig.getApiUrl());
        String reqBody = requestParam.toJSONString();
        log.info(LOG_PREFIX + "uid:{},orderNum:{},prizeNum:{},reqBody:{}", uid, orderNum, prizeNum, reqBody);
        //请求报文体加密
        byte[] encrypted = SM4Crypto.encrypt(reqBody.getBytes(), spdBankConfig.getSm4Secret());
        String encryptBody = DatatypeConverter.printBase64Binary(encrypted);
        log.info(LOG_PREFIX + "uid:{},orderNum:{},prizeNum:{},encryptBody:{}", uid, orderNum, prizeNum, encryptBody);
        //生成时间戳 start
        String timestamp = dateGMT();
        //拼接并生成签名字串
        String signStr = timestamp + spdBankConfig.getClientId() + encryptBody;
        //生成SM3摘要
        String digest = SM3Digest.digest(signStr.getBytes());
        //生成签名
        byte[] signByte = SM2Signature.createSign(spdBankConfig.getHzfPrivateKey(), spdBankConfig.getClientId()
                , digest.getBytes());
        String sign = Hex.toHexString(signByte).toUpperCase();
        //设置请求头
        post.setHeader("X-SPDBCCC-CLIENTID", spdBankConfig.getClientId());
        post.setHeader("X-SPDBCCC-TIMESTAMP", timestamp);
        post.setHeader("X-SPDBCCC-SIGNATURE", sign);
        post.setHeader("Content-Type", "application/json;charset=utf-8");
        post.setHeader("Accept-Type", "json");
        post.setEntity(new StringEntity(encryptBody, StandardCharsets.UTF_8));
        return post;
    }

    public String getPdCardVirtualResponse(SupplierRequest message, String body, HttpResponse response) {
        String url = message.getHttpUrl();
        log.info(LOG_PREFIX + "url:{}", url);
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramsStr);
        String orderNum = params.get("orderNum");
        String uid = params.get("uid");
        log.info(LOG_PREFIX + "uid:{},orderNum:{},encrypt body:{}", uid, orderNum, body);
        if (StringUtils.isBlank(body)) {
            throw new ThirdpatyException("spdBank pdCard sendCouponNotice result is blank");
        }
        Header header = response.getFirstHeader("X-SPDBCCC-SIGNATURE");
        if (null == header) {
            throw new ThirdpatyException("spdBank pdCard sendCouponNotice header is null");
        }
        String sign = header.getValue();
        String digest = SM3Digest.digest((spdBankConfig.getClientId() + body).getBytes());
        boolean verify = SM2Signature.verifySign(spdBankConfig.getKzxPublicKey(), spdBankConfig.getClientId()
                , Util.hexStringToBytes(sign), digest.getBytes());
        if (!verify) {
            throw new ThirdpatyException("spdBank pdCard sendCouponNotice verify fail");
        }
        //报文解密
        byte[] array = DatatypeConverter.parseBase64Binary(body);
        byte[] decrypt = SM4Crypto.decrypt(array, spdBankConfig.getSm4Secret());
        body = new String(decrypt);
        log.info(LOG_PREFIX + "uid:{},orderNum:{},decrypt body:{}", uid, orderNum, body);
        JSONObject result = new JSONObject();
        result.put("supplierBizId", "");
        try {
            JSONObject resp = JSONObject.parseObject(body);
            if ("000000".equals(resp.getString("code"))) {
                JSONObject data = resp.getJSONObject("data");
                if ("1".equals(data.getString("sendStatus"))) {
                    result.put("status", "success");
                    result.put("credits", "0");
                } else if ("4".equals(data.getString("sendStatus"))) {
                    result.put("status", "process");
                    result.put("credits", "0");
                } else {
                    result.put("status", "fail");
                    result.put("errorMessage", SpdBankConfig.ResultMessageEnum
                            .getMessageByCode(data.getString("sendStatus")));
                }
            } else {
                result.put("status", "fail");
                result.put("errorMessage", resp.getString("msg"));
            }
        } catch (Exception e) {
            log.error("spdBank pdCard result analysis error,body:{}", body, e);
            result.put("status", "fail");
            result.put("errorMessage", "三方发券接口响应信息解析异常");
        }
        return result.toJSONString();
    }

    /***
     * 获取当前时间戳
     * @return 当前时间戳
     */
    public static String dateGMT() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss 'GMT'", Locale.US);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
        Date date = new Date();
        return dateFormat.format(date);
    }
}