package cn.com.duiba.remoteimpl;

import cn.com.duiba.biz.alipay.domain.AlipayOfficialRequest;
import cn.com.duiba.biz.alipay.supplier.AlipayBatchExecutor;
import cn.com.duiba.biz.alipay.supplier.ZhongXiangSupplier;
import cn.com.duiba.biz.phonebill.supplier.JuhePhonebillSupplier;
import cn.com.duiba.biz.phonebill.supplier.PUUPPhoneBillSupplier;
import cn.com.duiba.biz.phonebill.supplier.XuanJiePhoneBillSupplier;
import cn.com.duiba.biz.phoneflow.supplier.MoPinPhoneflowSupplier;
import cn.com.duiba.biz.qb.supplier.KaiXinQbSupplier;
import cn.com.duiba.biz.virtual.supplier.VirtualSupplier;
import cn.com.duiba.domain.SupplierProductDO;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.thirdparty.api.RemoteChargeService;
import cn.com.duiba.thirdparty.dto.AlipayOfficialRequestDto;
import cn.com.duiba.thirdparty.dto.SupplierRequestDto;
import cn.com.duiba.tool.CodeException;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;

/**
 * 直冲类接口请求远程服务实现类
 */
@RestController
public class RemoteChargeServiceImpl implements RemoteChargeService {

	private static final Logger LOGGER = LoggerFactory.getLogger(RemoteChargeServiceImpl.class);

	public static final Set<String> PUUP_SUPPLIER_SET = ImmutableSet.of(SupplierProductDO.SUPPLIER_PHONE_BILL_PUUP_INCLUDE_TAX, SupplierProductDO.SUPPLIER_PHONE_BILL_PUUP_TAX_FREE);
	public static final Set<String> XUANJIE_SUPPLIER_SET = ImmutableSet.of(SupplierProductDO.SUPPLIER_PHONE_BILL_XUANJIE_INCLUDE_TAX, SupplierProductDO.SUPPLIER_PHONE_BILL_XUANJIE_TAX_FREE);


	@Autowired
	private AlipayBatchExecutor alipayBatchExecutor;
	@Autowired
	private ZhongXiangSupplier zhongXiangSupplier;
	@Autowired
	private MoPinPhoneflowSupplier moPinPhoneflowSupplier;
	@Autowired
	private VirtualSupplier virtualSupplier;
	@Autowired
	private HttpAsyncClientPool httpAsyncClientPool;
	@Autowired
	private JuhePhonebillSupplier juhePhonebillSupplier;
	@Autowired
	private KaiXinQbSupplier kaiXinQbSupplier;
	@Autowired
	private PUUPPhoneBillSupplier puupPhoneBillSupplier;
	@Autowired
	private XuanJiePhoneBillSupplier xuanJiePhoneBillSupplier;

	/**
	 * 提交官方支付宝请求
	 */
	@Override
	public DubboResult<Boolean> submitAlipay(AlipayOfficialRequestDto request) {
		try {
			AlipayOfficialRequest req = BeanUtils.copy(request, AlipayOfficialRequest.class);
			alipayBatchExecutor.addAlipayOrder(req);
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("submitAlipay", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	/**
	 * 提交众想支付宝接口请求
	 */
	@Override
	public DubboResult<Boolean> submitZxAlipay(SupplierRequestDto supplierRequestDto, String callbackTopic) {
		try {
			SupplierRequest sr = BeanUtils.copy(supplierRequestDto, SupplierRequest.class);
			validate(sr, callbackTopic);
			zhongXiangSupplier.asyncSubmit(sr, callbackTopic, null, null);
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("submitZxAlipay", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	/**
	 * 提交QB接口请求
	 */
	@Override
	public DubboResult<Boolean> submitQb(SupplierRequestDto request, String callbackTopic) {
		return submitQb(request, callbackTopic, null, null);
	}

	/**
	 * 提交话费接口请求
	 */
	@Override
	public DubboResult<Boolean> submitPhonebill(SupplierRequestDto supplierRequestDto, String callbackTopic) {
		return submitPhonebill(supplierRequestDto, callbackTopic, null, null);
	}

	/**
	 * 提交流量接口请求
	 */
	@Override
	public DubboResult<Boolean> submitPhoneflow(SupplierRequestDto supplierRequestDto, String callbackTopic) {
		return submitPhoneflow(supplierRequestDto, callbackTopic, null, null);
	}

	/**
	 * 提交虚拟商品接口请求
	 */
	@Override
	public DubboResult<Boolean> submitVirtual(SupplierRequestDto supplierRequestDto, String callbackTopic) {
		return submitVirtual(supplierRequestDto, callbackTopic, null, null);
	}

	private void validate(SupplierRequest request, String callbackTopic) throws CodeException {
		if (request == null) {
			throw new CodeException("0", "参数校验失败");
		}
		if (StringUtils.isEmpty(callbackTopic)) {
			throw new CodeException("0", "订阅Topic不能为空");
		}
	}

	private void validateVirtual(SupplierRequest request, String callbackTopic) throws CodeException {
		if (StringUtils.isEmpty(callbackTopic)) {
			throw new CodeException("0", "订阅Topic不能为空");
		}
		if (StringUtils.isEmpty(request.getHttpUrl())) {
			throw new CodeException("0", "请求URL不能为空");
		}
		if (request.getAppId() == null) {
			throw new CodeException("0", "APPID不能为空");
		}
		if (!httpAsyncClientPool.canSubmitToAppPool(request.getAppId())) {
			throw new CodeException("0", "APP队列超过阀值，APP接口繁忙, appId: " + request.getAppId());
		}
	}

	@Override
	public DubboResult<Boolean> submitZxAlipay(SupplierRequestDto supplierRequestDto, String callbackTopic, String msgTag, String msgKey) {
		try {
			SupplierRequest sr = BeanUtils.copy(supplierRequestDto, SupplierRequest.class);
			validate(sr, callbackTopic);
			zhongXiangSupplier.asyncSubmit(sr, callbackTopic, msgTag, msgKey);
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("submitZxAlipay", e);
			return DubboResult.failResult(e.getMessage());
		}
	}


	@Override
	public DubboResult<Boolean> submitQb(SupplierRequestDto request, String callbackTopic, String msgTag, String msgKey) {
		try {
			SupplierRequest req = BeanUtils.copy(request, SupplierRequest.class);
			validate(req, callbackTopic);
			kaiXinQbSupplier.asyncSubmit(req, callbackTopic, msgTag, msgKey);
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("submitQb", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> submitPhonebill(SupplierRequestDto supplierRequestDto, String callbackTopic, String msgTag, String msgKey) {
		try {
			SupplierRequest request = BeanUtils.copy(supplierRequestDto, SupplierRequest.class);
			validate(request, callbackTopic);
			String supplierName = request.getSupplierName();
			if(SupplierProductDO.SupplierPhonebillJuHe.equals(supplierName)){
				juhePhonebillSupplier.asyncSubmit(request, callbackTopic, msgTag, msgKey);
			} else if (PUUP_SUPPLIER_SET.contains(supplierName)) {
				puupPhoneBillSupplier.asyncSubmit(request, callbackTopic, msgTag, msgKey);
			}else if (XUANJIE_SUPPLIER_SET.contains(supplierName)) {
				xuanJiePhoneBillSupplier.asyncSubmit(request, callbackTopic, msgTag, msgKey);
			} else {
				throw new CodeException("0", "phonebill无对应上游：" + supplierName);
			}
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("submitPhonebill", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> submitPhoneflow(SupplierRequestDto supplierRequestDto, String callbackTopic, String msgTag, String msgKey) {
		try {
			SupplierRequest request = BeanUtils.copy(supplierRequestDto, SupplierRequest.class);
			validate(request, callbackTopic);
			moPinPhoneflowSupplier.asyncSubmit(request, callbackTopic, msgTag, msgKey);
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("submitPhoneflow", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> submitVirtual(SupplierRequestDto supplierRequestDto, String callbackTopic, String tag, String key) {
		try {
			SupplierRequest request = BeanUtils.copy(supplierRequestDto, SupplierRequest.class);
			validateVirtual(request, callbackTopic);
			virtualSupplier.asyncSubmit(request, callbackTopic, tag, key);
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("submitVirtual", e);
			return DubboResult.failResult(e.getMessage());
		}
	}


    /**
     * 仅同步调用，可不依赖于兑吧订单
     * 提交虚拟商品第三方接口
     * 返回Boolean是否提交成功
     *
     * @param supplierRequestDto
     * @return
     */
    @Override
    public DubboResult<String> submitVirtualSync(SupplierRequestDto supplierRequestDto) {
        try {
            SupplierRequest request = BeanUtils.copy(supplierRequestDto, SupplierRequest.class);
            if (StringUtils.isEmpty(request.getHttpUrl())) {
                throw new CodeException("0", "请求URL不能为空");
            }

            return virtualSupplier.syncSubmitWithoutDuiba(request);
        } catch (Exception e) {
            LOGGER.error("submitVirtual exception,supplierRequestDto={}", JSON.toJSONString(supplierRequestDto), e);
            return DubboResult.failResult(e.getMessage());
        }
    }
}
