package cn.com.duiba.service.virtual.impl;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.HuazhuConfig;
import cn.com.duiba.dao.AppDAO;
import cn.com.duiba.domain.AppDO;
import cn.com.duiba.enums.redis.RedisKeyEnum;
import cn.com.duiba.service.impl.AbstractDuibaVirtualSupplier;
import cn.com.duiba.thirdparty.dto.SupplierRequestDto;
import cn.com.duiba.thirdparty.enums.virtual.VirtualItemChannelEnum;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.redis.RedisLock;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author qiurenjie
 * @date 2021-07-16 11:11
 */
@Service
public class HuaZhuApiStrategy extends AbstractDuibaVirtualSupplier {

    private static final Logger LOGGER = LoggerFactory.getLogger(HuaZhuApiStrategy.class);

    private static final String LOGGER_PREFIX = "华住虚拟商品接口";

    private static final String RESPONSE_CODE_KEY = "ResponseCode";
    private static final String RESPONSE_DATA_KEY = "BusinessCode";
    private static final String RESPONSE_DESC_KEY = "CustomerDesc";
    private static final Integer SUCCESS_CODE = 200;
    private static final String TOKEN_URL = "/api/BD/token";
    private static final String RECEIVE_CARD_URL = "/api/BD/receiveCardRegister";

    private static final String Error4ConsumerMessage = "出了点小问题，请重新下单";

    /**
     * 手机号简单正则表达式
     */
    private static final String PHONE_SIMPLE_REGULAR = "^1\\d{10}$";

    @Autowired
    private AppDAO appDAO;
    @Autowired
    private HuazhuConfig huazhuConfig;
    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;
    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;
    @Resource(name = "redisTemplate")
    private RedisAtomicClient redisAtomicClient;

    public String getAccessToken(String appId, String secret) {
        String tokenLockKey = RedisKeyEnum.K6001.toString() + appId;
        RedisLock lock = redisAtomicClient.getLock(tokenLockKey, 3);
        if (lock == null) {
            throw new ThirdpatyException("系统繁忙");
        }

        String token = "";
        try {
            String queryTokenKey = RedisKeyEnum.K6002.toString() + appId;
            token = redisTemplate.opsForValue().get(queryTokenKey);
            if (StringUtils.isBlank(token)) {
                LOGGER.info(LOGGER_PREFIX + ",请求接口获取token,appId:{}", appId);
                token = requestForToken(appId, secret);
                if (StringUtils.isNotBlank(token)) {
                    redisTemplate.opsForValue().set(queryTokenKey, token, 50L, TimeUnit.MINUTES);
                }
            } else {
                LOGGER.info(LOGGER_PREFIX + ",获取缓存中的token:{}", token);
            }
        } finally {
            lock.unlock();
        }
        return token;
    }

    private String requestForToken(String appId, String secret) {
        Map<String, String> paramsMap = Maps.newHashMap();
        paramsMap.put("appId", appId);
        paramsMap.put("secret", secret);

        String url = huazhuConfig.getDomain() + TOKEN_URL;
        HttpPost httpPost = new HttpPost(url);
        httpPost.setEntity(new StringEntity(JSON.toJSONString(paramsMap), ContentType.APPLICATION_JSON));

        String response = "";
        try (CloseableHttpResponse closeableHttpResponse = httpClient.execute(httpPost)) {
            HttpEntity entity = closeableHttpResponse.getEntity();
            response = EntityUtils.toString(entity, "UTF-8");
        } catch (IOException e) {
            LOGGER.warn(LOGGER_PREFIX + "获取Token失败", e);
        }

        LOGGER.info(LOGGER_PREFIX + ",获取token接口返回结果:{}", response);

        if (StringUtils.isBlank(response)) {
            throw new ThirdpatyException("获取Token失败");
        }

        JSONObject responseJson = JSONObject.parseObject(response);

        Integer responseCode = responseJson.getInteger(RESPONSE_CODE_KEY);
        String responseData = responseJson.getString("DomainObj");
        if (!responseCode.equals(SUCCESS_CODE) || StringUtils.isBlank(responseData)) {
            String responseDesc = responseJson.getString(RESPONSE_DESC_KEY);
            responseDesc = StringUtils.isNotBlank(responseDesc) ? responseDesc : "请求接口失败";
            throw new ThirdpatyException(responseDesc);
        }
        return responseData;
    }

    @Override
    public String getVirtualTypeCode() {
        return VirtualItemChannelEnum.HUAZHU.getCode();
    }

    @Override
    public HttpRequestBase getVirtualRequest(SupplierRequestDto supplierRequest) {
        LOGGER.info(LOGGER_PREFIX + ",request={}", JSON.toJSONString(supplierRequest));
        //将请求URL的参数转换为MAP
        Map<String, String> params = supplierRequest.getParams();
        String uid = params.get("uid");
        String orderNum = params.get("orderNum");
        String account = params.get("account");
        String goodsNum = params.get("bizParams");

        if (StringUtils.isAnyBlank(uid, orderNum, account, goodsNum)) {
            throw new ThirdpatyException("缺少必要参数");
        }

        //手机号强制校验
        if (!account.matches(PHONE_SIMPLE_REGULAR)) {
            throw new ThirdpatyException("手机号格式错误");
        }

        AppDO appDO = appDAO.getAppByCache(Long.valueOf(supplierRequest.getAppId()));
        Object appConfig = huazhuConfig.getAppConfigBySubjectMap().get(String.valueOf(appDO.getSubject()));
        if (Objects.isNull(appConfig)) {
            throw new ThirdpatyException("配置异常");
        }
        JSONObject appConfigJson = (JSONObject) appConfig;
        String appKey = appConfigJson.getString("appKey");
        String appSecret = appConfigJson.getString("appSecret");

        Map<String, Object> paramsMap = Maps.newHashMap();
        paramsMap.put("mobile", account);
        paramsMap.put("salesId", goodsNum);
        paramsMap.put("orderNo", orderNum);
        paramsMap.put("deviceNo", uid);

        HttpPost httpPost = new HttpPost(huazhuConfig.getDomain() + RECEIVE_CARD_URL);
        //设置请求头
        httpPost.setHeader("AccessToken", getAccessToken(appKey, appSecret));
        httpPost.setHeader("ActivityID", huazhuConfig.getActivityId());
        httpPost.setEntity(new StringEntity(JSON.toJSONString(paramsMap), ContentType.APPLICATION_JSON));
        return httpPost;
    }

    @Override
    public String getVirtualResponse(SupplierRequestDto request, String body) {
        JSONObject result = new JSONObject();
        try {
            JSONObject responseJson = JSONObject.parseObject(body);
            LOGGER.info(LOGGER_PREFIX + ",兑换结果=[{}],订单ID:{}", body, request.getOrderId());
            Integer responseCode = responseJson.getInteger(RESPONSE_CODE_KEY);
            if (SUCCESS_CODE.equals(responseCode)) {
                result.put("status", "success");
            } else {
                result.put("status", "fail");
                result.put("errorMessage", responseJson.getString(RESPONSE_DESC_KEY));
            }
        } catch (Exception e) {
            result.put("status", "fail");
            result.put("errorMessage", e.getMessage());
            LOGGER.info(LOGGER_PREFIX + ",解析虚拟商品兑换请求结果异常,orderId=" + request.getOrderId(), e);
        }
        return result.toJSONString();
    }

    @Override
    public String getDefaultError4ConsumerMessage() {
        return Error4ConsumerMessage;
    }

}

