package cn.com.duiba.tool;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import com.google.common.collect.Maps;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.http.MediaType;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Date;
import java.util.Map;
import java.util.UUID;

/**
 * Created by fangdong on 2020/09/23
 */
public class GuMingSignTool {
    private static final String SIGN_ALGORITHM = "HmacSHA256";

    private GuMingSignTool() {}

    /**
     * 获取签名请求头Map
     * @param urlPath 开发者api接口地址path
     * @param jsonBody json字符串
     * @param appKey 开发者提供的appKey
     * @param appSecret 开发者提供的appSecret
     */
    public static Map<String, String> getSignHeaders(String urlPath, String jsonBody, String appKey, String appSecret) {
        Date now = new Date();
        String uuid = UUID.randomUUID().toString();
        String contentMD5 = getContentMD5(jsonBody);

        Map<String, String> headers = Maps.newHashMap();
        headers.put("x-ca-key", appKey);
        headers.put("x-ca-signature", getSign(now, urlPath, contentMD5, uuid, appKey, appSecret));
        headers.put("x-ca-timestamp", String.valueOf(now.getTime()));
        headers.put("x-ca-nonce", uuid);
        headers.put("content-md5", contentMD5);
        headers.put("x-ca-signature-headers", "x-ca-nonce,x-ca-timestamp,x-ca-key");
        headers.put("date", String.valueOf(now.getTime()));
        headers.put("accept", MediaType.APPLICATION_JSON_UTF8_VALUE);
        headers.put("content-type", MediaType.APPLICATION_JSON_UTF8_VALUE);

        return headers;
    }

    /**
     * 获取验签值
     */
    private static String getSign(Date date, String urlPath, String contentMD5, String uuid, String appKey, String appSecret) {
        String signHeaders = getSignHeaders(appKey, date, uuid);

        StringBuilder builder = new StringBuilder();
        builder.append("POST").append("\n");
        builder.append(MediaType.APPLICATION_JSON_UTF8_VALUE).append("\n");
        builder.append(contentMD5).append("\n");
        builder.append(MediaType.APPLICATION_JSON_UTF8_VALUE).append("\n");
        builder.append(date.getTime()).append("\n");
        builder.append(signHeaders);
        builder.append(urlPath);

        String stringToSign = builder.toString();

        try {
            Mac mac = Mac.getInstance(SIGN_ALGORITHM);
            byte[] keyBytes = appSecret.getBytes(StandardCharsets.UTF_8);
            mac.init(new SecretKeySpec(keyBytes, 0, keyBytes.length, SIGN_ALGORITHM));
            byte[] value = mac.doFinal(stringToSign.getBytes(StandardCharsets.UTF_8));
            return Base64.getEncoder().encodeToString(value);
        } catch (Exception e) {
            throw new ThirdpatyException("古茗sign生成失败");
        }
    }

    /**
     * 组装需要进行sign验签的header
     */
    private static String getSignHeaders(String appKey, Date now, String uuid) {
        Map<String, String> map = Maps.newTreeMap();
        map.put("x-ca-key", appKey);
        map.put("x-ca-nonce", uuid);
        map.put("x-ca-timestamp", String.valueOf(now.getTime()));

        StringBuilder builder = new StringBuilder();
        for (String key : map.keySet()) {
            String value = map.getOrDefault(key, "");
            String header = String.format("%s:%s", key, value);
            builder.append(header).append("\n");
        }
        return builder.toString();
    }

    /**
     * 输入内容md5后再base64
     */
    private static String getContentMD5(String body) {
        byte[] bytes = DigestUtils.md5(body.getBytes(StandardCharsets.UTF_8));
        return Base64.getEncoder().encodeToString(bytes);
    }
}
