package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.InoherbConstant;
import cn.com.duiba.constant.NewInoherbConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.InoherbTools;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.util.*;

/**
 * 相宜本草加减积分新接口
 *
 * @author fja
 * @date 2020-12-08
 */
@Service
public class NewInoherbApi {

    @Resource
    private NewInoherbConfig inoherbConfig;

    @Resource
    private InoherbConstant inoherbConstant;

    @Resource
    private InoherbApi oldInoherbApi;

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private final static String VIRTUAL_CREDITS_PREFIX = "xybc_virtual_credits";

    /**
     * 判断是不是新相宜本草的加减积分
     *
     * @param appId appId
     * @return boolean
     */
    public boolean isNewInoherb(Long appId) {
        return inoherbConfig.getAppIds().contains(appId);
    }

    /**
     * 加积分
     *
     * @param message mq数据
     * @return HttpRequestBase
     */
    public HttpRequestBase getAddCreditsHttpRequest(CreditsMessageDto message) {
        //判断是走老的逻辑还是新的逻辑
        return inoherbConfig.isNewLogic() ? genCreditsRequest(assembleRequestData(message.getAuthParams(), false), InoherbTools.getHostName(message))
                : oldInoherbApi.getAddCreditsHttpRequest(message);
    }

    /**
     * 减积分
     *
     * @param message mq数据
     * @return HttpRequestBase
     */
    public HttpRequestBase getSubCreditsHttpRequest(SubCreditsMsgWrapper message) {
        if (inoherbConfig.isNewLogic()) {
            //新的减积分逻辑
            SubCreditsMsgDto subCreditsMsg = message.getSubCreditsMsg();
            return genCreditsRequest(assembleRequestData(subCreditsMsg.getAuthParams(), true), InoherbTools.getHostName(message));
        } else {
            return oldInoherbApi.getSubCreditsHttpRequest(message);
        }
    }

    /**
     * 生成积分请求体
     *
     * @param params 参数
     * @return HttpPost
     */
    public HttpPost genCreditsRequest(Map<String, Object> params, String host) {
        HttpPost httpPost = new HttpPost(host);
        // 设置参数，为null的参数直接过滤
        List<NameValuePair> list = new ArrayList<>();
        Set<String> keySet = params.keySet();
        for (String key : keySet) {
            Object value = params.get(key);
            if (value instanceof JSONObject) {
                list.add(new BasicNameValuePair(key, JSON.toJSONString(value)));
            } else {
                list.add(new BasicNameValuePair(key, String.valueOf(value)));
            }
        }
        // 设置请求的字符编码及参数
        if (list.size() > 0) {
            try {
                log.info("相宜本草 加减积分请求参数 = {}", list);
                httpPost.setEntity(new UrlEncodedFormEntity(list, "UTF-8"));
            } catch (UnsupportedEncodingException e) {
                log.error("相宜本草生成entity错误 = {}", e.getMessage());
                throw new ThirdpatyException(e);
            }
        }
        return httpPost;
    }

    /**
     * 拼装参数
     *
     * @param originData 数据
     * @return requestData
     */
    private Map<String, Object> assembleRequestData(Map<String, String> originData, boolean isSub) {
        return assembleRequestData(originData, isSub, false);
    }


    private Map<String, Object> assembleRequestData(Map<String, String> originData, boolean isSub, boolean isVirtual) {
        //原始数据
        Map<String, Object> params = Maps.newHashMap();
        params.put("app_id", inoherbConfig.getDeveloperAppId());
        JSONObject requestData = new JSONObject();
        requestData.put("unionId", originData.get("uid"));
        if (isVirtual) {
            //虚拟商品加积分
            String num = originData.get("params");
            String[] s = StringUtils.split(num, "_");
            requestData.put("changeIntegral", new BigDecimal(s[inoherbConfig.getVirtualCreditsIndex()]));
        } else {
            requestData.put("changeIntegral", new BigDecimal(originData.get("credits")));
        }
        requestData.put("uuid", originData.get("orderNum"));
        int processType = isVirtual ? InoherbTools.locateVirtualProcessType(inoherbConfig) : InoherbTools.locateProcessType(originData, inoherbConfig, isSub);
        int issueType = InoherbTools.locateIssueType(processType, inoherbConfig);
        if(!isSub) {
            requestData.put("issueType", issueType);
        }
        requestData.put("processType", processType);
        params.put("requestData", requestData);
        params.put("sign", InoherbTools.genSign(params, inoherbConfig.getDevelopSecret()));
        return params;
    }


    /**
     * 解析返回值
     *
     * @param body       回包
     * @param addCredits 积分
     * @param authParams 数据
     * @return parseCreditsResponse
     */
    public String parseCreditsResponse(String body, Boolean addCredits, Map<String, String> authParams) {
        if(!inoherbConfig.isNewLogic()) {
            return oldInoherbApi.parseCreditsResponse(body, addCredits, authParams);
        }
        log.info("相宜本草加减积分response, add={}, body={}, authParams={}", addCredits, body, JSON.toJSONString(authParams));
        JSONObject responseBody = JSON.parseObject(body);
        JSONObject result = new JSONObject();
        if (Objects.equals("0", responseBody.getString("code"))) {
            result.put("status", "ok");
            // 开发者不返回订单号，自定义
            result.put("bizId", System.currentTimeMillis() + RandomStringUtils.randomNumeric(6));
            result.put("credits", responseBody.getString("data"));
        } else {
            result.put("status", "fail");
            result.put("errorMessage", responseBody.getString("message"));
        }
        return result.toString();
    }


    /**
     * 虚拟商品请求
     */
    public HttpRequestBase getVirtualRequest(SupplierRequest request) {
        if (!inoherbConfig.isNewLogic()) {
            return oldInoherbApi.getVirtualRequest(request);
        }
        String url = request.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> authParams = AssembleTool.getUrlParams(paramsStr);
        String tmp = authParams.get("params");
        log.info("相宜本草 虚拟商品发货 商品号 = {}", tmp);
        if (StringUtils.isBlank(tmp)) {
            throw new ThirdpatyException("相宜本草 虚拟商品发货 商品号为空");
        }
        if (tmp.contains(VIRTUAL_CREDITS_PREFIX)) {
            log.info("相宜本草 虚拟商品发货 加积分原始数据 = {}", authParams);
            return genCreditsRequest(assembleRequestData(authParams, false, true), inoherbConfig.getVirtualProductAddCreditsUrl());
        } else {
            String timestamp = String.valueOf(System.currentTimeMillis());
            String[] params = StringUtils.split(authParams.get("params"), "|");
            Map<String, String> body = Maps.newHashMap();
            body.put("code", "sendCouponsInfo");
            body.put("unionId", authParams.get("uid"));
            body.put("cardId", params[0]);
            body.put("money", params[1]);
            body.put("activityId", String.valueOf(System.currentTimeMillis() / 1000));
            body.put("timestamp", timestamp);
            body.put("token", this.getToken(timestamp));
            log.info("相宜本草 虚拟商品发货 url = {}, 数据 = {}", newUrl, JSON.toJSONString(body));
            request.setHttpUrl(newUrl);
            request.setAuthParams(body);
            return this.getHttpPost(newUrl, JSON.toJSONString(body));
        }
    }


    /**
     * 获取加减积分http请求
     */
    private HttpRequestBase getHttpPost(String httpUrl, String jsonBody) {
        HttpPost request = new HttpPost(httpUrl);
        request.setEntity(new StringEntity(jsonBody, ContentType.APPLICATION_JSON));

        return request;
    }


    /**
     * 获取开发者第三方token
     */
    public String getToken(String timestamp) {
        return DigestUtils.md5Hex(inoherbConstant.getDeveloperThirdPartySecret() + timestamp).toUpperCase();
    }


    /**
     * 解析虚拟商品充值响应结果
     */
    public String getVirtualResponse(SupplierRequest message, String body) {
        if (!inoherbConfig.isNewLogic()) {
            return oldInoherbApi.getVirtualResponse(message, body);
        }
        //log.info("相宜本草虚拟商品充值响应结果response, message = {}, body={}", JSON.toJSONString(message), body);
        JSONObject responseBody = JSON.parseObject(body);
        JSONObject result = new JSONObject();
        if (responseBody.containsKey("status")) {
            if (!Objects.equals(1, responseBody.getInteger("status"))) {
                result.put("status", "fail");
                result.put("errorMessage", ObjectUtils.firstNonNull(responseBody.getString("refusereason"), responseBody.getString("msg")));
            } else {
                result.put("status", "success");
            }
        } else {
            if (Objects.equals("0", responseBody.getString("code"))) {
                result.put("status", "success");
            } else {
                result.put("status", "fail");
                result.put("errorMessage", responseBody.getString("message"));
            }
        }
        return result.toString();
    }
}
