package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.boot.profiler.DBTimeProfiler;
import cn.com.duiba.constant.ReconciliationConfig;
import cn.com.duiba.credits.sdk.CreditConsumeParams;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.service.reconciliation.ReconciliationRecordService;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.thirdparty.dto.reconciliation.ReconciliationRecordDto;
import cn.com.duiba.thirdparty.enums.reconciliation.ReconciliationLogTypeEnum;
import cn.com.duiba.thirdparty.enums.reconciliation.ReconciliationRespStatusEnum;
import cn.com.duiba.tool.AssembleTool;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @program: thirdparty-all
 * @description: 对账定制api
 * @author: Simba
 * @create: 2020-02-18 15:49
 **/
@Service
public class ReconciliationBizService {

    private static final Logger logger = LoggerFactory.getLogger(ReconciliationBizService.class);

    @Autowired
    private ReconciliationConfig reconciliationConfig;

    @Resource(name = "reconExecutorService")
    private ExecutorService executorService;

    @Autowired
    private ReconciliationRecordService reconciliationRecordService;

    private boolean isReconcile(Long appId) {
    	if(appId == null || CollectionUtils.isEmpty(reconciliationConfig.getAppIdSet())){
    		return false;
		}

        return reconciliationConfig.getAppIdSet().contains(appId);
    }

    /**
     * 保存对账记录 入口是rpc
     * @param request
     * @return
     */
	public void saveRpcSubCreditsMsg(CreditsMessage request) {
        if(!isReconcile(Long.valueOf(request.getAppId()))){
            return;
        }
        logger.info("reconcile rpc减积分：{}", JSONObject.toJSONString(request));

        executorService.submit(() -> {
            try{
                String url = request.getHttpUrl();
                String orderNum = request.getRelationId();

                Map<String, String> authParamMap;
                if(Objects.equals(request.getHttpType(), CreditsMessage.HTTP_POST)){
                    authParamMap = request.getAuthParams();
                }else{
                    String authParams = url.substring(url.indexOf('?') + 1);
                    authParamMap = AssembleTool.getUrlParams(authParams);
                }

                if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
                    orderNum = authParamMap.get("orderNum");
                }

                ReconciliationRecordDto dto = wrapReconciliationLog(Long.valueOf(request.getAppId()), orderNum, request.getRelationType(), authParamMap, request.getParams(), ReconciliationLogTypeEnum.SUB_CREDITS);
                reconciliationRecordService.saveOrUpdate(dto);
            }catch (Exception e){
                logger.error("save对账记录出错, appId:{}, relationId:{}, relationType:{}, ", request.getAppId(), request.getRelationId(), request.getRelationType(), e);
            }
        });
	}

    /**
     * 保存对账记录 入口是rocketmq
     * @param subCreditsMsgWrapper
     * @return
     */
    public void saveMqSubCreditsMsg(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        if(!isReconcile(subCreditsMsgWrapper.getSubCreditsMsg().getAppId())){
            return;
        }
        logger.info("reconcile mq减积分：{}", JSONObject.toJSONString(subCreditsMsgWrapper));

        executorService.submit(() -> {
            try{
                ReconciliationRecordDto dto = wrapReconciliationLog(subCreditsMsgWrapper);
                reconciliationRecordService.saveOrUpdate(dto);
            }catch (Exception e){
                logger.error("save对账记录出错, appId:{}, relationId:{}, relationType:{}, ", subCreditsMsgWrapper.getSubCreditsMsg().getAppId(), subCreditsMsgWrapper.getSubCreditsMsg().getRelationId(), subCreditsMsgWrapper.getSubCreditsMsg().getRelationType(), e);
            }
        });
    }

    /**
     * 构造扣积分请求参数-入口是rocketmq
     * 加/扣积分：兑吧订单号、用户id、操作类型（加积分、扣积分）、时间、活动ID、状态（成功、失败）、积分值
     */
    private ReconciliationRecordDto wrapReconciliationLog(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        SubCreditsMsgDto subCreditsMsgDto = subCreditsMsgWrapper.getSubCreditsMsg();
        CreditConsumeParams creditConsumeParams = subCreditsMsgDto.getCreditConsumeParams();
	    Map<String, String> paramMap = subCreditsMsgDto.getParams();
        Map<String, String> authParamMap = subCreditsMsgDto.getAuthParams();

        ReconciliationRecordDto reconciliationRecordDto = new ReconciliationRecordDto();
        reconciliationRecordDto.setOrderNum(subCreditsMsgDto.getRelationId());
        reconciliationRecordDto.setPartnerUserId(creditConsumeParams.getUid());
        reconciliationRecordDto.setLogType(ReconciliationLogTypeEnum.SUB_CREDITS.getCode());
        reconciliationRecordDto.setSendTime(new Date());
        if(MapUtils.isNotEmpty(paramMap)){
            reconciliationRecordDto.setOptId(paramMap.get("opId"));
        }

        //覆写orderNum
        if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
            reconciliationRecordDto.setOrderNum(authParamMap.get("orderNum"));
        }

        if(subCreditsMsgDto.getRelationType() != null){
            reconciliationRecordDto.setOptType(subCreditsMsgDto.getRelationType().getMsg());
        }
        reconciliationRecordDto.setCredits(creditConsumeParams.getCredits());
        reconciliationRecordDto.setAppId(subCreditsMsgDto.getAppId());

        return reconciliationRecordDto;
    }

    /**
     *
     * 构造扣积分请求参数-入口是rpc
     * 加/扣积分：兑吧订单号、用户id、操作类型（加积分、扣积分）、时间、活动ID、状态（成功、失败）、积分值
     */
    private ReconciliationRecordDto wrapReconciliationLog(Long appId, String relationId, String relationType, Map<String, String> authParamMap, Map<String, String> paramMap, ReconciliationLogTypeEnum logTypeEnum) {
        ReconciliationRecordDto reconciliationRecordDto = new ReconciliationRecordDto();
        reconciliationRecordDto.setOrderNum(relationId);
        reconciliationRecordDto.setPartnerUserId(authParamMap.get("uid"));
        reconciliationRecordDto.setLogType(logTypeEnum.getCode());
        reconciliationRecordDto.setSendTime(new Date());

        if(MapUtils.isNotEmpty(paramMap)){
            reconciliationRecordDto.setOptId(paramMap.get("opId"));
        }

        if(StringUtils.isNotBlank(relationType)){
            reconciliationRecordDto.setOptType(relationType);
        }
        String credits = authParamMap.get("credits");
        if(StringUtils.isNotBlank(credits)){
            reconciliationRecordDto.setCredits(Long.valueOf(credits));
        }

        reconciliationRecordDto.setAppId(appId);

        return reconciliationRecordDto;
    }

    /**
     * 构造加积分请求
     */
    public void saveAddCreditsMsg(CreditsMessageDto request){
        if(!isReconcile(Long.valueOf(request.getAppId()))){
            return;
        }
        logger.info("reconcile 加积分：{}", JSONObject.toJSONString(request));

        executorService.submit(() -> {
            try{
                String url = request.getHttpUrl();
                String orderNum = request.getRelationId();
                Map<String, String> authParamMap;
                if(Objects.equals(request.getHttpType(), CreditsMessage.HTTP_POST)){
                    authParamMap = request.getAuthParams();
                }else{
                    String authParams = url.substring(url.indexOf('?') + 1);
                    authParamMap = AssembleTool.getUrlParams(authParams);
                }
                if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
                    orderNum = authParamMap.get("orderNum");
                }

                ReconciliationRecordDto dto = wrapReconciliationLog(Long.valueOf(request.getAppId()), orderNum, request.getRelationType(), authParamMap, request.getParams(), ReconciliationLogTypeEnum.ADD_CREDITS);
                reconciliationRecordService.saveOrUpdate(dto);
            }catch (Exception e){
                logger.error("save对账记录出错, appId:{}, relationId:{}, relationType:{}, ", request.getAppId(), request.getRelationId(), request.getRelationType(), e);
            }
        });
    }

    public void mallExchangeNotify(ReconciliationRecordDto dto){
        if(!isReconcile(dto.getAppId())){
            return;
        }
        logger.info("reconcile 普兑订单：{}", JSONObject.toJSONString(dto));
        reconciliationRecordService.saveOrUpdate(dto);
    }

    /**
     * 解析积分响应
     * @return
     */
    public void updateCreditsRsp(CreditsMessageDto request, String body, Boolean addCredits) {
        try{
            Long appId = Long.valueOf(request.getAppId());
            if(!isReconcile(appId)){
                return;
            }

            String relationType = request.getRelationType();
            String relationId = request.getRelationId();
            String url = request.getHttpUrl();
            Map<String, String> authParamMap;
            if(Objects.equals(request.getHttpType(), CreditsMessage.HTTP_POST)){
                authParamMap = request.getAuthParams();
            }else{
                String authParams = url.substring(url.indexOf('?') + 1);
                authParamMap = AssembleTool.getUrlParams(authParams);
            }
            if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
                relationId = authParamMap.get("orderNum");
            }
            logger.info("reconcile update：relation id:{}, appId:{}, relationType:{}, body:{}", request.getRelationId(), appId, relationType, body);

            updateCreditsRsp(appId, body, addCredits, relationId, relationType);
        }catch (Exception e){
            logger.error("更新对账记录出错, request:{}, body:{}, addCredits:{}, ", JSONObject.toJSONString(request), body, addCredits, e);
        }
    }

    /**
     * 解析积分响应
     * @return
     */
    public void updateCreditsRsp(CreditsMessage request, String body, Boolean addCredits) {
        try{
            Long appId = Long.valueOf(request.getAppId());
            if(!isReconcile(appId)){
                return;
            }

            String relationType = request.getRelationType();
            String relationId = request.getRelationId();
            String url = request.getHttpUrl();
            Map<String, String> authParamMap;
            if(Objects.equals(request.getHttpType(), CreditsMessage.HTTP_POST)){
                authParamMap = request.getAuthParams();
            }else{
                String authParams = url.substring(url.indexOf('?') + 1);
                authParamMap = AssembleTool.getUrlParams(authParams);
            }
            if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
                relationId = authParamMap.get("orderNum");
            }
            logger.info("reconcile update：relation id:{}, appId:{}, relationType:{}, body:{}", request.getRelationId(), appId, relationType, body);

            updateCreditsRsp(appId, body, addCredits, relationId, relationType);
        }catch (Exception e){
            logger.error("更新对账记录出错, request:{}, body:{}, addCredits:{}, ", JSONObject.toJSONString(request), body, addCredits, e);
        }
    }

    /**
     * 解析积分响应
     * @return
     */
    public void updateCreditsRsp(SubCreditsMsgWrapper subCreditsMsgWrapper, String body, Boolean addCredits) {
        try{
            SubCreditsMsgDto request = subCreditsMsgWrapper.getSubCreditsMsg();

            Long appId = Long.valueOf(request.getAppId());
            if(!isReconcile(appId)){
                return;
            }
            String relationId = request.getRelationId();
            Map<String, String> authParamMap = request.getAuthParams();
            if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
                relationId = authParamMap.get("orderNum");
            }
            String relationType = null;
            if(request.getRelationType() != null){
                relationType = request.getRelationType().getMsg();
            }

            logger.info("reconcile update：relation id:{}, appId:{}, relationType:{}, body:{}", request.getRelationId(), appId, relationType, body);

            updateCreditsRsp(appId, body, addCredits, relationId, relationType);
        }catch (Exception e){
            logger.error("更新对账记录出错, request:{}, body:{}, addCredits:{}, ", JSONObject.toJSONString(subCreditsMsgWrapper.getSubCreditsMsg()), body, addCredits, e);
        }
    }

    /**
     * 解析积分响应
     * @param appId
     * @param body
     * @param addCredits
     * @param relationId
     * @param relationType
     * @return
     */
    private void updateCreditsRsp(Long appId, String body, Boolean addCredits, String relationId, String relationType) {
        if(!isReconcile(appId)){
            return;
        }
        logger.info("reconcile update：relation id:{}, appId:{}, relationType:{}, body:{}", relationId, appId, relationType, body);

        executorService.submit(() -> {
            try{
                ReconciliationRecordDto reconciliationRecordDto = new ReconciliationRecordDto();
                reconciliationRecordDto.setOrderNum(relationId);
                reconciliationRecordDto.setLogType(ReconciliationLogTypeEnum.SUB_CREDITS.getCode());
                reconciliationRecordDto.setAppId(appId);
                reconciliationRecordDto.setOptType(relationType);
                if(addCredits){
                    reconciliationRecordDto.setLogType(ReconciliationLogTypeEnum.ADD_CREDITS.getCode());
                }
                parseRespBody(body, reconciliationRecordDto);

                Pair<Integer, Boolean> pair = reconciliationRecordService.update(reconciliationRecordDto);
                //注：更新线程可能比保存线程执行的更快，所以需要重试
                if(BooleanUtils.isFalse(pair.getRight())){
                    retry(0, reconciliationRecordDto);
                }
            }catch (Exception e){
                logger.error("update响应结果出错, appId:{}, relationId:{}, relationType:{}, body:{}, ", appId, relationId, relationType, body, e);
            }
        });
    }

    /**
     * 重试逻辑
     * @param retryTimes
     * @param reconciliationRecordDto
     */
    @DBTimeProfiler
    private void retry(int retryTimes, ReconciliationRecordDto reconciliationRecordDto) {
        retryTimes++;
        try {
            // 超过重试次数，不更新
            if (retryTimes > 3) {
                logger.error("对账更新重试{}次，依旧失败，取消重试，dto={}", 3, JSONObject.toJSONString(reconciliationRecordDto));
                return;
            }
            // 线程休眠1s
            TimeUnit.SECONDS.sleep(1 );
            Pair<Integer, Boolean> pair = reconciliationRecordService.update(reconciliationRecordDto);
            if(BooleanUtils.isFalse(pair.getRight())){
                retry(retryTimes, reconciliationRecordDto);
            }
        } catch (InterruptedException e) {
            logger.warn("", e);
        } catch (Exception e) {
            logger.warn("", e);
            retry(retryTimes, reconciliationRecordDto);
        }
    }

    private void parseRespBody(String body, ReconciliationRecordDto dto){
        String respBody = body;
        if(StringUtils.isNotBlank(body)){
            dto.setResponseStatus(ReconciliationRespStatusEnum.FAIL.getCode());
            JSONObject json = JSON.parseObject(body);
            if (json != null && ("success".equalsIgnoreCase(json.getString("status")) || "ok".equalsIgnoreCase(json.getString("status")))) {
                dto.setResponseStatus(ReconciliationRespStatusEnum.SUCCESS.getCode());
            }

            if(respBody.length() > 255){
                respBody = respBody.substring(0, 255);
            }
            dto.setResponseBody(respBody);
        }
    }

    public void saveVirtualAddCreditsMsg(SupplierRequest request) {
        if(!isReconcile(Long.valueOf(request.getAppId()))){
            return;
        }
        if (!request.isSaveReconciliationRecord()) {
            return;
        }
        logger.info("saveVirtualAddCreditsMsg 加积分：{}", JSONObject.toJSONString(request));

        executorService.submit(() -> {
            try{
                Map<String, String> authParamMap =request.getAuthParams() ;
                String orderNum = authParamMap.get("orderNum");

                ReconciliationRecordDto dto = wrapReconciliationLog(Long.valueOf(request.getAppId()), orderNum, null, authParamMap, null, ReconciliationLogTypeEnum.ADD_CREDITS);
                reconciliationRecordService.saveOrUpdate(dto);
            }catch (Exception e){
                logger.error("save对账记录出错, SupplierRequest:{}, ", JSON.toJSONString(request), e);
            }
        });
    }

    public void updateVirtualAddCreditsRsp(SupplierRequest request, String body) {
        try{
            Long appId = Long.valueOf(request.getAppId());
            if(!isReconcile(appId)){
                return;
            }
            if (!request.isSaveReconciliationRecord()) {
                return;
            }
            Map<String, String> authParamMap = request.getAuthParams();
            String relationId = authParamMap.get("orderNum");
            logger.info("reconcile update：relation id:{}, appId:{}, relationType:{}, body:{}",JSONObject.toJSONString(request), body);

            updateCreditsRsp(appId, body, true, relationId, null);
        }catch (Exception e){
            logger.error("更新对账记录出错", e);
        }
    }
}

