package cn.com.duiba.biz.credits;

import cn.com.duiba.api.tools.InnerLogger;
import cn.com.duiba.dao.HttpMessageDAO;
import cn.com.duiba.domain.HttpMessageDO;
import cn.com.duiba.order.center.api.dto.CreditsCallbackMessage;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.service.HttpRetryRulesService;
import cn.com.duiba.thirdparty.dto.HttpRequestMessageDto;
import cn.com.duiba.thirdparty.dto.VirtualCurrencyCallbackMessage;
import cn.com.duiba.thirdparty.dto.VirtualCurrencyMessage;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.CaiNiaoTool;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.entity.GzipDecompressingEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.common.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.nio.charset.Charset;
import java.util.Map;


/**
 * @program: thirdparty-all
 * @description: 扣减虚拟货币
 * @author: Simba
 * @create: 2019-03-15 18:10
 **/
@Service
public class SubVirtualCurrencyToDeveloper {

    private static final Logger LOGGER = LoggerFactory.getLogger(SubVirtualCurrencyToDeveloper.class);

    @Autowired
    private HttpAsyncClientPool httpAsyncClientPool;

    @Autowired
    private DefaultMQProducer rocketMqProducer;
    @Autowired
    private HttpMessageDAO httpMessageDAO;
    @Autowired
    private HttpRetryRulesService httpRetryRulesService;

    /**
     * 提交扣除虚拟货币HTTP请求
     *
     * @param request
     * @param msgTopic
     * @param msgTag
     * @param msgKey
     */
    public void submit(VirtualCurrencyMessage request, final String msgTopic, final String msgTag, final String msgKey, Long httpMessageId, Integer httpMessageNumber) {
        HttpRequestBase http;
        if (CreditsMessage.HTTP_POST.equals(request.getHttpType())) {
            Map<String, String> authParams = request.getAuthParams();
            if (MapUtils.isNotEmpty(authParams)) {
                // 移除appSecret
                authParams.remove("appSecret");
            }
            http = AssembleTool.assembleRequest(request.getHttpUrl(), request.getAuthParams());
            HttpRequestLog.logUrl("[action subVirtualCurrency][request bizId " + request.getRelationId() + "] [type " + request.getRelationType() + "] [post url " + request.getHttpUrl() + "][authParams " + request.getAuthParams() + "][consumerId " + request.getConsumerId() + "]");
        } else {
            http = new HttpGet(request.getHttpUrl());
            HttpRequestLog.logUrl("[action subVirtualCurrency][request bizId " + request.getRelationId() + "] [type " + request.getRelationType() + "] [get url " + request.getHttpUrl() + "][consumerId " + request.getConsumerId() + "]");
        }

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("consumerId", request.getConsumerId());
        jsonObject.put("appId", request.getAppId());
        jsonObject.put("currencies", getValue(request.getParams(), "currencies"));
        jsonObject.put("relationId", request.getRelationId());
        jsonObject.put("relationType", request.getRelationType());
        jsonObject.put("httpType", request.getHttpType());
        InnerLogger.log(2, 62, "", "", jsonObject);

        httpAsyncClient(request, http, msgTopic, msgTag, msgKey, httpMessageId, httpMessageNumber);
    }

    private String getValue(Map<String, String> params, String key) {
        if (MapUtils.isEmpty(params)) {
            return null;
        }
        return params.get(key);
    }

    private void httpAsyncClient(final VirtualCurrencyMessage request, HttpRequestBase http, final String msgTopic, final String msgTag, final String msgKey, Long httpMessageId, Integer httpMessageNumber) {

        httpAsyncClientPool.submit(request.getAppId(), http, new FutureCallback<HttpResponse>() {
            @Override
            public void completed(HttpResponse response) {
                VirtualCurrencyCallbackMessage resp = new VirtualCurrencyCallbackMessage();
                try {
                    resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_COMPLETED);
                    Header header = response.getEntity().getContentEncoding();
                    String result;
                    if (header != null && header.toString().contains(CaiNiaoTool.CONTENT_ENCODING_GZIP)) {
                        result = EntityUtils.toString(new GzipDecompressingEntity(response.getEntity()));
                    } else {
                        result = EntityUtils.toString(response.getEntity());
                    }
                    resp.setMessage(HttpRequestLog.subBody(result));
                } catch (Exception e) {
                    LOGGER.error("toDeveloper completed", e);
                } finally {
                    finallyBlock(request, resp, msgTopic, msgTag, msgKey, Integer.toString(response.getStatusLine().getStatusCode()), httpMessageId, httpMessageNumber);
                }
            }

            @Override
            public void failed(Exception ex) {
                LOGGER.error("toDeveloper failed bizId:" + request.getRelationId() + " bizType:" + request.getRelationType(), ex);
                VirtualCurrencyCallbackMessage resp = new VirtualCurrencyCallbackMessage();
                try {
                    resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_FAILED);
                    resp.setMessage(ex.getClass().getName() + ":" + ex.getMessage());
                } catch (Exception e) {
                    LOGGER.error("toDeveloper failed", e);
                } finally {
                    finallyBlock(request, resp, msgTopic, msgTag, msgKey, "", httpMessageId, httpMessageNumber);
                }
            }

            @Override
            public void cancelled() {
                LOGGER.info("toDeveloper cancelled bizId:" + request.getRelationId() + " bizType:" + request.getRelationType());
                VirtualCurrencyCallbackMessage resp = new VirtualCurrencyCallbackMessage();
                try {
                    resp.setCallbackType(CreditsCallbackMessage.CALLBACK_TYPE_FAILED);
                    resp.setMessage("http cancelled");
                } catch (Exception e) {
                    LOGGER.error("toDeveloper cancelled", e);
                } finally {
                    finallyBlock(request, resp, msgTopic, msgTag, msgKey, "", httpMessageId, httpMessageNumber);
                }
            }

        });
    }

    /**
     * 返回响应结果到业务方
     *
     * @param req
     * @param resp
     * @param msgTopic
     * @param msgTag
     * @param msgKey
     */
    private void finallyBlock(VirtualCurrencyMessage req, VirtualCurrencyCallbackMessage resp, String msgTopic, String msgTag, String msgKey, String code, Long httpMessageId, Integer httpMessageNumber) {
        HttpRequestLog.logUrl("[action subVirtualCurrency] [tag response] [code " + code + "] [bizId " + req.getRelationId() + "] [type " + req.getRelationType() + "] [callback " + resp.getCallbackType() + "] [body " + resp.getMessage() + "]");
        req.setMsgTopic(msgTopic);
        req.setMsgTag(msgTag);
        req.setMsgKey(msgKey);
        HttpMessageDO message = new HttpMessageDO();
        message.setAppId(Long.valueOf(req.getAppId()));
        message.setBizType(HttpRequestMessageDto.RETRY_SUB_VIRTUAL_CURRENCY);
        message.setBizParams(JsonTool.objectToJson(req));
        message.setId(httpMessageId);
        message.setNumber(httpMessageNumber);
        if (nextRetry(message, resp)) {
            return;
        }
        try {
            resp.setRecordId(req.getRecordId());
            resp.setAccountId(req.getAccountId());
            resp.setRelationId(req.getRelationId());
            resp.setRelationType(req.getRelationType());
            resp.setParams(req.getParams());
            resp.setAppId(req.getAppId());
            resp.setConsumerId(req.getConsumerId());
            resp.setHttpUrl(req.getHttpUrl());
            String body = JsonTool.objectToJson(resp);
            sendRocketMQMessage(msgTopic, msgTag, msgKey, body);

        } catch (Exception e) {
            LOGGER.error("virtualCurrency callback: bizId:" + req.getRelationId() + " bizType:" + req.getRelationType(), e);
        }
    }

    private void sendRocketMQMessage(String topic, String tag, String key, String message) {
        Message msg = new Message(topic, tag, key, message.getBytes(Charset.forName("utf-8")));
        try {
            rocketMqProducer.send(msg);
        } catch (Exception e) {
            LOGGER.error("subVirtualCurrency sendRocketMQMessage", e.getMessage());
        }
    }

    /**
     * 判断是否需要重试<br/>
     * 1.记录重试数据<br/>
     * 2.定时扫描<br/>
     *
     * @param message
     * @param resp
     * @return true 下次重试， false 无需重试
     */
    public boolean nextRetry(HttpMessageDO message, VirtualCurrencyCallbackMessage resp) {
        try {
            // 如果响应包含OK认为是成功
            if (isSuccess(resp)) {
                httpMessageDAO.delete(message.getId());
                return false;
            }
            // 是否超过重试次数
            if (message.getNumber() >= httpRetryRulesService.getRetryNumber(message)) {
                httpMessageDAO.delete(message.getId());
                return false;
            }
            // 更新下次重试时间
            httpRetryRulesService.updateNextTime(message);
            return true;
        } catch (Exception e) {
            LOGGER.error("nextRetry", e);
        }
        return false;
    }

    /**
     * 解析响应的内容是否为成功
     *
     * @param resp
     * @return
     */
    public boolean isSuccess(VirtualCurrencyCallbackMessage resp) {
        try {
            if (CreditsCallbackMessage.CALLBACK_TYPE_COMPLETED.equals(resp.getCallbackType())) {
                String body = resp.getMessage().toLowerCase();
                if (body.contains("ok") || body.contains("success")) {
                    return true;
                }
            }
            return false;
        } catch (Exception e) {
            LOGGER.error("isSuccess", e);
            return true;
        }
    }
}

