package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.SuningConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.JsonTool;
import cn.com.duiba.tool.suning.EncryptUtil;
import cn.com.duiba.tool.suning.SuningSignUtils;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by sunyan on 2020/1/16.
 */
@Service
public class SuningApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(SuningApi.class);
    private static final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyyMMddHHmmss");

    @Autowired
    private SuningConfig suningConfig;
    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;


    /**
     * 判断当前app是否是苏宁金融的app
     * @param appId
     * @return
     */
    public boolean isSuningApp(Long appId) {
        return suningConfig.getAppIds().contains(appId);
    }

    /**
     * 组装扣积分对象  remote形式发起的扣积分请求
     * @param request
     * @return
     */
    public CreditsMessage getSubCreditsMessage(CreditsMessage request) throws BizException {
        Map<String, String> originAuthParams = request.getAuthParams();
        JSONObject params = new JSONObject();
        params.put("openId",originAuthParams.get("uid"));
        params.put("sceneCode","01");
        params.put("deductionCatFood",originAuthParams.get("credits"));
        params.put("reqNo",originAuthParams.get("orderNum"));
        request.setHttpUrl(suningConfig.getDeductionUrl());
        Map<String, String> authParams = signParams(params);
        authParams.put("postType","raw");
        authParams.put("unicodeType","UTF-8");
        request.setAuthParams(authParams);
        request.setHttpType(CreditsMessage.HTTP_POST);
        return request;
    }


    /**
     * 组装扣积分对象  mq形式发起的扣积分请求
     * @param subCreditsMsgWrapper
     * @return
     */
    public SubCreditsMsgWrapper getSubCreditsMessage(SubCreditsMsgWrapper subCreditsMsgWrapper) throws BizException {
        SubCreditsMsgDto msg = subCreditsMsgWrapper.getSubCreditsMsg();
        Map<String, String> originAuthParams = msg.getAuthParams();
        JSONObject params = new JSONObject();
        params.put("openId",originAuthParams.get("uid"));
        params.put("sceneCode","01");
        params.put("deductionCatFood",originAuthParams.get("credits"));
        params.put("reqNo",originAuthParams.get("orderNum"));
        subCreditsMsgWrapper.setHttpUrl(suningConfig.getDeductionUrl());
        Map<String, String> authParams = signParams(params);
        authParams.put("postType","raw");
        authParams.put("unicodeType","UTF-8");
        msg.setAuthParams(authParams);
        msg.setHttpType(SubCreditsMsgDto.HTTP_POST);
        return subCreditsMsgWrapper;
    }

    public Map<String,String> signParams(JSONObject params){
        Map<String, Object> bizMap = new HashMap<>();
        bizMap.put("appId", suningConfig.getOpenAppId());
        bizMap.put("version", "1.0");
        bizMap.put("timestamp", getDateStr(new Date()));
        bizMap.put("signType", "RSA2");
        bizMap.put("signkeyIndex", suningConfig.getSignkeyIndex());
        String paramsJson = params.toJSONString();
        bizMap.put("params",paramsJson);
        //加签
        SuningSignUtils.getSign(bizMap, suningConfig.getDuibaPrivateKey());
        //加密
        //16位AES密钥字符串
        String aesKey = suningConfig.getDuibaAesKey();
        SuningSignUtils.encrypt(aesKey, suningConfig.getSuningPublicKey(), bizMap);
        Map<String,String> map = SuningSignUtils.toStringMap(bizMap);
        return map;
    }


    private static String getDateStr(Date date) {
        synchronized (simpleDateFormat) {
            if (date == null) {
                return "";
            }
            return simpleDateFormat.format(date);
        }
    }

    /**
     * 转义扣积分响应数据
     *
     * @param body
     * @return
     */
    public String getSuningResponseCredis(String body,Map<String, String> authParams) {
        //16位AES密钥字符串
        String aesKey = suningConfig.getDuibaAesKey();
        Map<String, String> duibaDoc = new HashMap<>();
        Map<String, Object>  ret = JSONObject.parseObject(body);
        Map<String, Object> paramMap = SuningSignUtils.decrypt(aesKey, ret);
        String code = String.valueOf(paramMap.get("responseCode"));
        if(StringUtils.equals("0000",code)){
            duibaDoc.put("status", "ok");
        } else {
            duibaDoc.put("status", "fail");
        }
        duibaDoc.put("errorMessage", String.valueOf(paramMap.get("responseMsg")));
        //获取用户积分
        String str = authParams.get("params");
        str = EncryptUtil.decryptBase64DecorateAES(str, aesKey);
        JSONObject requestParams = JSONObject.parseObject(str);
        JSONObject params = new JSONObject();
        duibaDoc.put("bizId", requestParams.get("reqNo")+"_suning");
        params.put("openId",requestParams.get("openId"));
        Map<String, String> creditsParams = signParams(params);
        String credits = getCredits(suningConfig.getUserInfoUrl(),creditsParams,aesKey);
        duibaDoc.put("credits", credits);
        return JsonTool.objectToJson(duibaDoc);
    }


    private String getCredits(String urlGrade, Map<String, String> params,String aesKey){
        String credits = null;
        try {
            HttpPost post = new HttpPost(urlGrade);
            StringEntity entity = new StringEntity(JSONObject.toJSONString(params));
            entity.setContentEncoding("utf-8");
            entity.setContentType("application/json");
            post.setEntity(entity);
            try (CloseableHttpResponse response = httpClient.execute(post)) {
                String result = EntityUtils.toString(response.getEntity());
                Map<String, Object>  ret = JSONObject.parseObject(result);
                Map<String, Object> paramMap = SuningSignUtils.decrypt(aesKey, ret);
                String code = String.valueOf(paramMap.get("responseCode"));
                if(StringUtils.equals("0000",code)){
                    credits = String.valueOf(paramMap.get("catFoodRemainNum"));
                }
            } catch (Exception e) {
                LOGGER.error("苏宁定制积分查询接口调用异常 params={},error={}", JSONObject.toJSONString(params), e.getMessage());
            }
            return credits;
        } catch (Exception e) {
            LOGGER.error("苏宁定制积分查询接口调用异常 params={},error={}", JSONObject.toJSONString(params), e.getMessage());
            return null;
        }
    }

    public SupplierRequest getVirtualRequest(SupplierRequest request) {
        String url = request.getHttpUrl();
        String requestParams = url.substring(url.indexOf('?') + 1, url.length());
        // 1.将请求URL的参数转换为MAP
        Map<String, String> crediMap = AssembleTool.getUrlParams(requestParams);
        JSONObject params = new JSONObject();
        params.put("openId",crediMap.get("uid"));
        params.put("source","04");
        //商家编码格式：活动编码|礼包编码
        String code = crediMap.get("params");
        String[] codes = code.split("\\|");
        if(codes.length==2){
            params.put("activityCode",codes[0]);
            params.put("giftCode",codes[1]);
        }else{
            params.put("activityCode",crediMap.get("params"));
        }
        params.put("outOrderNo",crediMap.get("orderNum"));
        //发放来源 固定传值
        params.put("grantSource",suningConfig.getGrantSource());
        request.setHttpUrl(suningConfig.getVirtualUrl());
        Map<String, String> authParams = signParams(params);
        authParams.put("postType","raw");
        authParams.put("unicodeType","UTF-8");
        request.setAuthParams(authParams);
        return request;
    }

    /**
     * 获取虚拟商品响应信息
     *
     * @param message
     * @param body
     * @return
     */
    public String getVirtualResponse(SupplierRequest message, String body) {
        Map<String, String> duibaDoc = new HashMap<>();
        try {
            String aesKey = suningConfig.getDuibaAesKey();
            Map<String, Object>  ret = JSONObject.parseObject(body);
            if (null == ret) {
                throw new BizException("开发者接口响应内容异常");
            }
            Map<String, Object> paramMap = SuningSignUtils.decrypt(aesKey, ret);
            String code = String.valueOf(paramMap.get("responseCode"));
            if (!StringUtils.equals("0000",code)) {
                throw new BizException(String.valueOf(paramMap.get("responseMsg")));
            }
            duibaDoc.put("status", "success");
            String data = JSONObject.toJSONString(paramMap);
            if (StringUtils.isNotBlank(data) && data.length() > 1900) {
                data = data.substring(0, 1900);
            }
            duibaDoc.put("data",data);
        } catch (Exception e) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", e.getMessage());
            LOGGER.error("[SuningApi-getVirtualResponse] Params:[message={},body={}], Msg:苏宁金融-虚拟商品接口响应解析异常", JSONObject.toJSONString(message), body, e);
        }
        return JsonTool.objectToJson(duibaDoc);
    }


}
