package cn.com.duiba.biz.credits.impl;

import cn.com.duiba.biz.credits.CiticApi;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.CiticBankConfig;
import cn.com.duiba.credits.sdk.SignTool;
import cn.com.duiba.dao.AppDAO;
import cn.com.duiba.domain.AppDO;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @Description:南京中信银行-虚拟商品接口
 * @Author: fxr
 * @Date: 2019/4/3
 */
@Service
public class CiticApiImpl implements CiticApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(CiticApiImpl.class);

    @Autowired
    private CiticBankConfig citicBankConfig;

    @Autowired
    private AppDAO appDAO;

    @Override
    public Boolean isCiticApp(Long appId) {
        return citicBankConfig.getAppIdSet().contains(appId);
    }

    @Override
    public SupplierRequest getVirturalRequest(SupplierRequest request) {
        try {
            String url = request.getHttpUrl();
            List<String> analysisList = analysisUrl(url);
            if (CollectionUtils.isEmpty(analysisList)) {
                return request;
            }
            String host = analysisList.get(0);
            Map<String, String> params = AssembleTool.getUrlParams(analysisList.get(1));
            request.setHttpUrl(AssembleTool.assembleUrl(host, rebuildVirtualParams(params, Long.valueOf(request.getAppId()))));
        } catch (Exception e) {
            LOGGER.info("[CiticApiImpl-getVirturalRequest] Params:[request={}], Msg:南京中信银行-虚拟商品定制逻辑异常", JSONObject.toJSONString(request), e);
        }
        return request;
    }


    @Override
    public String getVirturalResponse(SupplierRequest message, String body) {
        Map<String, String> duibaDoc = new HashMap<>();
        try {
            JSONObject resultJson = JSONObject.parseObject(body);
            if (null == resultJson) {
                throw new BizException("开发者接口响应内容异常");
            }
            String status = resultJson.getString("status");
            if (!Objects.equals("ok", status)) {
                throw new BizException(resultJson.getString("errorMessage"));
            }
            duibaDoc.put("status", "success");
            duibaDoc.put("data", resultJson.toString());
        } catch (Exception e) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", e.getMessage());
            LOGGER.info("[CiticApiImpl-getVirturalResponse] Params:[message={},body={}], Msg:南京中信银行-虚拟商品接口响应解析异常", JSONObject.toJSONString(message), body, e);
        }
        return JsonTool.objectToJson(duibaDoc);
    }


    /**
     * 封装虚拟商品请求参数
     *
     * @param params
     * @param appId
     * @return
     */
    private Map<String, String> rebuildVirtualParams(Map<String, String> params, Long appId) {
        Map<String, String> signParams = Maps.newHashMap();
        AppDO app = appDAO.getAppByCache(appId);
        signParams.put("appKey", app.getAppKey());
        signParams.put("uid", params.get("uid"));
        signParams.put("credits", params.get("params"));
        signParams.put("type", "virtual");
        signParams.put("orderNum", params.get("orderNum"));
        signParams.put("timestamp", String.valueOf(Instant.now().getEpochSecond()));
        signParams.put("description", params.get("description"));
        signParams.put("appSecret", appDAO.getAppSecret(app));
        signParams.put("sign", SignTool.sign(signParams));
        signParams.remove("appSecret");
        return signParams;
    }


    /**
     * 解析URL，获取网关地址和参数串
     *
     * @param url
     * @return java.lang.String
     * @throw
     */
    private static List<String> analysisUrl(String url) {
        List<String> analysis = Lists.newArrayList();
        if (StringUtils.isNotBlank(url)) {
            int index = url.indexOf("?");
            if (index != -1) {
                analysis.add(url.substring(0, index));
                analysis.add(url.substring(index + 1, url.length()));
            } else {
                analysis.add(url);
                analysis.add("");
            }
        }
        return analysis;
    }
}

