package cn.com.duiba.biz.credits.nongzonghang;

import cn.com.duiba.constant.nongzonghang.RuixinConfig;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.Charsets;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.Charset;
import java.security.Security;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * @author XuJing
 * @since 2020/3/16 2:29 下午
 */
@Service
public class RuixinApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(RuixinApi.class);
    @Autowired
    private RuixinConfig ruixinConfig;


    /**
     * 农总行下，特殊商品编码走瑞幸定制
     *
     * @param request
     * @return
     */
    public Boolean isRuiXin4NongZongHang(SupplierRequest request) {
        String url = request.getHttpUrl();
        String params = url.substring(url.indexOf('?') + 1, url.length());
        // 1.将请求URL的参数转换为MAP
        Map<String, String> creditMap = AssembleTool.getUrlParams(params);
        String goodsNum = creditMap.get("params");
        Set<String> goodsNos = ruixinConfig.getGoodsNos();
        //定制请求时的判断
        if (StringUtils.isNotBlank(goodsNum) && CollectionUtils.isNotEmpty(goodsNos)
                && goodsNos.contains(goodsNum)) {
            return true;
        }
        //定制响应时的判断
        if (request.getHttpUrl().startsWith(ruixinConfig.getAppUrl())) {
            return true;
        }
        return false;
    }

    public SupplierRequest getVirtualRequest(SupplierRequest request) {
        String url = request.getHttpUrl();
        String params = url.substring(url.indexOf('?') + 1, url.length());
        // 1.将请求URL的参数转换为MAP
        Map<String, String> crediMap = AssembleTool.getUrlParams(params);
        String goodsNum = crediMap.get("params");
        //构造请求参数
        TreeMap<String, String> map = buildParams(crediMap, goodsNum);
        String requestUrl = AssembleTool.assembleUrl(ruixinConfig.getAppUrl(), map);
        request.setHttpUrl(requestUrl);
        return request;
    }

    private TreeMap<String, String> buildParams(Map<String, String> crediMap, String goodsNum) {
        TreeMap<String, String> params = new TreeMap<>();
        params.put("cid", ruixinConfig.getCid());
        Map<String, Object> param = new HashMap<>();
        param.put("accountCode", ruixinConfig.getAccountCode());
        param.put("accountSecretkey", ruixinConfig.getAccountSecretkey());
        param.put("couponBatchNo", goodsNum);
        param.put("ticketNums", 1);
        param.put("mobile", crediMap.get("account"));
        param.put("serialNumber", crediMap.get("orderNum"));
        String q = JSON.toJSONString(param);
        q = encrypt(q, ruixinConfig.getAppKey());
        params.put("q", q);
        String sign = doMD5Sign(getSignStr(params) + ruixinConfig.getAppKey());
        params.put("sign", sign);
        return params;
    }

    public String getVirtualResponse(SupplierRequest request, String body) {
        String url = request.getHttpUrl();
        if (!url.startsWith(ruixinConfig.getAppUrl())) {
            return body;
        }
        //解密
        String json = decrypt(body, ruixinConfig.getAppKey());
        JSONObject jsonObject = null;
        Map<String, String> duibaDoc = new HashMap<>();
        try {
            jsonObject = JSON.parseObject(json);
        } catch (Exception e) {
            LOGGER.error("瑞辛接口返回参数异常:", e);
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", "出了点小问题，请联系客服处理");
            return JSON.toJSONString(duibaDoc);
        }
        Integer code = jsonObject.getInteger("code");
        if (code == 1) {
            duibaDoc.put("status", "success");
        } else {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", "出了点小问题，请联系客服处理");
            LOGGER.info("瑞辛接口返回json:{}", json);
        }
        return JSON.toJSONString(duibaDoc);
    }


    /**
     * 默认128位加密
     */
    private static final int KEY_BIT_SIZE = 128;

    /**
     * 密匙分段长度
     */
    private static final int BIT_SIZE = 8;

    /**
     * 编码
     */
    private static final Charset CHAR_SET = Charset.forName("utf-8");

    /**
     * AES，简单分组，填充7
     */
    private static final String ALGORITHM = "AES/ECB/PKCS7Padding";

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * 加密字符串。
     *
     * @param target 原始字符串
     * @param key    密钥字符串
     * @return 加密结果字符串
     */
    public static String encrypt(String target, String key) {
        try {
            Cipher cipher = Cipher.getInstance(ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, initKey(key));
            byte[] encryptResult = cipher.doFinal(target.getBytes(CHAR_SET));
            //兼容安卓环境的1.2codec
            String unsafeStr = new String(Base64.encodeBase64(encryptResult, false), CHAR_SET);
            return unsafeStr.replace('+', '-').replace('/', '_');
        } catch (Exception e) {
            throw new RuntimeException("敏感数据加密错误", e);
        }
    }

    /**
     * 解密字符串。
     *
     * @param target 加密结果字符串
     * @param key    密钥字符串
     * @return 原始字符串
     */
    public static String decrypt(String target, String key) {
        try {
            Cipher cipher = Cipher.getInstance(ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, initKey(key));
            String unsafeStr = target.replace('-', '+').replace('_', '/');
            byte[] decryptResult = cipher.doFinal(Base64.decodeBase64(unsafeStr.getBytes(CHAR_SET)));
            return new String(decryptResult, CHAR_SET);
        } catch (Exception e) {
            throw new RuntimeException("敏感数据解密错误", e);
        }
    }

    /**
     * 生成密钥字节数组，原始密钥字符串不足128位，补填0.
     *
     * @param originalKey 原始密钥
     * @return 加密
     */
    private static SecretKeySpec initKey(String originalKey) {
        byte[] keys = originalKey.getBytes(CHAR_SET);

        byte[] bytes = new byte[KEY_BIT_SIZE / BIT_SIZE];
        for (int i = 0; i < bytes.length; i++) {
            if (keys.length > i) {
                bytes[i] = keys[i];
            } else {
                bytes[i] = 0;
            }
        }

        return new SecretKeySpec(bytes, "AES");
    }


    /**
     * aid
     */
    public static final String API_KEY = "aid";

    /**
     * form 参数名sign
     */
    public static final String SIGN_KEY = "sign";


    /**
     * 计算签名串
     *
     * @param signMap 参数map
     * @return 签名
     */
    public static String getSignStr(SortedMap<String, String> signMap) {
        StringBuilder sb = new StringBuilder();
        Iterator<String> iterator = signMap.keySet().iterator();
        while (iterator.hasNext()) {
            String key = iterator.next();
            if (StringUtils.isNotEmpty(signMap.get(key))
                    && !key.equals(SIGN_KEY) && !key.equals(API_KEY)) {
                String value = signMap.get(key);
                sb.append(key).append("=").append(value).append(";");
            }
        }
        return sb.substring(0, sb.length() - 1);
    }

    /**
     * 截取块大小
     */
    private static final int BLOCK_LEN = 4;


    /**
     * md5签名
     *
     * @param targetStr 明文
     * @return 密文
     */
    public static String doMD5Sign(String targetStr) {
        byte[] md5Result = DigestUtils.md5(targetStr.getBytes(Charsets.UTF_8));
        if (md5Result.length != BLOCK_LEN * 4) {
            throw new IllegalArgumentException("MD5加密结果字节数组错误");
        }
        Integer first = Math.abs(bytesToInt(md5Result, BLOCK_LEN * 0));
        Integer second = Math.abs(bytesToInt(md5Result, BLOCK_LEN * 1));
        Integer third = Math.abs(bytesToInt(md5Result, BLOCK_LEN * 2));
        Integer fourth = Math.abs(bytesToInt(md5Result, BLOCK_LEN * 3));
        return first.toString() + second.toString() + third.toString() + fourth.toString();
    }

    /**
     * 与或常量
     */
    private static final int OX_FF = 0xFF;
    /**
     * 1bit
     */
    private static final int BIT = 8;

    /**
     * 高位前，低位后，字节数组转INT
     *
     * @param src    源字符
     * @param offset 偏移量
     * @return 结果
     */
    private static int bytesToInt(byte[] src, int offset) {
        int value;
        value = (int) (((src[offset] & OX_FF) << 3 * BIT)
                | ((src[offset + 1] & OX_FF) << 2 * BIT)
                | ((src[offset + 2] & OX_FF) << BIT)
                | (src[offset + 3] & OX_FF));
        return value;
    }


}
