package cn.com.duiba.biz.credits;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.HeBeiMobileConfig;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author: zhengjianhao
 * @date: 2020/4/17 17:38
 * @description: 河北移动定制
 */
@Service
public class HeBeiMobileApi {

    private static final Logger LOGGER = LoggerFactory.getLogger(HeBeiMobileApi.class);

    @Autowired
    private HeBeiMobileConfig heBeiMobileConfig;
	@Resource(name = "stringRedisTemplate")
    private StringRedisTemplate stringRedisTemplate;
	@Resource(name = "httpClient")
	private CloseableHttpClient httpClient;

	// token redis缓存key前缀
	private static final String TOKEN_CACHE_KEY = "heBeiMobileToken_";

	public boolean isHeBeiMobile(Long appId) {
		return heBeiMobileConfig.isHeBeiMobile(appId);
	}

	/**
	 * 获取token redis缓存key
	 */
	private String getTokenCacheKey(Long appId){
		return TOKEN_CACHE_KEY + appId;
	}

	/**
	 * 获取token，先从缓存获取，缓存20分钟，对方token 30分钟失效
	 */
	private String getToeknByCache(Long appId){
		if(!heBeiMobileConfig.isHeBeiMobile(appId)){
			return null;
		}
		String token;
		String key = getTokenCacheKey(appId);
		if(heBeiMobileConfig.getTokenByCache()){
			// 如果走缓存，先从缓存取
			token = stringRedisTemplate.opsForValue().get(key);
			if(StringUtils.isNotBlank(token)){
				return token;
			}
		}
		token = getTokenHttp();
		if (StringUtils.isNotBlank(token)) {
			stringRedisTemplate.opsForValue().set(key, token, 20, TimeUnit.MINUTES);
		}
		return token;
	}

	/**
	 * 调用http请求获取token
	 */
	private String getTokenHttp(){
		Map<String, String> params = new HashMap<>();
		params.put("ASKACC", heBeiMobileConfig.getAskacc());
		params.put("ASKPWD", heBeiMobileConfig.getAskcode());
		params.put("CHANNEL_CODE", heBeiMobileConfig.getChannelCode());
		params.put("REQURL", heBeiMobileConfig.getRequestUrlToken());

		String url = heBeiMobileConfig.getUrl();
		LOGGER.info("河北移动-获取token接口，请求url:{}, 参数params:{}", url, JSON.toJSON(params));

		HttpPost httpPost = new HttpPost(url);
		StringEntity stringEntity = new StringEntity(JSON.toJSONString(params), "UTF-8");
		httpPost.setEntity(stringEntity);

		String token = null;
		try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
			LOGGER.info("河北移动-获取token-返回response:{}", JSON.toJSON(response));
			HttpEntity entity = response.getEntity();
			if (entity == null) {
				return null;
			}
			String result = EntityUtils.toString(entity);
			LOGGER.info("河北移动-获取token-返回HttpEntity:{}", result);
			token = parseTokenResponse(result);
		} catch (IOException e) {
			LOGGER.warn("发送请求失败", e);
		}
		return token;
	}

	/**
	 * 解析获取token返回数据
	 */
	private String parseTokenResponse(String result){
		JSONObject responseBody;
		try {
			responseBody = JSON.parseObject(result);
		} catch (Exception e) {
			LOGGER.warn("河北移动-解析返回数据，result:{}, JSON.parseObject:", result, e);
			return null;
		}
		JSONObject resultObj = responseBody.getJSONObject("resultObj");
		if(null != resultObj){
			return resultObj.getString("TOKEN");
		}
		return null;
	}

	/**
     * 构造虚拟商品请求，河北移动虚拟商品走对方的发券接口
     */
	public SupplierRequest getVirtualRequest(SupplierRequest request){
		String url = request.getHttpUrl();
		String authParams = url.substring(url.indexOf('?') + 1);

		Map<String, String> requestParams = buildVirtualRequestParams(Long.valueOf(request.getAppId()), authParams);

		request.setHttpUrl(heBeiMobileConfig.getUrl());
		request.setAuthParams(requestParams);
		return request;
	}

    /**
     * 构造虚拟商品请求参数
     */
    private Map<String, String> buildVirtualRequestParams(Long appId, String authParams) {

        Map<String, String> authParamMap = AssembleTool.getUrlParams(authParams);

	    Map<String, String> requestParams = new HashMap<>();
	    requestParams.put("ASKACC", heBeiMobileConfig.getAskacc());
	    requestParams.put("ASKPWD", heBeiMobileConfig.getAskcode());
	    requestParams.put("CHANNEL_CODE", heBeiMobileConfig.getChannelCode());
	    requestParams.put("REQURL", heBeiMobileConfig.getRequestUrlSendCard());
	    requestParams.put("Prize_id", authParamMap.get("params"));
	    requestParams.put("MOBILE", authParamMap.get("uid"));
	    requestParams.put("date", LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
	    requestParams.put("order_id", authParamMap.get("orderNum"));
	    requestParams.put("TOKEN", getToeknByCache(appId));

	    return requestParams;
    }

	/**
	 * 解析虚拟商品请求返回数据
	 */
	public String getVirtualResponse(SupplierRequest message, String body){
		Map<String, String> duibaDoc = new HashMap<>();
		try {
			JSONObject resultJson = JSON.parseObject(body);
			if (null == resultJson) {
				throw new BizException("河北移动-虚拟商品发奖-开发者接口响应内容异常");
			}
			String status = resultJson.getString("resultCode");
			if (!Objects.equals("0", status)) {
				throw new BizException(resultJson.getString("resultMsg"));
			}
			duibaDoc.put("status", "success");
			duibaDoc.put("data", body);
		} catch (Exception e) {
			duibaDoc.put("status", "fail");
			duibaDoc.put("errorMessage", e.getMessage());
			LOGGER.info("[HeBeiMobileApi-getVirtualResponse] Params:[message={},body={}], Msg:河北移动-虚拟商品接口响应解析异常", JSON.toJSON(message), body, e);
		}
		return JsonTool.objectToJson(duibaDoc);
	}
}

