package cn.com.duiba.biz.qb.supplier;

import cn.com.duiba.api.enums.SubjectTypeEnum;
import cn.com.duiba.domain.MessageUniqueCheckDO;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.domain.SupplierResponse;
import cn.com.duiba.mq.RocketMQMsgProducer;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.service.MessageService;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * @author JunAngLiu
 * @Title: QbSupplier
 * @Description: qb
 * @date 2018/7/3110:36
 */
@Service
public class KaiXinQbSupplier {

    private static Logger log = LoggerFactory.getLogger(KaiXinQbSupplier.class);

    @Value("${qb.kaixin.md5Key}")
    private String md5Key;
    @Value("${qb.kaixin.duiba.md5Key}")
    private String duibaMd5Key;
    @Value("${qb.kaixin.duia2.md5Key}")
    private String duia2Md5Key;


    @Autowired
    private MessageService messageService;
    @Autowired
    private RocketMQMsgProducer rocketMQMsgProducer;
    @Autowired
    private HttpAsyncClientPool httpAsyncClientPool;

    /**
      * 提交到HTTP请求池
	 * @param message
	 * @param callbackTopic
	 * @param msgTag
	 * @param msgKey
	 */
    public void asyncSubmit(final SupplierRequest message, final String callbackTopic, final String msgTag, final String msgKey) {
        if (!messageService.messageUniqueCheck(message.getSupplierOrderId(), MessageUniqueCheckDO.TYPE_QB)) {
            return;
        }
        HttpRequestLog.logUrl("[action qb] [tag request] [bizId " + message.getOrderId() + "] [url " + message.getHttpUrl() + "]" + "[params  " + JSONObject.toJSONString(message.getParams())+"]");
        HttpRequestBase post = getHttp(message);
        httpAsyncClientPool.submit(message.getAppId(), post, new FutureCallback<HttpResponse>() {
            @Override
            public void completed(HttpResponse response) {
                SupplierResponse resp = new SupplierResponse();
                try {
                    resp.setOrderId(message.getOrderId());
                    resp.setSupplierOrderId(message.getSupplierOrderId());
                    resp.setSupplierName(message.getSupplierName());
                    resp.setUrl(message.getHttpUrl());
                    resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_COMPLETED);
                    resp.setAppId(message.getAppId());
                    resp.setConsumerId(message.getConsumerId());
                    String body = EntityUtils.toString(response.getEntity());
                    resp.setBody(body);
                } catch (Exception e) {
                    log.error("KaiXinQbSupplier completed", e);
                } finally {
                    finallyBlock(message, resp, callbackTopic, msgTag, msgKey);
                }
            }

            @Override
            public void failed(Exception ex) {
                log.error("KaiXinQbSupplier failed orderId:" + message.getOrderId(), ex);
                SupplierResponse resp = new SupplierResponse();
                try {
                    resp.setOrderId(message.getOrderId());
                    resp.setSupplierOrderId(message.getSupplierOrderId());
                    resp.setSupplierName(message.getSupplierName());
                    resp.setUrl(message.getHttpUrl());
                    resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_FAILED);
                    resp.setAppId(message.getAppId());
                    resp.setConsumerId(message.getConsumerId());
                    resp.setErrorMessage(ex.getClass().getName() + ":" + ex.getMessage());
                } catch (Exception e) {
                    log.error("KaiXinQbSupplier completed", e);
                } finally {
                    finallyBlock(message, resp, callbackTopic, msgTag, msgKey);
                }
            }

            @Override
            public void cancelled() {
                SupplierResponse resp = new SupplierResponse();
                try {
                    resp.setOrderId(message.getOrderId());
                    resp.setSupplierOrderId(message.getSupplierOrderId());
                    resp.setSupplierName(message.getSupplierName());
                    resp.setUrl(message.getHttpUrl());
                    resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_CANCELLED);
                    resp.setAppId(message.getAppId());
                    resp.setConsumerId(message.getConsumerId());
                } catch (Exception e) {
                    log.error("KaiXinQbSupplier cancelled", e);
                } finally {
                    finallyBlock(message, resp, callbackTopic, msgTag, msgKey);
                }
            }

        });

    }

    private void finallyBlock(SupplierRequest message, SupplierResponse response, final String callbackTopic, String msgTag, String msgKey) {
        try {
            response.setUrl(message.getHttpUrl());
            response.setSupplierName(message.getSupplierName());
            response.setSupplierOrderId(message.getSupplierOrderId());
            response.setOrderId(message.getOrderId());
            response.setAppId(message.getAppId());
            response.setConsumerId(message.getConsumerId());
            String body = JsonTool.objectToJson(response);
            HttpRequestLog.logUrl("[action qb] [tag response] [callback " + response.getCallbackType() + "] [bizId " + message.getOrderId() + "] [body " + response.getBody() + "]");
            if(callbackTopic.contains("MQ")){
                rocketMQMsgProducer.sendMsg(callbackTopic, msgTag, msgKey, body,false,null);
            }else{
                messageService.sendMsg(callbackTopic,msgTag,msgKey,body);
            }
        } catch (Exception e) {
            log.error("KaiXinQbSupplier callback:" + message.getOrderId(), e);
        }
    }



    private HttpRequestBase getHttp(SupplierRequest message){

        Map<String,String> params = message.getParams();
        Map<String,String> bodyMap = new HashMap<>();
        bodyMap.put("pay_password",params.get("pay_password"));
        bodyMap.put("order_no",params.get("order_no"));
        bodyMap.put("account_no",params.get("account_no"));
        bodyMap.put("product_id",params.get("product_id"));
        bodyMap.put("number",params.get("number"));
        bodyMap.put("type", params.get("type"));
        bodyMap.put("ip", params.get("ip"));

        Map<String,String> headerMap = new HashMap<>();
        headerMap.put("partner_id",params.get("partner_id"));
        headerMap.put("orgcode",params.get("orgcode"));
        headerMap.put("agent_id",params.get("agent_id"));
        headerMap.put("password",params.get("password"));
        headerMap.put("timestamp",params.get("timestamp"));
        headerMap.put("sign_type",params.get("sign_type"));
        headerMap.put("sign",sign(headerMap,bodyMap, params.get("subject")));

        Map<String,Map<String,String>> raws = new HashMap<>();
        raws.put("header",headerMap);
        raws.put("body",bodyMap);

        String body =  JSONObject.toJSONString(JSONObject.toJSON(raws));

        HttpRequestBase http = AssembleTool.assembleRequestJson(message.getHttpUrl(), body);
        return  http;
    }

    // 将基本元素与业务元素中的所有子元素（sign除外）按照“key=value”的格式拼接起来，
    // 并且将拼接以后的字符串以“&”字符再连接起来。顺序按首字母升序排列，值为空的不参与签名。
    // 最后将具体的密钥（key）放在末尾对组装好的字符串做MD5签名。
    private String sign(Map<String,String> headerMap,Map<String,String> bodyMap, String subject){
        Map<String,String> map = new TreeMap<>();
        map.putAll(headerMap);
        map.putAll(bodyMap);

        StringBuilder builder = new StringBuilder();
        int index = 0;
        for(Map.Entry<String, String> entry: map.entrySet()){
            if(StringUtils.isBlank(entry.getValue())){
                continue;
            }

            if(index > 0){
                builder.append("&");
            }
            builder.append(entry.getKey()).append("=").append(entry.getValue());
            index++;
        }
        if (StringUtils.isBlank(subject)) {
            log.info("qb签名没有主体流程");
            builder.append(md5Key);
        } else if (SubjectTypeEnum.DUIBA.getType().equals(subject)) {
            log.info("qb签名主体为兑吧流程");
            builder.append(duibaMd5Key);
        } else if (SubjectTypeEnum.DUIA.getType().equals(subject)) {
            log.info("qb签名主体为兑啊流程");
            builder.append(duia2Md5Key);
        }
        return DigestUtils.md5Hex(builder.toString());
    }

}
