package cn.com.duiba.service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.CiticBankConfig;
import org.apache.commons.codec.binary.Base64;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.impl.nio.conn.PoolingNHttpClientConnectionManager;
import org.apache.http.impl.nio.reactor.DefaultConnectingIOReactor;
import org.apache.http.nio.conn.NoopIOSessionStrategy;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.conn.ssl.SSLIOSessionStrategy;
import org.apache.http.protocol.HttpContext;
import org.apache.http.ssl.SSLContexts;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.net.ssl.SSLContext;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.KeyStore;

/**
 * Created by HePeng on 2019/05/15 16:55.
 */
@Configuration
@EnableConfigurationProperties(CiticBankConfig.class)
public class CustomHttpClientFactory {
    private static final Logger LOGGER = LoggerFactory.getLogger(CustomHttpClientFactory.class);

    @Autowired
    private CiticBankConfig citicBankConfig;

    /**
     * 连接超时，连接目标url最大超时
     */
    private static final int CONNECT_TIMEOUT = 10 * 1000;

    /**
     * 从连接池中获取连接的超时时间
     */
    private static final int DEFAULT_REQUEST_TIMEOUT = 3 * 1000;

    /**
     * 长链接空闲时间
     */
    private static final int KEEPALIVE_TIMEOUT = 5 * 1000;

    /**
     * 长链接空闲时间（连接有效时长3s）
     */
    private static final int SHOT_KEEPALIVE_TIMEOUT = 3 * 1000;

    /**
     * 处理超时，等待响应（读数据）最大超时
     */
    private static final int SOCKET_TIMEOUT = 10 * 1000;

    /**
     * 连接池中的最大连接数
     */
    private static final int MAX_CONNECT = 300;

    /**
     * 每个路由最大连接{并发}值，连接同一个route最大的并发数
     */
    private static final int MAX_ROUTE_CONNECT = 20;

    /**
     * 中信银行异步连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="citicBankHttpAsyncClient")
    public CloseableHttpAsyncClient citicBankHttpAsyncClient(){
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom()
                .setConnectionManager(getConnectionManager())
                .setDefaultRequestConfig(config)
                .setMaxConnTotal(MAX_CONNECT)
                .setMaxConnPerRoute(MAX_ROUTE_CONNECT)
                .setKeepAliveStrategy(getKeepAliveStrategy())
                .build();
        return httpClient;
    }

    /**
     * 中信银行通知连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="citicBankHttpAsyncNotifyClient")
    public CloseableHttpAsyncClient citicBankHttpAsyncNotifyClient(){
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom()
                .setConnectionManager(getConnectionManager())
                .setDefaultRequestConfig(config)
                .setMaxConnTotal(MAX_CONNECT)
                .setMaxConnPerRoute(MAX_ROUTE_CONNECT)
                .setKeepAliveStrategy(getKeepAliveStrategy())
                .build();
        return httpClient;
    }

    /**
     * 港中旅异步连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="ctsHttpAsyncClient")
    public CloseableHttpAsyncClient ctsHttpAsyncClient(){
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .setSocketTimeout(20 * 1000)
                .build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom()
                .setConnectionManager(getConnectionManager())
                .setDefaultRequestConfig(config)
                .setMaxConnTotal(MAX_CONNECT)
                .setMaxConnPerRoute(MAX_ROUTE_CONNECT)
                .setKeepAliveStrategy(getKeepAliveStrategy())
                .build();
        return httpClient;
    }


    @Bean(initMethod = "start", destroyMethod = "close", name = "hsbcHttpAsyncClient")
    public CloseableHttpAsyncClient hsbcHttpAsyncClient() {
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(10 * 1000)
                .setConnectionRequestTimeout(10 * 1000)
                .setSocketTimeout(10 * 1000)
                .build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom()
                .setDefaultRequestConfig(config)
                .setMaxConnTotal(3000)
                .setMaxConnPerRoute(500)
                .setKeepAliveStrategy(getKeepAliveStrategy())
                .build();
        return httpClient;
    }

    @Bean(destroyMethod = "close", name = "shotKeepAliveHttpClient")
    public CloseableHttpClient shotKeepAliveHttpClient() {
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(10 * 1000)
                .setConnectionRequestTimeout(10 * 1000)
                .setSocketTimeout(10 * 1000)
                .build();
        CloseableHttpClient httpClient = HttpClients.custom()
                .setDefaultRequestConfig(config)
//                .disableAutomaticRetries()//禁止重试
                .setMaxConnTotal(3000)
                .setMaxConnPerRoute(500)
                .setKeepAliveStrategy(getShotKeepAliveStrategy())
                .build();
        return httpClient;
    }

    /**
     * 获取ConnectionManager
     * @return
     */
    private PoolingNHttpClientConnectionManager getConnectionManager() {
        try {
            SSLContext sslContext = getSSLContext();
            if(sslContext == null) {
                throw new BizException("getConnectionManager failed, sslContext is null!!!");
            }
            return new PoolingNHttpClientConnectionManager(
                    new DefaultConnectingIOReactor()
                    , RegistryBuilder.<SchemeIOSessionStrategy>create()
                    .register("http", NoopIOSessionStrategy.INSTANCE)
                    .register("https", new SSLIOSessionStrategy(sslContext, NoopHostnameVerifier.INSTANCE))
                    .build()
            );
        } catch (Exception e) {
            LOGGER.warn("", e);
            return null;
        }
    }

    /**
     * 获取SSLContext
     * @return
     */
    private SSLContext getSSLContext() {
        try(InputStream inputStream = new ByteArrayInputStream(Base64.decodeBase64(citicBankConfig.getCaCertBase64()))) {
            KeyStore keyStore = KeyStore.getInstance("jks");
            keyStore.load(inputStream, citicBankConfig.getKeyStorePass().toCharArray());
            return SSLContexts.custom()
                    .loadTrustMaterial(keyStore, new TrustSelfSignedStrategy())
                    .build();
        } catch (Exception e) {
            LOGGER.warn("", e);
            return null;
        }
    }

    /**
     * 长连接策略
     * @return
     */
    private static DefaultConnectionKeepAliveStrategy getKeepAliveStrategy() {
        return new DefaultConnectionKeepAliveStrategy() {
            @Override
            public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                long duration = super.getKeepAliveDuration(response, context);
                if (duration == -1) {
                    return KEEPALIVE_TIMEOUT;
                }
                return duration;
            }
        };
    }

    /**
     * 长连接策略(连接有效时长3s)
     *
     * @return
     */
    private static DefaultConnectionKeepAliveStrategy getShotKeepAliveStrategy() {
        return new DefaultConnectionKeepAliveStrategy() {
            @Override
            public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                long duration = super.getKeepAliveDuration(response, context);
                if (duration == -1) {
                    return SHOT_KEEPALIVE_TIMEOUT;
                }
                return duration;
            }
        };
    }

}