package cn.com.duiba.tool.huawei;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * 华为签名工具类
 * Created by xuwei on 2020/12/02.
 **/
public class SignUtils {

    private static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;

    private static final String HMAC_SHA256 = "HmacSHA256";
    private static final String SHA256 = "SHA-256";
    private static final String SHA256_WITH_RSA_PSS = "SHA256withRSA/PSS";
    private static final String RSA = "RSA";

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * HmacSHA256签名
     * @param plainText 明文
     * @param key 密钥
     * @return
     * @throws Exception
     */
    public static String hmacSHA256(String plainText, String key) throws Exception {
        Mac mac = Mac.getInstance(HMAC_SHA256);
        SecretKeySpec secretKeySpec = new SecretKeySpec(key.getBytes(DEFAULT_CHARSET), HMAC_SHA256);
        mac.init(secretKeySpec);
        return Hex.encodeHexString(mac.doFinal(plainText.getBytes(DEFAULT_CHARSET)));
    }

    /**
     * SHA256签名
     *
     * @param plainText 明文
     * @return
     */
    public static String sha256(String plainText) throws Exception {
        MessageDigest messageDigest = MessageDigest.getInstance(SHA256);
        messageDigest.update(plainText.getBytes(DEFAULT_CHARSET));
        return Hex.encodeHexString(messageDigest.digest());
    }

    /**
     * SHA256WithRSA/PSS 签名
     * @param plainText 明文
     * @param privateKeyStr 私钥base64编码字符串
     * @return
     * @throws Exception
     */
    public static String sign(String plainText, String privateKeyStr) throws Exception {
        return sign(plainText, getPrivateKey(privateKeyStr));
    }

    /**
     * SHA256WithRSA/PSS 签名
     * @param plainText 明文
     * @param privateKey 私钥
     * @return
     */
    public static String sign(String plainText, PrivateKey privateKey) throws Exception {
        Signature Sign = Signature.getInstance(SHA256_WITH_RSA_PSS);
        Sign.initSign(privateKey);
        Sign.update(plainText.getBytes(DEFAULT_CHARSET));
        return Base64.encodeBase64String(Sign.sign());
    }

    /**
     * SHA256WithRSA/PSS 验签
     * @param plainText 明文
     * @param signText 签名
     * @param publicKeyStr 公钥base64编码字符串
     * @return
     * @throws Exception
     */
    public static boolean verifySign(String plainText, String signText, String publicKeyStr) throws Exception {
        return verifySign(plainText, signText, getPublicKey(publicKeyStr));
    }

    /**
     * SHA256WithRSA/PSS 验签
     * @param plainText 明文
     * @param signText 签名
     * @param publicKey 公钥
     */
    public static boolean verifySign(String plainText, String signText, PublicKey publicKey) throws Exception {
        Signature verifySign = Signature.getInstance(SHA256_WITH_RSA_PSS);
        verifySign.initVerify(publicKey);
        verifySign.update(plainText.getBytes(DEFAULT_CHARSET));
        return verifySign.verify(Base64.decodeBase64(signText));
    }

    /**
     * String转公钥PublicKey
     * @param publicKeyStr
     * @return
     * @throws Exception
     */
    public static PublicKey getPublicKey(String publicKeyStr) throws Exception {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKeyStr));
        KeyFactory keyFactory = KeyFactory.getInstance(RSA);
        return keyFactory.generatePublic(keySpec);
    }

    /**
     * String转私钥PrivateKey
     * @param key
     * @return
     * @throws Exception
     */
    public static PrivateKey getPrivateKey(String key) throws Exception {
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(Base64.decodeBase64(key));
        KeyFactory keyFactory = KeyFactory.getInstance(RSA);
        return keyFactory.generatePrivate(keySpec);
    }

}
