package cn.com.duiba.biz.credits;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.config.SinaCreditsConfig;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * Created with IntelliJ IDEA.
 * Description: 新浪微博积分服务
 * CreatedBy: fengchao
 * Date: 2020-04-21
 * Time: 2:17 PM
 */
@Service
public class SinaCreditsApi {
    private static final Logger LOGGRER = LoggerFactory.getLogger(SinaCreditsApi.class);

    public static final String SIGN_TYPE = "md5";
    public static final Set<Long> SINA_APPIDS = Sets.newHashSet(56617L);

    private static final String DELIMITER = "?";

    @Autowired
    private SinaCreditsConfig sinaCreditsConfig;


    /**
     * 判断当前app是否是新浪微博的app
     *
     * @param appId
     * @return
     */
    public boolean isSinaApp(Long appId) {
        return SINA_APPIDS.contains(appId);
    }

    /**
     * 组装加积分请求
     *
     * @param request
     * @return
     * @throws BizException
     */
    public CreditsMessageDto getAddCreditsMessage(CreditsMessageDto request) throws BizException {

        Map<String, String> originAuthParams = request.getAuthParams();
        TreeMap<String, String> treeMap = convertParam(originAuthParams);
        String sign = getSign(treeMap, sinaCreditsConfig.getSecertKey());
        treeMap.put("sign", sign);
        treeMap.put("sign_type", SIGN_TYPE);
        request.setAuthParams(treeMap);
        request.setHttpType(CreditsMessage.HTTP_GET);
        String host = this.getGateway(request.getHttpUrl());
        request.setHttpUrl(AssembleTool.assembleUrl(host, treeMap));

        return request;
    }


    /**
     * 解析加积分接口返回结果
     *
     * @param body
     * @return
     */
    public String parseResponse(String body) {
        Map<String, String> duibaDoc = new HashMap<>();
        JSONObject resultJson = new JSONObject();
        if (StringUtils.isEmpty(body) || body.trim().startsWith("[")) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("message", "返回体 body 异常");
            LOGGRER.info("[SinaCreditsApi-parseResponse]新浪微博响应异常[body={}]", body);
            return JsonTool.objectToJson(duibaDoc);
        }
        try {
            resultJson = JSONObject.parseObject(body);
        } catch (Exception e) {
            LOGGRER.error("新浪微博转账加积分接口返回 json反序列化失败, body={}", body, e);
        }
        String code = resultJson.getString("code");
        if ("100000".equals(code)) {
            duibaDoc.put("status", "ok");
            duibaDoc.put("transfer_id", resultJson.getString("transfer_id"));
        } else {
            duibaDoc.put("status", "fail");
            duibaDoc.put("message", resultJson.getString("msg"));
        }
        duibaDoc.put("data", body);
        return JsonTool.objectToJson(duibaDoc);
    }

    /**
     * 参数处理
     *
     * @param originAuthParams
     * @return
     * @throws BizException
     */
    public TreeMap<String, String> convertParam(Map<String, String> originAuthParams) throws BizException {

        //兑吧订单ID
        String out_apply_id = originAuthParams.get("orderNum");
        //收入人uid(用户uid)
        String granted_uid = originAuthParams.get("uid");
        //积分
        String jf = originAuthParams.get("credits");
        //描述
        String subject = originAuthParams.get("description");
        if (StringUtils.isEmpty(out_apply_id) || StringUtils.isEmpty(granted_uid) || !NumberUtils.isNumber(jf)) {
            LOGGRER.info("sinaCredit request params: " + JSONObject.toJSONString(originAuthParams));
            throw new BizException("请求参数异常");
        }
        TreeMap<String, String> treeMap = new TreeMap();
        treeMap.put("source", sinaCreditsConfig.getSource());
        treeMap.put("type", sinaCreditsConfig.getDefaultType());
        treeMap.put("out_apply_id", out_apply_id);
        treeMap.put("grant_uid", sinaCreditsConfig.getGrantUid());
        treeMap.put("granted_uid", granted_uid);
        treeMap.put("jf", jf);
        treeMap.put("subject", StringUtils.isNotBlank(subject) ? subject : sinaCreditsConfig.getDefaultSubject());
        return treeMap;
    }

    /**
     * 二行制转字符串
     */
    private String byte2hex(byte[] b) {
        StringBuilder hs = new StringBuilder();
        String stmp = "";
        for (byte aB : b) {
            stmp = (Integer.toHexString(aB & 0XFF));
            if (stmp.length() == 1) {
                hs.append("0").append(stmp);
            } else {
                hs.append(stmp);
            }
        }
        return hs.toString().toUpperCase();
    }

    /**
     * 获取签名
     *
     * @param params 传给服务器的参数
     * @param secret 签名命牌
     * @return
     */
    private String getSign(TreeMap<String, String> params, String secret) {
        List<String> keys = new ArrayList(params.keySet());
        Collections.sort(keys);
        StringBuilder orgin = new StringBuilder();
        for (String key : keys) {
            orgin.append(key).append("=");
            if (StringUtils.isNotBlank(params.get(key))) {
                orgin.append(params.get(key));
            }
            orgin.append("&");
        }
        String paramStr = orgin.substring(0, orgin.length() - 1) + secret;
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            return byte2hex(md.digest(paramStr.getBytes(StandardCharsets.UTF_8))).toLowerCase();
        } catch (Exception e) {
            return null;
        }
    }

    private String getGateway(String url) {
        if (StringUtils.isNotBlank(url)) {
            int index = url.indexOf(DELIMITER);
            if (index > 0) {
                return url.substring(0, index);
            }
        }
        return url;
    }
}
